<?php
/**
 * GravityBoard Entry Meta Box
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Attachments;

defined( 'ABSPATH' ) || exit;

use GFCommon;
use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Attachments\Attachments;

/**
 * Class Entry_Meta
 *
 * Handles adding a meta box to the Gravity Forms entry detail screen
 * to display GravityBoard attachments.
 *
 * @since NEXT_VERSION
 */
class Entry_Meta {

	/**
	 * Instance of the class.
	 *
	 * @var Entry_Meta|null
	 */
	private static $instance = null;

	/**
	 * Get the instance of the class.
	 *
	 * @since NEXT_VERSION
	 * @return Entry_Meta The instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'gform_entry_detail_meta_boxes', [ $this, 'add_attachments_meta_box' ], 10, 3 );
	}

	/**
	 * Adds the attachments meta box to the entry detail screen.
	 *
	 * @since NEXT_VERSION
	 *
	 * @param array $meta_boxes The existing meta boxes.
	 * @param array $entry      The current entry object.
	 * @param array $form       The current form object.
	 * @return array The modified meta boxes array.
	 */
	public function add_attachments_meta_box( $meta_boxes, $entry, $form ) {
		// Check all feeds, not just active ones. Attachments may have been uploaded to inactive feeds.
		$all_feeds = Feed::get_instance()->get_feeds( $form['id'] );

		// Only add meta box if we have at least one feed.
		if ( empty( $all_feeds ) ) {
			return $meta_boxes;
		}

		$meta_boxes[ Attachments::ATTACHMENT_META_KEY ] = [
			'title'    => esc_html__( 'GravityBoard Attachments', 'gk-gravityboard' ),
			'callback' => [ $this, 'render_attachments_meta_box' ],
			'context'  => 'normal',
		];

		return $meta_boxes;
	}

	/**
	 * Renders the content of the attachments meta box.
	 *
	 * @since NEXT_VERSION
	 *
	 * @param array $args An array of arguments. Expected keys: 'form', 'entry', 'feeds'.
	 * @return void
	 */
	public function render_attachments_meta_box( $args ) {
		$entry = $args['entry'];

		// REST API nonce for deletion.
		$delete_nonce = wp_create_nonce( 'wp_rest' );

		echo '<div class="gk-gravityboard-attachments-metabox" id="gk-board-attachments-list-' . esc_attr( $entry['id'] ) . '">';
		echo '<div id="gk-board-attachment-status-' . esc_attr( $entry['id'] ) . '" class="gk-attachment-status" aria-live="polite" aria-atomic="true"></div>';
		echo '<div id="gk-board-no-attachments-' . esc_attr( $entry['id'] ) . '" class="gk-no-attachments-message" aria-live="polite">' . esc_html__( 'No attachments found for this entry.', 'gk-gravityboard' ) . '</div>';

		$attachments = Attachments::get_attachments( $entry['id'] );

		// If no attachments exist, show the no attachments message.
		if ( empty( $attachments ) || is_wp_error( $attachments ) ) {
			echo '<script>';
			echo 'document.addEventListener("DOMContentLoaded", function() {';
			echo 'var emptyStateElement = document.getElementById("gk-board-no-attachments-' . esc_attr( $entry['id'] ) . '");';
			echo 'if (emptyStateElement) { emptyStateElement.classList.add("visible"); }';
			echo '});';
			echo '</script>';
			echo '</div>';
			// Add inline styles and scripts even when no attachments exist.
			$this->render_inline_styles();
			$this->render_inline_scripts();
			return;
		}

		echo '<div class="gk-board-feed-attachments">';
		echo '<h4 class="gk-board-feed-title">' . esc_html__( 'Attachments', 'gk-gravityboard' ) . '</h4>';
		echo '<ul>';

		foreach ( $attachments as $attachment ) {
			$attachment_id = $attachment['id'] ?? uniqid( 'att_' );
			$file_name     = esc_html( $attachment['name'] );
			$file_url      = esc_url( $attachment['url'] );
			$file_type     = $attachment['type'] ?? '';
			$date_created  = $attachment['date_created'] ?? null;
			$user_id       = isset( $attachment['user_id'] ) ? (int) $attachment['user_id'] : null;

			$uploader_name = esc_html__( 'Unknown User', 'gk-gravityboard' );
			if ( 0 === $user_id ) {
				$uploader_name = esc_html__( 'Logged-out user', 'gk-gravityboard' );
			} elseif ( $user_id > 0 ) {
				$user_data = get_userdata( $user_id );
				if ( false !== $user_data ) {
					$uploader_name = esc_html( $user_data->display_name );
				}
			}

			$upload_date_str = esc_html__( 'Unknown Date', 'gk-gravityboard' );
			if ( $date_created ) {
				$upload_date_str = esc_html( GFCommon::format_date( gmdate( 'Y-m-d H:i:s', strtotime( $date_created ) ), false, '', true ) );
			}

			echo '<li id="gk-attachment-item-' . esc_attr( $attachment_id ) . '">';
			echo '<div class="attachment-preview">';
			if ( strpos( $file_type, 'image/' ) === 0 ) {
				echo '<img src="' . esc_url( $file_url ) . '" alt="' . esc_attr( $file_name ) . '" class="attachment-thumbnail" />';
			} else {
				$icon_class = 'dashicons dashicons-media-default';
				if ( strpos( $file_type, 'application/pdf' ) === 0 ) {
					$icon_class = 'dashicons dashicons-media-document';
				} elseif ( strpos( $file_type, 'text/' ) === 0 ) {
					$icon_class = 'dashicons dashicons-text';
				} elseif ( strpos( $file_type, 'video/' ) === 0 ) {
					$icon_class = 'dashicons dashicons-media-video';
				} elseif ( strpos( $file_type, 'audio/' ) === 0 ) {
					$icon_class = 'dashicons dashicons-media-audio';
				}
				echo '<span class="' . esc_attr( $icon_class ) . ' attachment-icon" aria-hidden="true"></span>';
			}
			echo '</div>';

			echo '<div class="attachment-info">';
			echo '<a href="' . esc_url( $file_url ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $file_name ) . '</a>';

			// Delete link - only if we have a feed context and user can perform the action.
			echo ' <button type="button" ';
			echo 'data-entry-id="' . esc_attr( $entry['id'] ) . '" ';
			echo 'data-attachment-id="' . esc_attr( $attachment_id ) . '" ';
			echo 'data-feed-id="' . esc_attr( $attachment['feed_id'] ) . '" ';
			echo 'data-nonce="' . esc_attr( $delete_nonce ) . '" ';
			echo 'aria-label="' . esc_attr( sprintf( __( 'Delete attachment %s', 'gk-gravityboard' ), $file_name ) ) . '" ';
			echo 'class="gk-delete-attachment-button button-link">' . esc_html__( 'Delete', 'gk-gravityboard' ) . '</button>';

			echo '<div class="attachment-meta">';
			printf(
				// translators: %1$s: User display name. %2$s: Upload date.
				esc_html__( 'Uploaded by: %1$s on %2$s', 'gk-gravityboard' ),
				esc_html( $uploader_name ),
				esc_html( $upload_date_str )
			);
			echo '</div>';
			echo '</div>';
			echo '</li>';
		}

		echo '</ul>';
		echo '</div>';

		echo '</div>';

		// Add inline styles for metabox.
		$this->render_inline_styles();
		// Add inline JavaScript for attachment deletion.
		$this->render_inline_scripts();
	}

	/**
	 * Renders inline CSS for the attachments metabox.
	 *
	 * @since TODO
	 *
	 * @return void
	 */
	private function render_inline_styles() {
		?>
		<style type="text/css">
			.gk-gravityboard-attachments-metabox ul {
				list-style: none;
				margin: 0;
				padding: 0;
			}
			.gk-gravityboard-attachments-metabox li {
				padding: 10px 0;
				border-bottom: 1px solid #eee;
				display: flex;
				align-items: flex-start;
			}
			.gk-gravityboard-attachments-metabox li:last-child {
				border-bottom: none;
			}
			.gk-gravityboard-attachments-metabox .attachment-preview {
				margin-right: 10px;
				margin-bottom: 5px;
				flex-shrink: 0;
			}
			.gk-gravityboard-attachments-metabox .attachment-thumbnail {
				max-width: 100px;
				max-height: 100px;
				border-radius: 3px;
			}
			.gk-gravityboard-attachments-metabox .attachment-icon {
				font-size: 40px;
				width: 40px;
				height: 40px;
				display: inline-block;
			}
			.gk-gravityboard-attachments-metabox .attachment-info a {
				text-decoration: none;
			}
			.gk-gravityboard-attachments-metabox .attachment-meta {
				margin-top: 4px;
				font-size: 0.9em;
				color: #777;
			}
			.gk-gravityboard-attachments-metabox .gk-delete-attachment-button {
				background: none;
				border: none;
				color: #a00;
				text-decoration: none;
				margin-left: 8px;
				cursor: pointer;
				padding: 0;
				font-size: inherit;
			}
			.gk-gravityboard-attachments-metabox .gk-delete-attachment-button:hover {
				text-decoration: underline;
			}
			.gk-gravityboard-attachments-metabox .gk-delete-attachment-button:disabled {
				opacity: 0.6;
				cursor: not-allowed;
			}
			.gk-gravityboard-attachments-metabox .gk-attachment-status {

				padding: 10px 8px 11px 8px;

				align-items: center;
				background: #fff;
				border: 1px solid #d5d7e9;
				border-radius: 3px;
				box-shadow: 0 2px 1px rgba(28, 31, 63, .0634624);
				font-size: 13px;
				margin-bottom: 0.8125rem;
				display: none;
			}
			.gk-gravityboard-attachments-metabox .gk-attachment-status.visible {
				display: flex;
			}
			.gk-gravityboard-attachments-metabox .gk-attachment-status.error {
				background: #fef7f7;
				border-left-color: #d63638;
				color: #d63638;
			}
			.gk-gravityboard-attachments-metabox .gk-no-attachments-message {
				display: none;
			}
			.gk-gravityboard-attachments-metabox .gk-no-attachments-message.visible {
				display: block;
			}
			.gk-board-feed-title {
				margin: 15px 0 10px;
				padding-bottom: 5px;
				border-bottom: 1px solid #ddd;
			}
			.gk-board-feed-attachments:first-child .gk-board-feed-title {
				margin-top: 0;
			}
		</style>
		<?php
	}

	/**
	 * Renders inline JavaScript for attachment deletion functionality.
	 *
	 * @since TODO
	 *
	 * @return void
	 */
	private function render_inline_scripts() {
		$rest_url = rest_url( 'gravityboard/v1/boards/' );
		?>
		<script>
			jQuery(document).ready(function($) {
				// Configuration constants
				const SUCCESS_MESSAGE_DELAY = 7000;
				const FADE_OUT_DELAY = 300;

				$('.gk-gravityboard-attachments-metabox').on('click', '.gk-delete-attachment-button', function(e) {
					e.preventDefault();

					if ( ! confirm('<?php echo esc_js( __( 'Are you sure you want to delete this attachment?', 'gk-gravityboard' ) ); ?>') ) {
						return;
					}

					var $link = $(this);
					var entryId = $link.data('entry-id');
					var attachmentId = $link.data('attachment-id');
					var feedId = $link.data('feed-id');
					var nonce = $link.data('nonce');
					var $listItem = $('#gk-attachment-item-' + attachmentId);
					var $statusP = $('#gk-board-attachment-status-' + entryId);
					var $feedSection = $listItem.closest('.gk-board-feed-attachments');

					// Disable button to prevent multiple clicks and update for screen readers.
					$link.prop('disabled', true).attr('aria-busy', 'true');
					$statusP.text('<?php echo esc_js( __( 'Deleting…', 'gk-gravityboard' ) ); ?>').addClass('visible');

					$.ajax({
						url: '<?php echo esc_js( $rest_url ); ?>' + feedId + '/cards/' + entryId + '/attachments/' + attachmentId,
						type: 'DELETE',
						beforeSend: function(xhr) {
							xhr.setRequestHeader('X-WP-Nonce', nonce);
						},
						success: function(response) {
							// REST API returns data directly, not wrapped in success/data.
							if ( response && response.message ) {
								$listItem.fadeOut(FADE_OUT_DELAY, function() {
									$(this).remove();

									// Function to handle the final UI state after all DOM updates complete
									function handlePostDeletionUiState() {
										var remainingAttachmentSections = $('.gk-board-feed-attachments').length;
										var hasRemainingAttachments = remainingAttachmentSections > 0;

										if ( ! hasRemainingAttachments ) {
											$statusP.removeClass('error').text('<?php echo esc_js( __( 'Attachment deleted.', 'gk-gravityboard' ) ); ?>').addClass('visible').delay(SUCCESS_MESSAGE_DELAY).fadeOut(FADE_OUT_DELAY, function() {
												var $emptyStateMessage = $('#gk-board-no-attachments-' + entryId);
												if ($emptyStateMessage.length > 0) {
													$emptyStateMessage.addClass('visible');
												}
											});
										} else {
											$statusP.removeClass('error').text('<?php echo esc_js( __( 'Attachment deleted.', 'gk-gravityboard' ) ); ?>').addClass('visible').delay(SUCCESS_MESSAGE_DELAY).fadeOut();
										}
									}

									// Check if the current feed section is now empty and should be removed.
									var remainingAttachments = $feedSection.find('li').length;
									if ( remainingAttachments === 0 ) {
										$feedSection.fadeOut(FADE_OUT_DELAY, function() {
											$(this).remove();
											// Handle UI state AFTER the empty section is fully removed
											handlePostDeletionUiState();
										});
									} else {
										// Handle UI state immediately since no section removal is needed
										handlePostDeletionUiState();
									}
								});
							} else {
								$statusP.text('<?php echo esc_js( __( 'Error deleting attachment.', 'gk-gravityboard' ) ); ?>').addClass('error visible');
								$link.prop('disabled', false).removeAttr('aria-busy');
							}
						},
						error: function(xhr, status, error) {
							var errorMessage = '<?php echo esc_js( __( 'Error deleting attachment.', 'gk-gravityboard' ) ); ?>';

							if ( xhr.responseJSON && xhr.responseJSON.message ) {
								errorMessage = xhr.responseJSON.message;
							} else if ( xhr.responseText ) {
								try {
									var response = JSON.parse(xhr.responseText);
									if ( response.message ) {
										errorMessage = response.message;
									}
								} catch (e) {
									// Keep default error message.
								}
							}

							$statusP.text(errorMessage).addClass('error visible');
							$link.prop('disabled', false).removeAttr('aria-busy');
						}
					});
				});
			});
		</script>
		<?php
	}
}