<?php
/**
 * Register GravityBoard block
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Blocks\GravityBoard;

use GravityKit\GravityBoard\Helpers;
use GravityKit\GravityBoard\Feed;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register the GravityBoard block
 */
class Block {

	/**
	 * Instance of this class
	 *
	 * @var Block
	 */
	private static $instance = null;

	/**
	 * Get the instance of this class
	 *
	 * @return Block
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'init', [ $this, 'register_block' ] );
	}

	/**
	 * Register the block
	 */
	public function register_block() {
		// Skip block registration if Gutenberg is not available.
		if ( ! function_exists( 'register_block_type' ) ) {
			return;
		}

		// Check if block assets exist.
		$asset_file_path = GRAVITYBOARD_PATH . 'src/Blocks/GravityBoard/build/index.asset.php';

		if ( ! file_exists( $asset_file_path ) ) {
			return;
		}

		$asset_file = include $asset_file_path;

		wp_register_script(
			'gravityboard-block-editor',
			plugins_url( 'build/index.js', __FILE__ ),
			$asset_file['dependencies'],
			$asset_file['version'],
			true
		);

		wp_register_style(
			'gravityboard-block-editor',
			plugins_url( 'build/index.css', __FILE__ ),
			[],
			$asset_file['version']
		);

		wp_localize_script(
			'gravityboard-block-editor',
			'gravityBoardBlock',
			[
				'feeds'         => $this->get_feeds_for_block(),
				'edit_feed_url' => \GravityKit\GravityBoard\Feed::get_edit_feed_url( '{feedId}', '{formId}' ),
				'view_feed_url' => \GravityKit\GravityBoard\Feed::get_view_feed_url( '{feedId}' ),
			]
		);

		register_block_type(
			'gravityboard/board',
			[
				'api_version'     => 2,
				'editor_script'   => 'gravityboard-block-editor',
				'editor_style'    => 'gravityboard-block-editor',
				'render_callback' => [ $this, 'render_block' ],
				'attributes'      => [
					'feedId'    => [
						'type'    => 'string',
						'default' => '',
					],
					'minHeight' => [
						'type'    => 'string',
						'default' => '',
					],
				],
			]
		);
	}

	/**
	 * Get feeds for block
	 *
	 * @return array Array of feeds for dropdown in block
	 */
	private function get_feeds_for_block() {
		$feed_class = \GravityKit\GravityBoard\Feed::get_instance();
		$feeds      = $feed_class->get_active_feeds();

		if ( empty( $feeds ) ) {
			return [];
		}

		$grouped_feeds = [];
		foreach ( $feeds as $feed ) {
			$form_id = ! empty( $feed['form_id'] ) ? $feed['form_id'] : 0;

			// Group feeds by form.
			if ( ! isset( $grouped_feeds[ $form_id ] ) ) {
				$form = Helpers::get_form( $form_id );

				if ( is_wp_error( $form ) || empty( $form ) ) {
					continue;
				}

				$form_title_value = isset( $form['title'] ) && '' !== $form['title'] ? $form['title'] : esc_html__( '(Untitled)', 'gk-gravityboard' );
				// translators: %1$s is Form title, %2$d is Form ID.
				$form_title                = sprintf( esc_html__( '%1$s (Form #%2$d)', 'gk-gravityboard' ), esc_html( $form_title_value ), $form_id );
				$grouped_feeds[ $form_id ] = [
					'form_title' => $form_title,
					'form_id'    => $form_id,
					'feeds'      => [],
				];
			}

			// translators: %d: Board ID.
			$feed_label = ! empty( $feed['meta']['feed_name'] ) ? $feed['meta']['feed_name'] : sprintf( esc_html__( 'Board #%d', 'gk-gravityboard' ), $feed['id'] );

			// Prepare meta data for the block, including background color and image URL.
			$board_background_color     = rgars( $feed, 'meta/board_background_color', 'rgb(241, 242, 244);' );
			$board_background_image_id  = rgars( $feed, 'meta/board_background_image_id' );
			$board_background_image_url = $board_background_image_id ? wp_get_attachment_image_url( $board_background_image_id, 'full' ) : '';

			$grouped_feeds[ $form_id ]['feeds'][] = [
				'id'    => $feed['id'],
				'label' => $feed_label,
				'meta'  => [
					'board_background_color'     => $board_background_color,
					'board_background_image_url' => $board_background_image_url,
					'board_min_height'           => rgars( $feed, 'meta/board_min_height', '400px' ),
				],
			];
		}

		return array_values( $grouped_feeds );
	}

	/**
	 * Render the block
	 *
	 * @param array $attributes Block attributes.
	 * @return string HTML output
	 */
	public function render_block( $attributes ) {
		$feed_id    = ! empty( $attributes['feedId'] ) ? $attributes['feedId'] : '';
		$min_height = ! empty( $attributes['minHeight'] ) ? $attributes['minHeight'] : '';

		if ( empty( $feed_id ) ) {
			return '<div class="gravityboard-block-error">' . esc_html__( 'Please select a GravityBoard to display.', 'gk-gravityboard' ) . '</div>';
		}

		$feed = Feed::get_instance()->get_feed( $feed_id );

		// Check if we're in the admin editor context.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Block render callback, checking REST context.
		if ( defined( 'REST_REQUEST' ) && REST_REQUEST && ! empty( $_GET['context'] ) && 'edit' === $_GET['context'] ) {
			// Get the board name.
			// translators: %s: Board ID.
			$board_name = Feed::get_instance()->get_board_name( $feed );

			// Return a placeholder for the admin editor.
			return sprintf(
				'<div class="gravityboard-admin-placeholder">
					<div class="gravityboard-admin-placeholder-header">
						%1$s
						%2$s
					</div>
					<div class="gravityboard-admin-placeholder-content">
						<p>%3$s</p>
						<p class="gravityboard-admin-placeholder-id">%4$s</p>
					</div>
				</div>',
				'<svg width="52" height="52" viewBox="0 0 52 52" fill="none" xmlns="http://www.w3.org/2000/svg">
					<path d="M19.2773 6C21.9172 6 24.084 8.23854 24.084 11V41C24.0839 43.7614 21.9172 46 19.2773 46H10.8066C8.13213 45.9998 6.00004 43.7613 6 41V11C6 8.23867 8.1321 6.00021 10.8066 6H19.2773ZM41.1934 6C43.833 6.00021 46 8.23867 46 11V41C46 43.7613 43.833 45.9998 41.1934 46H32.7227C30.048 46 27.916 43.7614 27.916 41V11C27.916 8.23854 30.0479 6 32.7227 6H41.1934Z" fill="currentColor"/>
				</svg>',
				esc_html( $board_name ),
				esc_html__( 'Your GravityBoard will appear here on the frontend of your site. To view this board in the Dashboard, visit the GravityKit menu, then click on All Boards.', 'gk-gravityboard' ),
				// translators: %s: Board ID.
				sprintf( esc_html__( 'Board ID: %s', 'gk-gravityboard' ), esc_html( $feed_id ) )
			);
		}

		/**
		 * Add min-height to the CSS variable overrides.
		 *
		 * @param array $css_overrides The CSS variable overrides.
		 *
		 * @return array The CSS variable overrides.
		 */
		$css_overrides_callback = function ( $css_overrides ) use ( $min_height ) {
			if ( ! empty( $min_height ) ) {
				$css_overrides['--gravityboard-min-height'] = sprintf( '%s !important', esc_attr( $min_height ) );
			}
			return $css_overrides;
		};

		add_filter( 'gk/gravityboard/renderer/css-variable-overrides', $css_overrides_callback, 10, 1 );

		$shortcode = sprintf( '[gravityboard id="%s"]', esc_attr( $feed_id ) );
		$output    = do_shortcode( $shortcode );

		remove_filter( 'gk/gravityboard/renderer/css-variable-overrides', $css_overrides_callback, 10 );

		return $output;
	}
}
