<?php
/**
 * Boards overview page controller.
 *
 * @package GravityKit\GravityBoards
 */

namespace GravityKit\GravityBoard\BoardsOverview;

use GFForms;
use GFFormsModel;
use GFAPI;
use GFCommon;
use GravityKit\GravityBoard\Foundation\Helpers\Arr;
use GravityKit\GravityBoard\Foundation\Helpers\Core as CoreHelpers;
use GravityKit\GravityBoard\Helpers;
use GravityKit\GravityBoard\Feed;

/**
 * Boards overview class.
 */
class Boards_Overview {

	const PAGE_ID = 'gk_gravityboards';

	const SCREEN_OPTION_PER_PAGE = 'gk_gravityboard_boards_per_page';

	/**
	 * Singleton instance.
	 *
	 * @var Boards_Overview|null
	 */
	private static $instance = null;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_filter( 'gk/foundation/inline-styles', [ $this, 'add_styles' ] );
		add_filter( 'gk/foundation/inline-scripts', [ $this, 'add_scripts' ] );
		add_filter( 'gk/foundation/admin-menu/submenus', [ $this, 'add_gk_submenu' ], 30 );
		add_filter( 'gk/foundation/integrations/helpscout/display', [ $this, 'enable_helpscout' ] );
		add_filter( 'gk/foundation/integrations/helpscout/configuration', [ $this, 'set_helpscout_configuration' ], 10, 1 );
		add_action( 'admin_init', [ $this, 'disable_notices' ] );
		add_action( 'admin_init', [ $this, 'enqueue_form_script' ] );
		add_action( 'current_screen', [ $this, 'setup_screen_options' ] );
		add_filter( 'set-screen-option', [ $this, 'set_screen_options' ], 10, 3 );
	}

	/**
	 * Enables HelpScout beacon for the boards overview page.
	 *
	 * @since 1.0
	 *
	 * @param bool $display Whether to display the beacon.
	 *
	 * @return bool
	 */
	public function enable_helpscout( $display ) {
		if ( $this->is_boards_overview_page() ) {
			return true;
		}
		return $display;
	}

	/**
	 * Sets the HelpScout beacon configuration for the boards overview page.
	 *
	 * @since 1.0
	 *
	 * @param array $configuration The HelpScout beacon configuration.
	 *
	 * @return array
	 */
	public function set_helpscout_configuration( $configuration ) {
		if ( $this->is_boards_overview_page() ) {
			$configuration['suggest'] = Helpers::get_beacon_suggestions();
		}

		return $configuration;
	}

	/**
	 * Returns class instance.
	 *
	 * @since 1.0.0
	 *
	 * @return Boards_Overview
	 */
	public static function get_instance(): self {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Enqueues the form admin script to enable status toggling functionality.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function enqueue_form_script() {
		wp_enqueue_script( 'gform_form_admin' );
	}

	/**
	 * Disables all notices.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function disable_notices() {
		if ( ! $this->is_boards_overview_page() ) {
			return;
		}
		add_action(
			'admin_enqueue_scripts',
			function () {
				remove_all_actions( 'admin_notices' );
			}
		);
		add_filter(
			'admin_enqueue_scripts',
			function () {
				remove_all_filters( 'update_footer' );
			}
		);
		add_action(
			'admin_footer_text',
			function () {
				return '';
			}
		);
	}

	/**
	 * Adds GravityMigrate submenu to the GravityView menu above "Grant Support Access" .
	 *
	 * @since 1.0.0
	 *
	 * @param array $submenus List of submenus.
	 *
	 * @return array
	 */
	public function add_gk_submenu( $submenus ) {

		$page_id          = self::PAGE_ID;
		$submenu_position = 'before_bottom';

		if ( ! did_action( 'gk/foundation/initialized' ) || Arr::get( $submenus, "{$submenu_position}.{$page_id}" ) ) {
			return $submenus;
		}

		if ( CoreHelpers::is_network_admin() ) {
			return $submenus;
		}

		// Get a list of all boards. Then check to see if the user has access to view any of them.
		$board_feeds = GFAPI::get_feeds( null, null, Feed::get_instance()->get_slug(), true );

		$user_can_edit_forms = GFCommon::current_user_can_any( 'gravityforms_edit_forms' );

		$user_has_access = Helpers::can_current_user_perform_action_on_any_feed( 'view_board', $board_feeds );

		// If the user doesn't have access to view any boards, don't show the submenu. Again, if they can't edit forms, they can't create boards.
		if ( ! $user_has_access && ! $user_can_edit_forms ) {
			return $submenus;
		}

		$submenus[ $submenu_position ][ $page_id ] = [
			'page_title' => esc_html__( 'Boards', 'gk-gravityboard' ),
			'menu_title' => esc_html__( 'All Boards', 'gk-gravityboard' ),
			'capability' => 'read', // We already determined that the user has access to the boards page.
			'id'         => $page_id,
			'order'      => 2,
			'callback'   => [ $this, 'boards_list' ],
		];

		// Ensure submenu positions are set in the correct order.
		$positions = [ 'top', 'center', 'before_bottom', 'bottom' ];
		foreach ( $positions as $position ) {
			$ordered_submenus[ $position ] = $submenus[ $position ] ?? [];
		}

		return array_merge( $ordered_submenus, $submenus );
	}


	/**
	 * Prints boards overview table.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function boards_list(): void {
		$form_id = GFForms::get( 'id' );
		$form_id = $form_id ? intval( $form_id ) : null;

		$boards_overview_feed = new Boards_Overview_Feed();
		$table                = $boards_overview_feed->get_boards_table( $form_id );
		$table->prepare_items();

		$class_name = '';
		// Don't show forms if the user doesn't have permission to edit forms.
		if ( GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			$class_name = 'has-forms-dropdown';
		}
		?>
		<form class="gk-gravityboards-overview wrap <?php echo esc_attr( $class_name ); ?>">
			<div class="gk-title">
				<svg width="80" height="80" viewBox="0 0 80 80" fill="none" xmlns="http://www.w3.org/2000/svg">
					<g clip-path="url(#clip0_7850_1254)">
						<rect width="100" height="100" fill="#0079BF" />
						<path fill-rule="evenodd" clip-rule="evenodd" d="M37.7971 54.6666V56.5C37.7971 59.5376 35.3049 62 32.2691 62H22.5281C19.4521 62 17 59.5376 17 56.5L17 29.0001V25.3333V23.5001C17 20.4625 19.4521 18.0001 22.5281 18.0001L32.2691 18.0001C35.3049 18.0001 37.7971 20.4625 37.7971 23.5001V51.0001V54.6666ZM32.2691 21.6667C33.2691 21.6667 34.1115 22.4875 34.1115 23.5001V56.5C34.1115 57.5125 33.2691 58.3334 32.2691 58.3334H22.5281C21.488 58.3334 20.6856 57.5125 20.6856 56.5L20.6856 23.5001C20.6856 22.4875 21.488 21.6667 22.5281 21.6667H32.2691Z" fill="white" />
						<path fill-rule="evenodd" clip-rule="evenodd" d="M43 46.8333L43 45C43 41.9624 45.4922 39.5 48.5281 39.5L58.2691 39.5C61.345 39.5 63.7971 41.9624 63.7971 45L63.7971 44L63.7971 47.6667L63.7971 56.5C63.7971 59.5376 61.345 62 58.2691 62L48.5281 62C45.4922 62 43 59.5376 43 56.5L43 57.4999L43 46.8333ZM48.5281 58.3333C47.528 58.3333 46.6856 57.5126 46.6856 56.5L46.6856 45C46.6856 43.9875 47.528 43.1666 48.5281 43.1666L58.2691 43.1666C59.3092 43.1666 60.1115 43.9875 60.1115 45L60.1115 56.5C60.1115 57.5126 59.3092 58.3333 58.2691 58.3333L48.5281 58.3333Z" fill="white" />
						<path fill-rule="evenodd" clip-rule="evenodd" d="M63.7971 28.1667V30C63.7971 33.0376 61.3049 35.5 58.2691 35.5H48.5281C45.4521 35.5 43 33.0376 43 30V31V27.3333V23.5C43 20.4624 45.4521 18 48.5281 18L58.2691 18C61.3049 18 63.7971 20.4624 63.7971 23.5V22.5001V28.1667ZM58.2691 21.6667C59.2691 21.6667 60.1115 22.4874 60.1115 23.5V30C60.1115 31.0125 59.2691 31.8334 58.2691 31.8334H48.5281C47.488 31.8334 46.6856 31.0125 46.6856 30V23.5C46.6856 22.4874 47.488 21.6667 48.5281 21.6667H58.2691Z" fill="white" />
					</g>
					<defs>
						<clipPath id="clip0_7850_1254">
							<rect width="80" height="80" rx="8" fill="white" />
						</clipPath>
					</defs>
				</svg>

				<h1><?php esc_html_e( 'Boards', 'gk-gravityboard' ); ?></h1>
			</div>

			<div>
				<?php $table->display(); ?>
				<?php wp_nonce_field( 'feed_list', 'feed_list' ); ?>
			</div>
		</form>
		<?php
	}


	/**
	 * Checks if the current page is the plugin settings page.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function is_boards_overview_page(): bool {
		if ( ! did_action( 'gk/foundation/initialized' ) ) {
			return false;
		}

		return filter_input( INPUT_GET, 'page' ) === self::PAGE_ID;
	}

	/**
	 * Adds styles that hide GravityKit Settings sidebar and styles the table.
	 *
	 * @since 1.0.0
	 *
	 * @param array $styles Inline styles to be added to the page.
	 *
	 * @return array
	 */
	public function add_styles( $styles ) {
		if ( ! $this->is_boards_overview_page() ) {
			return $styles;
		}

		$styles[] = [
			'style' => <<<CSS
				.gk-forms-dropdown {
					display: flex;
					align-items: center;
					justify-content: flex-end;
				   margin: 32px 0 16px;
				}

				@media only screen and (max-width: 780px) {
				  .gk-forms-dropdown {
					   margin-bottom: 64px;
					   text-align: center;
				   }
				}

				.gk-forms-dropdown select {
					border: 1px solid #9092b2;
					border-radius: 3px;
					line-height: 2;
					padding: 0.2rem 2rem 0.2rem 1rem;
					max-width: 12.5rem;
					margin-right: 6px;
					color: #242748;
					background: url(data:image/svg+xml;charset=US-ASCII,%3Csvg%20width%3D%2220%22%20height%3D%2220%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M5%206l5%205%205-5%202%201-7%207-7-7%202-1z%22%20fill%3D%22%239092B2%22%2F%3E%3C%2Fsvg%3E) no-repeat right .6rem top 55%;
					background-size: 1rem 1rem;
					background-color: #fff;
				}

				.gk-forms-dropdown select:hover {
					box-shadow: 0 .25rem .25rem rgba(18, 25, 97, .0405344);
					color: #242748;
				}

				.gk-forms-dropdown .button {
					min-height: auto;
					background: #fff;
					border: 1px solid #3e7da6;
					border-radius: 3px;
					box-shadow: 0 2px 1px rgba(28,31,63,.0634624);
					height: 2.3125rem;
					line-height: .875rem;
					padding: .625rem 1.125rem;
					font-size: .875rem;
					color: #3e7da6;
					font-weight: 500;
				}

				.gk-forms-dropdown .button:hover,
				.gk-forms-dropdown .button:active {
					box-shadow: 0 4px 6px rgba(28,31,63,.0837013);
					border: 1px solid #3e7da6;
					background: #fff;
					color: #3985b7;
				}

				.gk-title {
				    background: white;
                    margin-left: -20px;
                    margin-right: -20px;
                    margin-top: -10px;
                    padding-left: 20px;
                    padding-right: 20px;
                    padding-top: 10px;
				}

                .gk-title svg {
                    width: 36px;
                    height: 36px;
                    display: inline-block;
                    position: relative;
                    top: 11px;
                    margin-right: 16px;
				}

				.gk-gravityboards-overview h1 {
				   margin-bottom: 20px;
				   display: inline-block;
				}

				.gk-gravityboards-overview.has-forms-dropdown .tablenav.top {
				   margin-top: -48px;
				}

				.gk-gravityboards-overview .column-is_active {
                   width: 100px;
                }

                .gk-gravityboards-overview .column-board_id {
                   width: 50px;
                }

                .gk-gravityboards-overview .column-preview_link {
                   width: 150px;
                }
CSS
			,
		];

		return $styles;
	}

	/**
	 * Adds scripts.
	 *
	 * @since 1.0.0
	 *
	 * @param array $scripts Scripts array.
	 *
	 * @return array
	 */
	public function add_scripts( $scripts ) {
		if ( ! $this->is_boards_overview_page() ) {
			return $scripts;
		}

		$scripts[] = [
			'script' => <<<JS
				jQuery(document).ready(function($){
					const dropdown = $('.gk-forms-dropdown select');

					// Handle form submission for dropdowns and pagination.
					$('form.gk-gravityboards-overview').on('submit', function(e) {
					  e.preventDefault();

					  const url = new URL(window.location.href);
					  const id = Number(dropdown.find(":selected").val()) || 0;
					  const currentId = Number(url.searchParams.get('id')) || 0;

					  if( id !== currentId ){
						id > 0 ? url.searchParams.set('id', String(id)) : url.searchParams.delete('id');
						url.searchParams.delete('paged');
						window.location.href = url.toString();
						return;
					  }

					  const page = $(this).find('.current-page').val();
					  page > 1 ? url.searchParams.set('paged', page) : url.searchParams.delete('paged');

					  window.location.href = url.toString();
					});

					// Handle live count updates when toggling status.
					if (typeof window.gaddon !== 'undefined' && typeof window.gaddon.toggleFeedSwitch === 'function') {
						// Store the original gaddon.toggleFeedSwitch function.
						const originalToggleFeedSwitch = window.gaddon.toggleFeedSwitch;

						// Override gaddon.toggleFeedSwitch to update counts.
						window.gaddon.toggleFeedSwitch = function(btn, is_active) {
							originalToggleFeedSwitch(btn, is_active);
							updateStatusCounts(is_active);
						};
					}

					/**
					 * Updates the displayed counts for active and inactive items.
					 *
					 * @param {boolean} isActive - Whether the item is currently active.
					 *                             If true, it's being deactivated. If false, it's being activated.
					 */
					function updateStatusCounts( is_active ) {
						const activeCountEl = $('.gk-gravityboards-overview .subsubsub .active .count');
						const inactiveCountEl = $('.gk-gravityboards-overview .subsubsub .inactive .count');

						// Return the number inside parentheses.
						function parseCount( text ) {
							return parseInt( text.replace( '(', '' ).replace( ')', '' ), 10 ) || 0;
						}

						// Get the current counts.
						let activeCount = parseCount( activeCountEl.text() );
						let inactiveCount = parseCount( inactiveCountEl.text() );

						// Update the counts based on the current state.
						if ( is_active ) {
							activeCount--;
							inactiveCount++;
						} else {
							activeCount++;
							inactiveCount--;
						}

						// Update the displayed counts, with parentheses and leading space.
						activeCountEl.text( ' (' + activeCount + ')' );
						inactiveCountEl.text( ' (' + inactiveCount + ')' );
					}
				});
JS
			,
		];

		return $scripts;
	}

	/**
	 * Setup screen options for the boards overview page.
	 *
	 * @since 1.1
	 *
	 * @return void
	 */
	public function setup_screen_options() {
		if ( ! $this->is_boards_overview_page() ) {
			return;
		}

		add_screen_option(
            'per_page',
            [
				'label'   => esc_html__( 'Boards per page', 'gk-gravityboard' ),
				'default' => 10,
				'option'  => self::SCREEN_OPTION_PER_PAGE,
			]
        );
	}

	/**
	 * Handle screen options saving.
	 *
	 * @since 1.1
	 *
	 * @param mixed  $screen_option The value to save instead of the option value. Default false.
	 * @param string $option The option name.
	 * @param int    $value  The option value.
	 *
	 * @return bool|int
	 */
	public function set_screen_options( $screen_option, $option, $value ) {
		if ( self::SCREEN_OPTION_PER_PAGE !== $option ) {
			return $screen_option;
		}

		return $value;
	}
}
