<?php
/**
 * Gravity Forms Boards Overview Feed Add-on.
 *
 * @package GravityKit\GravityBoards
 */

namespace GravityKit\GravityBoard\BoardsOverview;

use GFCommon;
use GFForms;
use GFFormsModel;
use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Helpers;

/**
 * GravityBoards Board Feed.
 */
class Boards_Overview_Feed extends Feed {

	/**
	 * Configures which columns should be displayed on the feed list page.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function feed_list_columns(): array {
		$columns = [
			'board_id'   => esc_html__( 'ID', 'gk-gravityboard' ),
			'board_name' => esc_html__( 'Name', 'gk-gravityboard' ),
			'form_name'  => esc_html__( 'Form', 'gk-gravityboard' ),
			'shortcode'  => sprintf(
				'%s <small>(%s)</small>',
				esc_html__( 'Shortcode', 'gk-gravityboard' ),
				esc_html__( 'Click to copy', 'gk-gravityboard' )
			),
			'view_board' => esc_html__( 'View Board', 'gk-gravityboard' ),
		];

		// If the user doesn't have permission to edit posts, no need for the shortcode column.
		if ( ! current_user_can( 'edit_posts' ) ) {
			unset( $columns['shortcode'] );
		}

		return $columns;
	}

	/**
	 * Gets Boards Overview Table class.
	 *
	 * @since 1.0.0
	 *
	 * @param ?int $form_id Form ID.
	 *
	 * @return Boards_Overview_Table
	 */
	public function get_boards_table( ?int $form_id = null ): Boards_Overview_Table {
		$columns               = $this->feed_list_columns();
		$column_value_callback = [ $this, 'get_column_value' ];
		$feeds                 = $this->get_feeds( $form_id );
		$bulk_actions          = '';
		$action_links          = [];
		$no_item_callback      = [ $this, 'empty_feeds_message_string' ];
		$message_callback      = [ $this, 'feed_list_message' ];

		return new Boards_Overview_Table( $feeds, $this->get_slug(), $columns, $bulk_actions, $action_links, $column_value_callback, $no_item_callback, $message_callback, $this );
	}

	/**
	 * Filter the feeds to only include the ones that the user has access to.
	 *
	 * @since 1.0.2
	 *
	 * @param ?int $form_id Form ID.
	 *
	 * @return array
	 */
	public function get_feeds( $form_id = null ): array {
		$feeds = parent::get_feeds( $form_id );

		// If the user doesn't have permission to edit forms, filter the feeds to only include the ones that the user has access to.
		// This means no inactive feeds will be shown.
		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			$feeds = array_filter(
                $feeds,
                function ( $feed ) {
					return Helpers::user_has_permission( 'view_board', $feed );
				}
            );
		}

		return $feeds;
	}

	/**
	 * Always show the items that have been prepared for the table.
	 *
	 * @since 1.0.2
	 *
	 * @return false
	 */
	public function feed_list_message() {
		return false;
	}

	/**
	 * Generates empty feeds message string, based on whether there are forms configured or not.
	 *
	 * @since 1.0.0
	 *
	 * @param string $url URL of the create feed link.
	 *
	 * @return string
	 */
	public function empty_feeds_message_string( $url = '' ): string {
		// Get the current form id or the first one from the list.
		$form_id = GFForms::get( 'id' );

		$form_count = count( GFFormsModel::get_form_ids() );

		$admin_url = admin_url( 'admin.php' );

		if ( 0 === $form_count ) {
			$url = add_query_arg( [ 'page' => 'gf_new_form' ], $admin_url );

			// translators: [url] is replaced by create new form page URL.
			$template = esc_html__( "There are no forms configured yet. Let's go [url]create one[/url]!", 'gk-gravityboard' );

			return strtr(
				$template,
				[
					'[url]'  => "<a href='" . esc_url( $url ) . "'>",
					'[/url]' => '</a>',
				]
			);
		}

		if ( $form_id ) {

			// translators: [form_name] is replaced by the form name, [url] is replaced by the create new feed URL.
			$template = esc_html__( "There are no boards configured yet for [form_name]. Let's go [url]create one[/url]!", 'gk-gravityboard' );

			// Create link to create new feed for this form.
			$url = Feed::get_edit_feed_url( 0, $form_id );

			$form_name = GFFormsModel::get_form_meta( $form_id )['title'];

			return strtr(
				$template,
				[
					'[form_name]' => esc_html( $form_name ),
					'[url]'       => "<a href='" . esc_url( $url ) . "'>",
					'[/url]'      => '</a>',
				]
			);
		}

		// translators: [url] is replaced by get started URL.
		$template = esc_html__( 'There are no boards configured yet. [url]Learn how to get started![/url]', 'gk-gravityboard' );
		$url      = Helpers::get_getting_started_url();

		return strtr(
			$template,
			[
				'[url]'  => "<a href='" . esc_url( $url ) . "' rel='noopener noreferrer' target='_blank'>",
				'[/url]' => '<span class="screen-reader-text"> ' . esc_html__( '(This link opens in a new window.)', 'gk-gravityboard' ) . '</span></a>',
			]
		);
	}

	/**
	 * Shows the Board Name value.
	 *
	 * @since 1.0.0
	 * @used-by GFFeedAddOn::get_column_value()
	 *
	 * @param array $feed The feed being included in the feed list.
	 *
	 * @return string
	 */
	public function get_column_value_board_name( array $feed ): string {
		$name = rgars( $feed, 'meta/feed_name' ) ?
			$feed['meta']['feed_name'] :
			/* translators: board name. %d: board ID. */
			sprintf( __( 'Board #%d', 'gk-gravityboard' ), $feed['id'] );

		// Only show the link if the user has permission to edit forms.
		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			return esc_html( $name );
		}

		$url = admin_url(
			sprintf(
				'admin.php?page=gf_edit_forms&view=settings&subview=%1$s&fid=%2$s&id=%3$s',
				$this->_slug,
				$feed['id'],
				$feed['form_id']
			)
		);

		return sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url ), esc_html( $name ) );
	}

	/**
	 * Shows the Form Name value.
	 *
	 * @since 1.0.0
	 * @used-by GFFeedAddOn::get_column_value()
	 *
	 * @param array $feed The feed being included in the feed list.
	 *
	 * @return string
	 */
	public function get_column_value_form_name( array $feed ): string {
		$name = rgar( $feed['meta'], 'formName' );

		// Only show the link if the user has permission to edit forms.
		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			return esc_html( $name );
		}

		$url = admin_url(
			sprintf(
				'admin.php?page=gf_edit_forms&view=settings&subview=%1$s&fid=%2$d&id=%3$d',
				'settings',
				intval( $feed['id'] ),
				intval( $feed['form_id'] )
			)
		);

		return sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url ), esc_html( $name ) );
	}

	/**
	 * Get the Board ID value.
	 *
	 * @since 1.0.0
	 *
	 * @used-by GFFeedAddOn::get_column_value()
	 *
	 * @param array $feed The feed being included in the feed list.
	 *
	 * @return string
	 */
	public function get_column_value_board_id( array $feed ): string {
		return esc_html( $feed['id'] );
	}
}
