<?php
/**
 * Gravity Forms Boards Overview Table.
 *
 * @package GravityKit\GravityBoards
 */

namespace GravityKit\GravityBoard\BoardsOverview;

use GravityKit\GravityBoard\Admin\Board_View;
use GravityKit\GravityBoard\Feed;
use GFCommon;
use GFAddOnFeedsTable;

/**
 * Gravity Forms Boards Overview Table.
 */
class Boards_Overview_Table extends GFAddOnFeedsTable {

	const FEEDS_PER_PAGE = 10;

	/**
	 * The current filter (all, active, inactive).
	 *
	 * @var string
	 */
	public $filter = '';

	/**
	 * Original feeds data for counting.
	 *
	 * @var array
	 */
	private $original_feeds = [];

	/**
	 * Whether the table navigation has been displayed.
	 *
	 * @var bool
	 */
	private static $displayed_bulk_actions = false;

	/**
	 * Table constructor.
	 *
	 * @since 1.0.0
	 *
	 * @param array    $feeds Feeds array.
	 * @param string   $slug Slug.
	 * @param string[] $columns Columns.
	 * @param array    $bulk_actions Bulk actions array.
	 * @param array    $action_links Action links array.
	 * @param callable $column_value_callback Column value callback.
	 * @param callable $no_items_callback No items callback.
	 * @param callable $message_callback Message callback.
	 * @param object   $addon_class Addon class.
	 */
	public function __construct( $feeds, $slug, $columns, $bulk_actions, $action_links, $column_value_callback, $no_items_callback, $message_callback, $addon_class ) {
		parent::__construct( $feeds, $slug, $columns, $bulk_actions, $action_links, $column_value_callback, $no_items_callback, $message_callback, $addon_class );

		// Store original feeds for counting.
		$this->original_feeds = $feeds;
		$this->filter         = $this->get_filter();

		// Don't show the checkbox column for bulk actions.
		unset( $this->_column_headers[0]['cb'] );

		// Don't show the active column if the user doesn't have permission to edit forms.
		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			unset( $this->_column_headers[0]['is_active'] );
		}

		// Reorder the board ID column.
		$this->_column_headers[0] = array_merge(
			[ 'board_id' => $this->_column_headers[0]['board_id'] ],
			$this->_column_headers[0]
		);

		$this->set_columns();
	}

	/**
	 * Get the sanitized current filter from the URL parameter. Only users who can edit forms can filter by status.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	private function get_filter() {

		// Only users who can edit forms can filter by status.
		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			return 'active';
		}

		$raw_filter = sanitize_text_field( $_GET['filter'] ?? '' );
		return in_array( $raw_filter, [ 'all', 'active', 'inactive' ], true ) ? $raw_filter : '';
	}

	/**
	 * Disable the Add New button.
	 *
	 * @since 1.0
	 *
	 * @param string $which The location.
	 *
	 * @return bool
	 */
	protected function is_new_button_supported( $which ) {
		return false;
	}

	/**
	 * Set the hidden, sortable and primary columns.
	 *
	 * @since 1.0.0
	 */
	public function set_columns() {
		$columns               = $this->get_columns();
		$hidden                = [];
		$sortable              = $this->get_sortable_columns();
		$primary               = $this->get_primary_column_name();
		$this->_column_headers = [ $columns, $hidden, $sortable, $primary ];
	}

	/**
	 * Order and paginate the items.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function prepare_items(): void {
		// Start with the original feeds we stored in constructor.
		$this->items = $this->original_feeds;

		// Apply all our custom filters.
		$this->items = $this->filter_items( $this->items );
		$this->items = $this->add_form_names( $this->items );

		// Apply status filtering BEFORE counting and pagination.
		$this->items = $this->filter_by_status( $this->items );

		$per_page = get_user_option( Boards_Overview::SCREEN_OPTION_PER_PAGE ) ?: self::FEEDS_PER_PAGE;

		$page = filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT );
		$page = $page > 0 ? $page : 1;

		$orderby = filter_input( INPUT_GET, 'orderby' );

		if ( $orderby ) {
			$this->items = $this->sort_items( $this->items, $orderby, filter_input( INPUT_GET, 'order' ) );
		}

		// Count AFTER filtering.
		$total_items = $this->items ? count( $this->items ) : 0;

		$this->items = $this->items ? array_slice( $this->items, ( $page - 1 ) * $per_page, $per_page ) : [];

		$this->set_pagination_args(
			[
				'total_items' => $total_items,
				'per_page'    => $per_page,
			]
		);
	}

	/**
	 * Filters out items from trashed or missing forms.
	 *
	 * @param array $items Table items.
	 *
	 * @return array
	 */
	protected function filter_items( array $items ): array {
		return $this->filter_items_by_callback(
            $items,
            function ( $item ) {
				$form = \GFFormsModel::get_form( $item['form_id'] );
				return (bool) $form;
			}
        );
	}

	/**
	 * Adds missing form names to the table items.
	 *
	 * @param array $items Table items.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	protected function add_form_names( array $items ): array {
		return $this->transform_items(
            $items,
            function ( $item ) {
				$form_meta = \GFFormsModel::get_form_meta( $item['form_id'] );

				$item['meta']['formName'] = rgars( $form_meta, 'title' ) ?
				$form_meta['title'] :
				/* translators: Form name. %d: Form ID. */
				sprintf( __( 'Form #%d', 'gk-gravityboard' ), $item['form_id'] );

				return $item;
			}
        );
	}

	/**
	 * Filter items by status based on the current filter
	 *
	 * @since 1.0
	 *
	 * @param array $items Table items.
	 *
	 * @return array
	 */
	protected function filter_by_status( array $items ): array {
		if ( empty( $this->filter ) || $this->filter === 'all' ) {
			return $items;
		}

		return $this->filter_items_by_callback(
            $items,
            function ( $item ) {
				$is_active = $this->is_item_active( $item );

				return ( $this->filter === 'active' && $is_active ) ||
					( $this->filter === 'inactive' && ! $is_active );
			}
        );
	}

	/**
	 * Check if an item is active
	 *
	 * @since 1.1
	 *
	 * @param array $item The item to check.
	 *
	 * @return bool
	 */
	protected function is_item_active( array $item ): bool {
		$is_active = rgar( $item, 'is_active' );
		return filter_var( $is_active, FILTER_VALIDATE_BOOLEAN );
	}

	/**
	 * Generic method to filter items using a callback
	 *
	 * @since 1.1
	 *
	 * @param array    $items    Items to filter.
	 * @param callable $callback Callback function that returns true to keep the item.
	 *
	 * @return array
	 */
	protected function filter_items_by_callback( array $items, callable $callback ): array {
		$filtered_items = [];
		foreach ( $items as $key => $item ) {
			if ( $callback( $item, $key ) ) {
				$filtered_items[] = $item;
			}
		}
		return $filtered_items;
	}

	/**
	 * Generic method to transform items using a callback
	 *
	 * @since 1.1
	 *
	 * @param array    $items    Items to transform.
	 * @param callable $callback Callback function that returns the transformed item.
	 *
	 * @return array
	 */
	protected function transform_items( array $items, callable $callback ): array {
		$transformed_items = [];
		foreach ( $items as $key => $item ) {
			$transformed_items[] = $callback( $item, $key );
		}
		return $transformed_items;
	}

	/**
	 * Generic method to count items based on conditions
	 *
	 * @since 1.1
	 *
	 * @param array $items      Items to count.
	 * @param array $conditions Array of condition callbacks.
	 *
	 * @return array
	 */
	protected function count_items_by_conditions( array $items, array $conditions ): array {
		$counts = array_fill_keys( array_keys( $conditions ), 0 );

		foreach ( $items as $item ) {
			foreach ( $conditions as $key => $callback ) {
				if ( $callback( $item ) ) {
					++$counts[ $key ];
				}
			}
		}

		return $counts;
	}

	/**
	 * Get the filter links with counts
	 *
	 * @since 1.1
	 * @return array
	 */
	protected function get_filter_links() {
		// Use original feeds and apply the same filtering logic as prepare_items.
		$all_items = $this->original_feeds;

		// Apply the same filtering that happens in prepare_items.
		if ( ! empty( $all_items ) ) {
			$all_items = $this->filter_items( $all_items );
			$all_items = $this->add_form_names( $all_items );
		}

		// Count using the generic counting method.
		$counts = $this->count_items_by_conditions(
            $all_items,
            [
				'active'   => function ( $item ) {
					return $this->is_item_active( $item ); },
				'inactive' => function ( $item ) {
					return ! $this->is_item_active( $item ); },
			]
        );

		$all_count = count( $all_items );

		return [
			[
				'id'    => 'all',
				'count' => $all_count,
				'label' => esc_html__( 'All', 'gk-gravityboard' ),
			],
			[
				'id'    => 'active',
				'count' => $counts['active'],
				'label' => esc_html__( 'Active', 'gk-gravityboard' ),
			],
			[
				'id'    => 'inactive',
				'count' => $counts['inactive'],
				'label' => esc_html__( 'Inactive', 'gk-gravityboard' ),
			],
		];
	}

	/**
	 * Sorts an array
	 *
	 * @since 1.0.0
	 *
	 * @param array       $items Items to sort.
	 * @param string      $orderby Column name to order by.
	 * @param string|null $order Order - asc or desc.
	 *
	 * @return array
	 */
	protected function sort_items( array $items, string $orderby, ?string $order = 'asc' ): array {
		$orderby_map = [
			'board_id'   => 'id',
			'board_name' => 'meta.feed_name',
			'form_name'  => 'meta.formName',
		];

		if ( ! array_key_exists( $orderby, $orderby_map ) ) {
			return $items;
		}

		$path = $orderby_map[ $orderby ];

		usort(
			$items,
			function ( $a, $b ) use ( $path, $order ) {
				$a_value = $this->get_nested_value( $a, $path );
				$b_value = $this->get_nested_value( $b, $path );

				return ( 'asc' === $order ) ? $a_value <=> $b_value : $b_value <=> $a_value;
			}
		);

		return $items;
	}

	/**
	 * Get the value of the item using path string.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $item Item to check.
	 * @param string $path Path to the item's property.
	 *
	 * @return mixed
	 */
	protected function get_nested_value( array $item, string $path ) {
		$path_segments = explode( '.', $path );
		foreach ( $path_segments as $segment ) {
			if ( ! isset( $item[ $segment ] ) ) {
				return null;
			}
			$item = $item[ $segment ];
		}
		return $item;
	}

	/**
	 * Determines sortable columns.
	 *
	 * @since 1.0.0
	 *
	 * @return array[]
	 */
	protected function get_sortable_columns() {
		return [
			'board_id'   => [ 'board_id', false ],
			'board_name' => [ 'board_name', false ],
			'form_name'  => [ 'form_name', false ],
			'view_board' => [ 'view_board', false ],
		];
	}

	/**
	 * Get value for the view_board column.
	 *
	 * @since 1.0
	 *
	 * @param array $item Current item.
	 *
	 * @return string
	 */
	public function get_column_view_board( $item ) {
		return Feed::get_instance()->get_column_value_view_board( $item );
	}

	/**
	 * Override the display_tablenav() method to ensure the bulk actions show up when there are no items.
	 *
	 * @since 1.1
	 *
	 * @param string $which The location.
	 */
	protected function display_tablenav( $which ) {
		parent::display_tablenav( $which );
		self::$displayed_bulk_actions = true;
	}

	/**
	 * Override the has_items() method to ensure the bulk actions show up when there are no items.
	 *
	 * @since 1.1
	 *
	 * @return bool
	 */
	public function has_items() {

		// If the table navigation has not been displayed, we're above the table navigation, so we need to show the bulk actions.
		if ( ! self::$displayed_bulk_actions ) {
			return true;
		}

		return parent::has_items();
	}

	/**
	 * Gets forms dropdown.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	protected function bulk_actions( $which = '' ) {

		if ( 'top' !== $which ) {
			return;
		}

		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			return;
		}

		$forms   = \GFFormsModel::get_forms(); // More performant than GFAPI::get_forms().
		$form_id = (int) \GFForms::get( 'id' );
		?>
		<div class="gk-forms-dropdown">
			<select name="gravity_forms_dropdown">
				<option value="0"><?php esc_html_e( 'Filter by form', 'gk-gravityboard' ); ?></option>
			<?php
			foreach ( $forms as $form ) {
				printf(
					'<option value="%d" %s>%s</option>',
					esc_attr( $form->id ),
					selected( $form_id, $form->id, false ),
					esc_html( $form->title )
				);
			}
			?>
			</select>
			<input type="submit" id="doaction" class="button action" value="<?php esc_html_e( 'Apply', 'gk-gravityboard' ); ?>">
		</div>
		<?php
	}

	/**
	 * Extra table nav—adds the views to the top of the table.
	 *
	 * @since 1.0.0
	 *
	 * @param string $which The location.
	 */
	protected function extra_tablenav( $which ) {
		if ( 'top' !== $which ) {
			return;
		}

		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			return;
		}

		echo '<br class="clear" />';
		$this->views();
	}

	/**
	 * Get the views for the status filter tabs
	 *
	 * @since 1.0
	 * @return array
	 */
	protected function get_views() {
		if ( ! GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			return [];
		}

		$views = [];

		// Get status counts.
		$filter_links = $this->get_filter_links();

		foreach ( $filter_links as $filter_link ) {
			$selected = '';
			if ( $this->filter == '' ) {
				$selected = $filter_link['id'] == 'all' ? 'current' : '';
			} else {
				$selected = ( $this->filter == $filter_link['id'] ) ? 'current' : '';
			}

			// Build the URL preserving existing parameters.
			$url_args = [
				'page' => rgget( 'page' ),
			];

			// Preserve form ID if present.
			if ( rgget( 'id' ) ) {
				$url_args['id'] = rgget( 'id' );
			}

			// Add filter parameter (except for 'all').
			if ( $filter_link['id'] !== 'all' ) {
				$url_args['filter'] = $filter_link['id'];
			}

			$url = add_query_arg( $url_args, admin_url( 'admin.php' ) );

			$link = '<a class="' . $selected . '" href="' . esc_url( $url ) . '">' . esc_html( $filter_link['label'] ) .
			        '<span class="count"> (<span id="' . esc_attr( $filter_link['id'] ) . '_count">' . absint( $filter_link['count'] ) . '</span>)</span></a>';

			$views[ $filter_link['id'] ] = $link;
		}

		return $views;
	}
}
