<?php
/**
 * GravityBoard Card
 *
 * @package GravityBoard
 * @since   1.0
 */

namespace GravityKit\GravityBoard;

use GravityKit\GravityBoard\Cards\CardCollection;
use GFAPI;
use WP_Error;
use WP_REST_Request;

defined( 'ABSPATH' ) || exit;

/**
 * Class Card
 *
 * Handles card creation, validation, and data management for GravityBoard.
 *
 * @since 1.0.0
 */
class Card {

	/**
	 * The card fields that can be mapped.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	const MAPPABLE_FIELDS = [ 'title', 'description', 'label', 'due_date' ];

	/**
	 * Default sanitization functions for unmapped fields.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	const DEFAULT_SANITIZERS = [
		'title'       => 'sanitize_text_field',
		'description' => 'sanitize_textarea_field',
		'label'       => 'sanitize_text_field',
		'due_date'    => 'sanitize_text_field',
	];

	/**
	 * Card ID.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	private string $id;

	/**
	 * Card data.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	private array $data;

	/**
	 * Card lane.
	 *
	 * @since 1.0
	 *
	 * @var Lane|null
	 */
	private ?Lane $lane;

	/**
	 * Feed data.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	private array $feed;

	/**
	 * Form data.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	private array $form;

	/**
	 * Card constructor.
	 *
	 * @since 1.0
	 *
	 * @param string    $id   The card ID.
	 * @param array     $data The card data.
	 * @param Lane|null $lane The card lane.
	 * @param array     $feed The feed data (optional, for operations).
	 * @param array     $form The form data (optional, for operations).
	 */
	public function __construct(
		string $id = '',
		array $data = [],
		?Lane $lane = null,
		array $feed = [],
		array $form = []
	) {
		$this->id   = $id;
		$this->data = array_merge(
			[
				'title'              => '',
				'description'        => '',
				'label'              => '',
				'position'           => 0,
				'due_date'           => '',
				'due_date_timestamp' => null,
			],
			$data
		);
		$this->lane = $lane;
		$this->feed = $feed;
		$this->form = $form;
	}

	/**
	 * Get card ID.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Get a data field value.
	 *
	 * @since 1.0
	 *
	 * @param string $key     The data key.
	 * @param mixed  $default_value Default value if key doesn't exist.
	 *
	 * @return mixed
	 */
	public function get( string $key, $default_value = null ) {
		return $this->data[ $key ] ?? $default_value;
	}

	/**
	 * Set a data field value.
	 *
	 * @since 1.0
	 *
	 * @param string $key   The data key.
	 * @param mixed  $value The value to set.
	 */
	public function set( string $key, $value ): void {
		$this->data[ $key ] = $value;
	}

	/**
	 * Get card title.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_title(): string {
		return (string) $this->get( 'title', '' );
	}

	/**
	 * Set card title.
	 *
	 * @since 1.0
	 *
	 * @param string $title The card title.
	 */
	public function set_title( string $title ): void {
		$this->set( 'title', $title );
	}

	/**
	 * Get card description.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_description(): string {
		return (string) $this->get( 'description', '' );
	}

	/**
	 * Set card description.
	 *
	 * @since 1.0
	 *
	 * @param string $description The card description.
	 */
	public function set_description( string $description ): void {
		$this->set( 'description', $description );
	}

	/**
	 * Get card label.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_label(): string {
		return (string) $this->get( 'label', '' );
	}

	/**
	 * Set card label.
	 *
	 * @since 1.0
	 *
	 * @param string $label The card label.
	 */
	public function set_label( string $label ): void {
		$this->set( 'label', $label );
	}

	/**
	 * Get card position.
	 *
	 * @since 1.0
	 *
	 * @return int
	 */
	public function get_position(): int {
		return (int) $this->get( 'position', 0 );
	}

	/**
	 * Set card position.
	 *
	 * @since 1.0
	 *
	 * @param int $position The card position.
	 */
	public function set_position( int $position ): void {
		$this->set( 'position', $position );
	}

	/**
	 * Get card due date.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_due_date(): string {
		return (string) $this->get( 'due_date', '' );
	}

	/**
	 * Set card due date.
	 *
	 * @since 1.0
	 *
	 * @param string $due_date The card due date.
	 */
	public function set_due_date( string $due_date ): void {
		$this->set( 'due_date', $due_date );
	}

	/**
	 * Get card due date timestamp.
	 *
	 * @since 1.0
	 *
	 * @return int|null
	 */
	public function get_due_date_timestamp(): ?int {
		return $this->get( 'due_date_timestamp' );
	}

	/**
	 * Set card due date timestamp.
	 *
	 * @since 1.0
	 *
	 * @param int|null $timestamp The card due date timestamp.
	 */
	public function set_due_date_timestamp( ?int $timestamp ): void {
		$this->set( 'due_date_timestamp', $timestamp );
	}

	/**
	 * Get card lane.
	 *
	 * @since 1.0
	 *
	 * @return Lane|null
	 */
	public function get_lane(): ?Lane {
		return $this->lane;
	}

	/**
	 * Set card lane.
	 *
	 * @since 1.0
	 *
	 * @param Lane|null $lane The card lane.
	 */
	public function set_lane( ?Lane $lane ): void {
		$this->lane = $lane;
	}

	/**
	 * Convert card to array format.
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	public function to_array(): array {
		return array_merge(
			[ 'id' => $this->id ],
			$this->data,
			[ 'lane_value' => $this->lane ? $this->lane->get_value() : '' ]
		);
	}

	/**
	 * Create a card from request data.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 * @param array           $feed    The feed data.
	 * @param int             $lane_id The lane ID.
	 * @param int             $position The position.
	 *
	 * @return array|WP_Error|ErrorCollection Card data on success, WP_Error on failure, ErrorCollection on validation failure.
	 */
	public static function create_from_request( WP_REST_Request $request, array $feed, int $lane_id, int $position ) {
		$form = Helpers::get_form( $feed['form_id'] );
		if ( is_wp_error( $form ) ) {
			return $form;
		}

		// Validate lane.
		$lane_field = Feed::get_instance()->get_lane_field( $feed );
		if ( ! isset( $lane_field->choices[ $lane_id ] ) ) {
			return new WP_Error( 'invalid_lane', __( 'Invalid lane ID.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		// Create Lane object.
		$lane = new Lane(
			(string) $lane_id,
			$lane_field->choices[ $lane_id ]['text'],
			$lane_field->choices[ $lane_id ]['value'],
			'',        // color.
			new CardCollection(),        // cards.
			$feed,     // feed.
			$form      // form.
		);

		// Create card with new constructor.
		$card = new self( '', [], $lane, $feed, $form );
		$card->set_from_request( $request );

		// Set lane value in data.
		$card->set( 'lane_value', $lane_field->choices[ $lane_id ]['value'] );

		// Validate all fields.
		$validation_result = $card->validate();

		if ( true !== $validation_result ) {
			return $validation_result;
		}

		// Create the entry.
		$entry_data                    = $card->to_entry_array();
		$entry_data['form_id']         = $feed['form_id'];
		$entry_data['status']          = 'active';
		$entry_data['created_by']      = get_current_user_id();
		$entry_data['date_created']    = gmdate( 'Y-m-d H:i:s' );
		$entry_data[ $lane_field->id ] = $lane_field->choices[ $lane_id ]['value'];

		$entry_id = GFAPI::add_entry( $entry_data );
		if ( is_wp_error( $entry_id ) ) {
			return $entry_id;
		}

		// Set position.
		$new_position = $position + 1;
		gform_update_meta( $entry_id, Feed::get_entry_meta_position_key( $feed['id'] ), $new_position );

		return Helpers::get_card( $entry_id );
	}

	/**
	 * Update a card from request data.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 * @param array           $entry   The entry data.
	 * @param array           $feed    The feed data.
	 *
	 * @return array|WP_Error Update result on success, WP_Error on failure.
	 */
	public static function update_from_request( WP_REST_Request $request, array $entry, array $feed ) {
		$form = Helpers::get_form( $feed['form_id'] );
		if ( is_wp_error( $form ) ) {
			return $form;
		}

		// Create card with new constructor (empty data for now).
		$card    = new self( (string) $entry['id'], [], null, $feed, $form );
		$changes = $card->process_field_updates( $request, $entry );

		// Check if validation failed (returns array of WP_Error objects).
		if ( is_array( $changes ) && ! empty( $changes ) && is_wp_error( $changes[0] ) ) {
			// Create a properly formatted validation error response for multiple errors.
			$error_data = [
				'errors' => [],
			];

			foreach ( $changes as $error ) {
				if ( is_wp_error( $error ) ) {
					$error_info             = $error->get_error_data();
					$error_data['errors'][] = [
						'field_name' => $error_info['field_name'] ?? '',
						'message'    => $error->get_error_message(),
					];
				}
			}

			return new WP_Error(
				'validation_failed',
				// translators: %d is the number of fields that failed validation.
				_n( 'Validation failed for %d field.', 'Validation failed for %d fields.', count( $error_data['errors'] ), 'gk-gravityboard' ),
				array_merge( [ 'status' => 400 ], $error_data )
			);
		}

		return [
			'changes'       => $changes,
			'updated_entry' => Helpers::get_card( $entry['id'] ),
		];
	}

	/**
	 * Build card data from entry.
	 *
	 * @since 1.0
	 *
	 * @param array $entry The entry data.
	 * @param array $feed  The feed data.
	 *
	 * @return array The card data.
	 */
	public static function build_data_from_entry( array $entry, array $feed ): array {
		$entry_id = rgar( $entry, 'id' );

		$card = [
			'id'          => $entry_id,
			'title'       => Feed::get_instance()->get_feed_setting( $feed, 'title', $entry, '' ),
			'description' => Feed::get_instance()->get_feed_setting( $feed, 'description', $entry, '' ),
			'label'       => Feed::get_instance()->get_feed_setting( $feed, 'label', $entry, '' ),
			'position'    => (int) gform_get_meta( $entry_id, Feed::get_entry_meta_position_key( $feed['id'] ) ),
		];

		$due_date = Feed::get_instance()->get_feed_setting( $feed, 'due_date', $entry, '' );
		if ( '' !== $due_date ) {
			$timestamp                  = strtotime( $due_date );
			$card['due_date']           = $timestamp ? date_i18n( get_option( 'date_format' ), $timestamp ) : '';
			$card['due_date_timestamp'] = $timestamp ? $timestamp : null;
		} else {
			$card['due_date']           = '';
			$card['due_date_timestamp'] = null;
		}

		$lane_field_id      = Feed::get_instance()->get_lane_field_id( $feed );
		$card['lane_value'] = rgar( $entry, (string) $lane_field_id );

		/**
		 * Filter the card data returned from the entry.
		 *
		 * @since 1.1
		 *
		 * @param array $card The card data.
		 * @param array $entry The entry data.
		 * @param array $feed The feed data.
		 */
		return apply_filters( 'gk/gravityboard/card/data', $card, $entry, $feed );
	}

	/**
	 * Set card data from request.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 */
	private function set_from_request( WP_REST_Request $request ) {
		foreach ( self::MAPPABLE_FIELDS as $field_key ) {
			$value = $request->get_param( $field_key );
			if ( null !== $value ) {
				$this->set_field( $field_key, $value );
			}
		}
	}

	/**
	 * Set a field value with automatic sanitization.
	 *
	 * @since 1.0
	 *
	 * @param string $field_key The field key.
	 * @param mixed  $value     The field value.
	 */
	private function set_field( string $field_key, $value ) {
		$field_id = $this->get_mapped_field_id( $field_key );

		if ( $field_id ) {
			$this->data[ $field_key ] = Helpers::sanitize_field_value( $value, $this->form, $field_id );
		} else {
			$sanitizer                = self::DEFAULT_SANITIZERS[ $field_key ] ?? 'sanitize_text_field';
			$this->data[ $field_key ] = $sanitizer( $value );
		}
	}

	/**
	 * Get the mapped field ID for a field key.
	 *
	 * @since 1.0
	 *
	 * @param string $field_key The field key.
	 *
	 * @return string|null The field ID or null if not mapped.
	 */
	private function get_mapped_field_id( string $field_key ) {
		$field_id = Feed::get_instance()::get_feed_field_by_key( $this->feed, $field_key );
		return $field_id ? $field_id : null;
	}

	/**
	 * Validate all card fields.
	 *
	 * @since 1.0
	 *
	 * @return true|ErrorCollection True on success, ErrorCollection on validation failure.
	 */
	private function validate() {
		// Build field values for validation and field mappings for error handling.
		$field_values   = [];
		$field_mappings = [];

		// Process all mappable fields.
		foreach ( self::MAPPABLE_FIELDS as $field_key ) {
			$field_id = $this->get_mapped_field_id( $field_key );
			if ( $field_id && isset( $this->data[ $field_key ] ) ) {
				$field_values[ $field_id ]            = $this->data[ $field_key ];
				$field_mappings[ (string) $field_id ] = $field_key;
			}
		}

		// Add lane value if set.
		if ( isset( $this->data['lane_value'] ) ) {
			$lane_field                                 = Feed::get_instance()->get_lane_field( $this->feed );
			$field_values[ $lane_field->id ]            = $this->data['lane_value'];
			$field_mappings[ (string) $lane_field->id ] = 'lane_value';
		}

		$validation_data = Helpers::validate_fields( $field_values, $this->form, [], $field_mappings );

		if ( true === $validation_data ) {
			return true;
		}

		return $validation_data;
	}

	/**
	 * Convert card data to entry array.
	 *
	 * @since 1.0
	 *
	 * @return array The entry array.
	 */
	private function to_entry_array(): array {
		$entry = [];

		foreach ( self::MAPPABLE_FIELDS as $field_key ) {
			$field_id = $this->get_mapped_field_id( $field_key );
			$value    = $this->data[ $field_key ] ?? '';

			if ( $field_id && '' !== $value ) {
				$entry[ $field_id ] = $value;
			}
		}

		return $entry;
	}

	/**
	 * Process field updates from request.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 * @param array           $entry   The entry data.
	 *
	 * @return array|WP_Error[]|null Array of changes on success, array of WP_Error objects on validation failure, null if no changes.
	 */
	private function process_field_updates( WP_REST_Request $request, array $entry ) {
		$mapped_fields = Feed::get_instance()->get_mapped_fields( $this->feed );
		$changes       = [];
		$field_updates = [];

		// Build field values for validation, including both changed and unchanged fields.
		$field_values   = [];
		$field_mappings = [];

		foreach ( $mapped_fields as $field_key => $field_id ) {
			$new_value_raw = $request->get_param( $field_key );
			$current_value = rgar( $entry, (string) $field_id );

			// Use existing value for validation.
			$field_values[ $field_id ] = $current_value;

			// Determine the value to use for validation.
			if ( null !== $new_value_raw ) {
				$new_value = Helpers::sanitize_field_value( $new_value_raw, $this->form, $field_id );

				// Track if this field actually changed.
				if ( $new_value !== $current_value ) {
					$field_updates[] = [
						'key'       => $field_key,
						'field_id'  => $field_id,
						'new_value' => $new_value,
					];
				}

				$field_values[ $field_id ] = $new_value;
			}

			$field_mappings[ (string) $field_id ] = $field_key;
		}

		// If no changes, return null.
		if ( empty( $field_updates ) ) {
			return null;
		}

		// Validate all mapped fields to ensure the entry remains valid.
		$validation_result = Helpers::validate_fields( $field_values, $this->form, $entry, $field_mappings );

		if ( true !== $validation_result ) {
			// Return array of WP_Error objects for frontend processing.
			return $validation_result;
		}

		// Apply updates if validation passed.
		foreach ( $field_updates as $update ) {
			$result = is_numeric( $update['field_id'] )
				? GFAPI::update_entry_field( $entry['id'], $update['field_id'], $update['new_value'] )
				: GFAPI::update_entry_property( $entry['id'], $update['field_id'], $update['new_value'] );

			if ( true !== $result && ! is_int( $result ) ) {
				return [ new WP_Error( 'update_failed', __( 'Failed to update field.', 'gk-gravityboard' ), [ 'field_name' => $update['key'] ] ) ];
			}

			$changes[] = $update['key'];
		}

		return $changes;
	}
}
