<?php
/**
 * GravityBoard CardCollection
 *
 * @package GravityBoard
 * @since   1.0
 */

namespace GravityKit\GravityBoard\Cards;

use GravityKit\GravityBoard\Card;
use GravityKit\GravityBoard\Feed;
use IteratorAggregate;
use Countable;
use ArrayIterator;

/**
 * Class CardCollection
 *
 * A container for managing multiple Card instances.
 *
 * @since 1.0.0
 */
class CardCollection implements IteratorAggregate, Countable {

	/**
	 * Array of Card objects.
	 *
	 * @since 1.0
	 *
	 * @var Card[]
	 */
	private array $cards = [];

	/**
	 * CardCollection constructor.
	 *
	 * @since 1.0
	 *
	 * @param Card[] $cards Optional initial cards.
	 */
	public function __construct( array $cards = [] ) {
		$this->set_cards( $cards );
	}

	/**
	 * Set cards for the collection.
	 *
	 * @since 1.0
	 *
	 * @param Card[] $cards Array of Card objects.
	 */
	public function set_cards( array $cards ): void {
		foreach ( $cards as $key => $card ) {
			if ( ! $card instanceof Card ) {
				Feed::get_instance()->log_error( 'All items must be instances of Card.' );
				unset( $cards[ $key ] );
				continue;
			}
		}

		$this->cards = array_values( $cards );
	}

	/**
	 * Add a Card to the collection.
	 *
	 * @since 1.0
	 *
	 * @param Card $card The card to add.
	 */
	public function add_card( Card $card ): void {
		$this->cards[] = $card;
	}

	/**
	 * Remove a card from the collection by ID.
	 *
	 * @since 1.0
	 *
	 * @param string $id The card ID to remove.
	 */
	public function remove_card_by_id( string $id ): void {
		$this->cards = array_filter(
			$this->cards,
			fn( Card $card ) => $card->get_id() !== $id
		);
	}

	/**
	 * Get cards sorted by position.
	 *
	 * @since 1.0
	 *
	 * @return self New CardCollection instance with sorted cards.
	 */
	public function get_sorted_by_position(): self {
		$sorted = $this->cards;
		usort( $sorted, fn( Card $a, Card $b ) => $a->get_position() <=> $b->get_position() );

		return new self( $sorted );
	}

	/**
	 * Find a card by ID.
	 *
	 * @since 1.0
	 *
	 * @param string $id The card ID to search for.
	 *
	 * @return Card|null The found card or null.
	 */
	public function find_by_id( string $id ): ?Card {
		foreach ( $this->cards as $card ) {
			if ( $card->get_id() === $id ) {
				return $card;
			}
		}

		return null;
	}

	/**
	 * Convert collection to array of card data.
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	public function to_array(): array {
		return array_map( fn( Card $card ) => $card->to_array(), $this->cards );
	}

	/**
	 * Get an iterator for the collection.
	 *
	 * @since 1.0
	 *
	 * @return \Traversable
	 */
	public function getIterator(): \Traversable {
		return new ArrayIterator( $this->cards );
	}

	/**
	 * Count the number of cards in the collection.
	 *
	 * @since 1.0
	 *
	 * @return int
	 */
	public function count(): int {
		return count( $this->cards );
	}
}
