<?php
/**
 * GravityBoard Checklist
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Checklists;

use GravityKit\GravityBoard\Feed;

defined( 'ABSPATH' ) || die();

/**
 * Class Checklist
 *
 * Represents a checklist with items, providing validation and utility methods.
 *
 * @since 1.1
 */
class Checklist {

	/**
	 * The checklist ID.
	 *
	 * @var string
	 */
	private string $id;

	/**
	 * The checklist title.
	 *
	 * @var string
	 */
	private string $title;

	/**
	 * The checklist items.
	 *
	 * @var ChecklistItem[]
	 */
	private array $items;

	/**
	 * Constructor.
	 *
	 * @param string|null     $id    The checklist ID, or null to auto-generate.
	 * @param string          $title The checklist title.
	 * @param ChecklistItem[] $items The checklist items.
	 */
	public function __construct( $id = null, string $title = '', array $items = [] ) {
		// Generate ID if null is passed.
		if ( null === $id ) {
			// Use timestamp in milliseconds to match JavaScript's Date.now() pattern.
			$this->id = (string) round( microtime( true ) * 1000 );
		} else {
			// Set ID with fallback to generated ID if validation fails.
			$id_result = $this->set_id( $id );
			if ( is_wp_error( $id_result ) ) {
				$this->id = (string) round( microtime( true ) * 1000 );
			}
		}

		$this->set_title( $title );

		// Set items with fallback to empty array if validation fails.
		$items_result = $this->set_items( $items );
		if ( is_wp_error( $items_result ) ) {
			Feed::get_instance()->log_error( 'Failed to set items for checklist: ' . $items_result->get_error_message() );
			$this->items = [];
		}
	}

	/**
	 * Create from array data.
	 *
	 * @param array $data The array data.
	 * @return Checklist|\WP_Error
	 */
	public static function from_array( array $data ) {
		$id    = sanitize_text_field( $data['id'] ?? uniqid() );
		$title = sanitize_text_field( $data['title'] ?? '' );
		$items = [];

		if ( isset( $data['items'] ) && is_array( $data['items'] ) ) {
			foreach ( $data['items'] as $item_data ) {
				if ( is_array( $item_data ) && ! empty( $item_data['label'] ) ) {
					$item_result = ChecklistItem::from_array( $item_data );
					if ( is_wp_error( $item_result ) ) {
						return new \WP_Error(
							'invalid_checklist_item',
							sprintf(
								/* translators: %s: Error message from checklist item. */
								esc_html__( 'Invalid checklist item: %s', 'gk-gravityboard' ),
								$item_result->get_error_message()
							)
						);
					}
					$items[] = $item_result;
				}
			}
		}

		$checklist = new self( $id, $title, $items );

		$validation_result = $checklist->validate();
		if ( is_wp_error( $validation_result ) ) {
			Feed::get_instance()->log_error( 'Failed to validate checklist: ' . $validation_result->get_error_message() );
			return $validation_result;
		}

		return $checklist;
	}

	/**
	 * Convert to array for API responses (includes calculated progress).
	 *
	 * @return array
	 */
	public function to_array(): array {
		return [
			'id'       => $this->id,
			'title'    => $this->title,
			'items'    => array_map(
                function ( ChecklistItem $item ) {
                    return $item->to_array();
                },
                $this->items
            ),
			'progress' => $this->get_progress(),
		];
	}

	/**
	 * Convert to array for storage (excludes calculated fields like progress).
	 *
	 * @since 1.1
	 * @return array
	 */
	public function to_storage_array(): array {
		return [
			'id'    => $this->id,
			'title' => $this->title,
			'items' => array_map(
                function ( ChecklistItem $item ) {
                    return $item->to_array();
                },
                $this->items
            ),
		];
	}

	/**
	 * Get the checklist ID.
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Set the checklist ID.
	 *
	 * @param string $id The checklist ID.
	 * @return true|\WP_Error True on success, WP_Error on failure.
	 */
	public function set_id( string $id ) {
		if ( empty( trim( $id ) ) ) {
			return new \WP_Error( 'empty_checklist_id', esc_html__( 'Checklist ID cannot be empty', 'gk-gravityboard' ) );
		}
		$this->id = sanitize_text_field( $id );
		return true;
	}

	/**
	 * Get the checklist title.
	 *
	 * @return string
	 */
	public function get_title(): string {
		return $this->title;
	}

	/**
	 * Set the checklist title.
	 *
	 * @param string $title The checklist title.
	 */
	public function set_title( string $title ): void {
		$this->title = sanitize_text_field( $title );
	}

	/**
	 * Get the checklist items.
	 *
	 * @return ChecklistItem[]
	 */
	public function get_items(): array {
		return $this->items;
	}

	/**
	 * Set the checklist items.
	 *
	 * @param ChecklistItem[] $items The checklist items.
	 * @return true|\WP_Error True on success, WP_Error on failure.
	 */
	public function set_items( array $items ) {
		foreach ( $items as $index => $item ) {
			if ( ! $item instanceof ChecklistItem ) {
				return new \WP_Error(
					'invalid_checklist_items',
					sprintf(
						/* translators: %d: A number representing the index of the item. */
						esc_html__( 'Checklist item at index %d is not valid.', 'gk-gravityboard' ),
						$index
					)
				);
			}
		}

		$this->items = $items;
		$this->reorder_items();
		return true;
	}

	/**
	 * Add an item to the checklist.
	 *
	 * @param ChecklistItem $item The item to add.
	 *
	 * @return true|\WP_Error True if item was added, WP_Error if not.
	 */
	public function add_item( ChecklistItem $item ) {
		// Small delay to ensure unique millisecond timestamps for item IDs.
		usleep( 1000 );

		$validation_result = $item->validate();

		if ( is_wp_error( $validation_result ) ) {
			return $validation_result;
		}

		$item->set_position( count( $this->items ) );
		$this->items[] = $item;

		return true;
	}

	/**
	 * Add multiple items to the checklist in batch.
	 * 
	 * This method is optimized for bulk operations and doesn't sleep between items.
	 * It's useful when merging checklists or applying templates where unique IDs
	 * are already guaranteed by the ChecklistItem constructor.
	 *
	 * @since TODO
	 * 
	 * @param ChecklistItem[] $items Array of items to add.
	 *
	 * @return true|\WP_Error True if all items were added, WP_Error if validation fails.
	 */
	public function add_items( array $items ) {
		// Validate all items first.
		foreach ( $items as $index => $item ) {
			if ( ! $item instanceof ChecklistItem ) {
				return new \WP_Error(
					'invalid_checklist_item_type',
					sprintf(
						/* translators: %d: The index of the invalid item. */
						esc_html__( 'Item at index %d is not a ChecklistItem instance.', 'gk-gravityboard' ),
						$index
					)
				);
			}
			
			$validation_result = $item->validate();
			if ( is_wp_error( $validation_result ) ) {
				return new \WP_Error(
					'invalid_checklist_item',
					sprintf(
						/* translators: 1: The index of the invalid item. 2: The error message. */
						esc_html__( 'Item at index %1$d is invalid: %2$s', 'gk-gravityboard' ),
						$index,
						$validation_result->get_error_message()
					)
				);
			}
		}

		// Add all items and set their positions.
		$starting_position = count( $this->items );
		foreach ( $items as $index => $item ) {
			$item->set_position( $starting_position + $index );
			$this->items[] = $item;
		}

		return true;
	}

	/**
	 * Remove an item from the checklist.
	 *
	 * @param string $item_id The item ID to remove.
	 * @return bool True if item was removed, false if not found.
	 */
	public function remove_item( string $item_id ): bool {
		foreach ( $this->items as $index => $item ) {
			if ( $item->get_id() === $item_id ) {
				unset( $this->items[ $index ] );
				$this->items = array_values( $this->items ); // Re-index.
				$this->reorder_items();
				return true;
			}
		}
		return false;
	}

	/**
	 * Find an item by ID.
	 *
	 * @param string $item_id The item ID.
	 * @return ChecklistItem|null The item if found, null otherwise.
	 */
	public function find_item( string $item_id ): ?ChecklistItem {
		foreach ( $this->items as $item ) {
			if ( $item->get_id() === $item_id ) {
				return $item;
			}
		}
		return null;
	}

	/**
	 * Get the index of an item by ID.
	 *
	 * @param string $item_id The item ID.
	 * @return int|null The item index if found, null otherwise.
	 */
	public function find_item_index( string $item_id ): ?int {
		foreach ( $this->items as $index => $item ) {
			if ( $item->get_id() === $item_id ) {
				return $index;
			}
		}
		return null;
	}

	/**
	 * Update an item.
	 *
	 * @param string $item_id The item ID.
	 * @param array  $updates The updates to apply.
	 * @return bool|\WP_Error True if item was updated, false if not found, WP_Error on validation failure.
	 */
	public function update_item( string $item_id, array $updates ) {
		$item = $this->find_item( $item_id );
		if ( null === $item ) {
			return false;
		}

		$update_result = $item->update( $updates );
		if ( is_wp_error( $update_result ) ) {
			return $update_result;
		}

		// If position was updated, reorder items.
		if ( isset( $updates['position'] ) ) {
			$this->reorder_items();
		}

		return true;
	}

	/**
	 * Get completion progress.
	 *
	 * @return array Array with 'completed', 'total', and 'percentage' keys.
	 */
	public function get_progress(): array {
		$total     = count( $this->items );
		$completed = 0;

		foreach ( $this->items as $item ) {
			if ( $item->is_complete() ) {
				++$completed;
			}
		}

		$percentage = $total > 0 ? round( ( $completed / $total ) * 100 ) : 0;

		return [
			'completed'  => $completed,
			'total'      => $total,
			'percentage' => $percentage,
		];
	}

	/**
	 * Check if the checklist is empty.
	 *
	 * @return bool True if empty (no items and no title), false otherwise.
	 */
	public function is_empty(): bool {
		return empty( $this->items ) && '' === trim( $this->title );
	}

	/**
	 * Get the count of items.
	 *
	 * @return int The number of items.
	 */
	public function count_items(): int {
		return count( $this->items );
	}

	/**
	 * Get the count of completed items.
	 *
	 * @return int The number of completed items.
	 */
	public function count_completed_items(): int {
		return count(
            array_filter(
                $this->items,
                function ( ChecklistItem $item ) {
                    return $item->is_complete();
                }
            )
        );
	}

	/**
	 * Check if all items are complete.
	 *
	 * @return bool True if all items are complete, false otherwise.
	 */
	public function is_complete(): bool {
		if ( empty( $this->items ) ) {
			return false;
		}

		foreach ( $this->items as $item ) {
			if ( ! $item->is_complete() ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Validate the checklist.
	 *
	 * @return true|\WP_Error True if valid, WP_Error if not.
	 */
	public function validate() {
		if ( empty( trim( $this->id ) ) ) {
			return new \WP_Error( 'empty_checklist_id', esc_html__( 'Checklist ID cannot be empty', 'gk-gravityboard' ) );
		}

		foreach ( $this->items as $index => $item ) {
			if ( ! $item instanceof ChecklistItem ) {
				return new \WP_Error(
					'invalid_checklist_items',
					sprintf(
						/* translators: %d: A number representing the index of the item. */
						esc_html__( 'Invalid item at index %1$d: %2$s', 'gk-gravityboard' ),
						$index
					)
				);
			}

			$item_validation = $item->validate();
			if ( is_wp_error( $item_validation ) ) {
				return new \WP_Error(
					'invalid_checklist_item',
					sprintf(
						/* translators: 1: ChecklistItem index, 2: Error message. */
						esc_html__( 'Invalid item at index %1$d: %2$s', 'gk-gravityboard' ),
						$index,
						$item_validation->get_error_message()
					)
				);
			}
		}

		return true;
	}

	/**
	 * Reorder items by position to ensure proper ordering.
	 */
	private function reorder_items(): void {
		// Sort items by position.
		usort(
            $this->items,
            function ( ChecklistItem $a, ChecklistItem $b ) {
				return $a->get_position() - $b->get_position();
			}
        );

		// Ensure positions are consecutive starting from 0.
		foreach ( $this->items as $index => $item ) {
			$item->set_position( $index );
		}
	}
}
