<?php
/**
 * GravityBoard Checklist ChecklistCollection
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Checklists;

use GravityKit\GravityBoard\Feed;
use WP_Error;

defined( 'ABSPATH' ) || die();

/**
 * Class ChecklistCollection
 *
 * Manages a collection of checklists with utility methods.
 *
 * @since 1.1
 */
class ChecklistCollection {

	/**
	 * The checklists.
	 *
	 * @var Checklist[]
	 */
	private array $checklists;

	/**
	 * Constructor.
	 *
	 * @param Checklist[] $checklists The checklists.
	 */
	public function __construct( array $checklists = [] ) {
		$this->set_checklists( $checklists );
	}

	/**
	 * Create from array data.
	 *
	 * @param array $data The array data.
	 *
	 * @return ChecklistCollection|WP_Error The checklist collection or WP_Error if invalid data.
	 */
	public static function from_array( array $data ) {
		$checklists = [];

		foreach ( $data as $checklist_data ) {
			if ( ! is_array( $checklist_data ) ) {
				return new WP_Error( 'gk_gravityboard_invalid_checklists_data', esc_html__( 'Invalid checklists data.', 'gk-gravityboard' ), [ 'status' => 400 ] );
			}

			if ( empty( $checklist_data['items'] ) && empty( $checklist_data['title'] ) ) {
				continue;
			}

			$checklist = Checklist::from_array( $checklist_data );

			if ( is_wp_error( $checklist ) ) {
				Feed::get_instance()->log_error( 'Failed to create checklist: ' . $checklist->get_error_message() );
				continue;
			}

			if ( ! $checklist->is_empty() ) {
				$checklists[] = $checklist;
			}
		}

		return new self( $checklists );
	}

	/**
	 * Convert to array for API responses (includes calculated progress for each checklist).
	 *
	 * @return array
	 */
	public function to_array(): array {
		return array_map(
            function ( Checklist $checklist ) {
                return $checklist->to_array();
            },
            $this->checklists
        );
	}

	/**
	 * Convert to array for storage (excludes calculated fields like progress).
	 *
	 * @since 1.1
	 * @return array
	 */
	public function to_storage_array(): array {
		return array_map(
            function ( Checklist $checklist ) {
                return $checklist->to_storage_array();
            },
            $this->checklists
        );
	}

	/**
	 * Get all checklists.
	 *
	 * @return Checklist[]
	 */
	public function get_checklists(): array {
		return $this->checklists;
	}

	/**
	 * Set the checklists.
	 *
	 * @param Checklist[] $checklists The checklists.
	 */
	public function set_checklists( array $checklists ): void {
		foreach ( $checklists as $index => $checklist ) {
			if ( ! $checklist instanceof Checklist ) {
				Feed::get_instance()->log_error( 'All checklists must be Checklist instances.' );
				unset( $checklists[ $index ] );
			}
		}

		$this->checklists = $checklists;
	}

	/**
	 * Add a checklist.
	 *
	 * @param Checklist $checklist The checklist to add.
	 */
	public function add_checklist( Checklist $checklist ): void {
		$this->checklists[] = $checklist;
	}

	/**
	 * Remove a checklist by ID.
	 *
	 * @param string $checklist_id The checklist ID.
	 * @return bool True if removed, false if not found.
	 */
	public function remove_checklist( string $checklist_id ): bool {
		foreach ( $this->checklists as $index => $checklist ) {
			if ( $checklist->get_id() === $checklist_id ) {
				unset( $this->checklists[ $index ] );
				$this->checklists = array_values( $this->checklists ); // Re-index.
				return true;
			}
		}
		return false;
	}

	/**
	 * Find a checklist by ID.
	 *
	 * @param string $checklist_id The checklist ID.
	 * @return Checklist|null The checklist if found, null otherwise.
	 */
	public function find_checklist( string $checklist_id ): ?Checklist {
		foreach ( $this->checklists as $checklist ) {
			if ( $checklist->get_id() === $checklist_id ) {
				return $checklist;
			}
		}
		return null;
	}

	/**
	 * Find the index of a checklist by ID.
	 *
	 * @param string $checklist_id The checklist ID.
	 * @return int|null The checklist index if found, null otherwise.
	 */
	public function find_checklist_index( string $checklist_id ): ?int {
		foreach ( $this->checklists as $index => $checklist ) {
			if ( $checklist->get_id() === $checklist_id ) {
				return $index;
			}
		}
		return null;
	}

	/**
	 * Find an item by ID across all checklists.
	 *
	 * @param string $item_id The item ID.
	 * @return array|null Array with 'checklist_index', 'checklist', 'item_index', 'item' keys, or null if not found.
	 */
	public function find_item( string $item_id ): ?array {
		foreach ( $this->checklists as $checklist_index => $checklist ) {
			$item = $checklist->find_item( $item_id );
			if ( null !== $item ) {
				$item_index = $checklist->find_item_index( $item_id );
				return [
					'checklist_index' => $checklist_index,
					'checklist'       => $checklist,
					'item_index'      => $item_index,
					'item'            => $item,
				];
			}
		}
		return null;
	}

	/**
	 * Update an item across all checklists.
	 *
	 * @param string $item_id The item ID.
	 * @param array  $updates The updates to apply.
	 * @return bool True if updated, false if not found.
	 */
	public function update_item( string $item_id, array $updates ): bool {
		foreach ( $this->checklists as $checklist ) {
			if ( $checklist->update_item( $item_id, $updates ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Remove an item from all checklists.
	 *
	 * @param string $item_id The item ID.
	 * @return bool True if removed, false if not found.
	 */
	public function remove_item( string $item_id ): bool {
		foreach ( $this->checklists as $index => $checklist ) {
			if ( $checklist->remove_item( $item_id ) ) {
				if ( $checklist->is_empty() ) {
					unset( $this->checklists[ $index ] );
					$this->checklists = array_values( $this->checklists ); // Re-index.
				}
				return true;
			}
		}
		return false;
	}

	/**
	 * Get overall progress for all checklists.
	 *
	 * @return array Array with 'completed', 'total', and 'percentage' keys.
	 */
	public function get_progress(): array {
		$total_items     = 0;
		$completed_items = 0;

		foreach ( $this->checklists as $checklist ) {
			$progress         = $checklist->get_progress();
			$total_items     += $progress['total'];
			$completed_items += $progress['completed'];
		}

		$percentage = $total_items > 0 ? round( ( $completed_items / $total_items ) * 100 ) : 0;

		return [
			'completed'  => $completed_items,
			'total'      => $total_items,
			'percentage' => $percentage,
		];
	}

	/**
	 * Check if the collection is empty.
	 *
	 * @return bool True if empty, false otherwise.
	 */
	public function is_empty(): bool {
		return empty( $this->checklists );
	}

	/**
	 * Get the count of checklists.
	 *
	 * @return int The number of checklists.
	 */
	public function count_checklists(): int {
		return count( $this->checklists );
	}

	/**
	 * Get the total count of items across all checklists.
	 *
	 * @return int The total number of items.
	 */
	public function count_items(): int {
		return array_sum(
            array_map(
				function ( Checklist $checklist ) {
					return $checklist->count_items();
				},
				$this->checklists
            )
        );
	}

	/**
	 * Get the total count of completed items across all checklists.
	 *
	 * @return int The total number of completed items.
	 */
	public function count_completed_items(): int {
		return array_sum(
            array_map(
				function ( Checklist $checklist ) {
					return $checklist->count_completed_items();
				},
				$this->checklists
            )
        );
	}

	/**
	 * Get all item labels for search purposes.
	 *
	 * @return array Array of item labels.
	 */
	public function get_all_item_labels(): array {
		$labels = [];
		foreach ( $this->checklists as $checklist ) {
			foreach ( $checklist->get_items() as $item ) {
				$labels[] = $item->get_label();
			}
		}
		return $labels;
	}

	/**
	 * Flatten all checklists into a single checklist.
	 *
	 * This method combines all items from all checklists in the collection
	 * into a single checklist. Items are added sequentially with proper
	 * position numbering. New item instances are created to avoid reference issues.
	 *
	 * @since TODO
	 *
	 * @param string|null $checklist_id Optional ID for the flattened checklist. If null, generates a new ID.
	 * @param string      $title        Optional title for the flattened checklist.
	 *
	 * @return Checklist|null The flattened checklist, or null if the collection is empty.
	 */
	public function flatten( ?string $checklist_id = null, string $title = '' ): ?Checklist {
		if ( $this->is_empty() ) {
			return null;
		}

		// Create the flattened checklist with provided or generated ID.
		$flattened = new Checklist( $checklist_id, $title );

		// Collect all items from all checklists.
		$all_items = [];
		foreach ( $this->checklists as $checklist ) {
			foreach ( $checklist->get_items() as $item ) {
				$all_items[] = new ChecklistItem(
					null, // Generate new ID.
					$item->get_label(),
					$item->is_complete(),
					0 // Position will be set by add_items.
				);
			}
		}

		// Batch add all items for better performance.
		if ( ! empty( $all_items ) ) {
			$add_result = $flattened->add_items( $all_items );
			if ( is_wp_error( $add_result ) ) {
				Feed::get_instance()->log_error(
					'Failed to add items during flatten: ' . $add_result->get_error_message()
				);
				// Return null if we couldn't add any items.
				return null;
			}
		}

		return $flattened;
	}

}
