<?php
/**
 * GravityBoard Checklist Item
 *
 * @package GravityBoard
 */
namespace GravityKit\GravityBoard\Checklists;

use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class ChecklistItem
 *
 * Represents a single checklist item with validation and utility methods.
 *
 * @since 1.1
 */
class ChecklistItem {

	/**
	 * The item ID.
	 *
	 * @var string
	 */
	private string $id;

	/**
	 * The item label.
	 *
	 * @var string
	 */
	private string $label;

	/**
	 * Whether the item is complete.
	 *
	 * @var bool
	 */
	private bool $is_complete;

	/**
	 * The item position.
	 *
	 * @var int
	 */
	private int $position;

	/**
	 * The timestamp when the item was last updated.
	 *
	 * @var int|null
	 */
	private ?int $updated_at;

	/**
	 * Map of editable properties and their setter methods.
	 *
	 * @var array
	 */
	private array $editable_properties = [
		'label',
		'is_complete',
		'position',
		'updated_at',
	];

	/**
	 * Constructor.
	 *
	 * @param string|null $id           The item ID, or null to auto-generate.
	 * @param string      $label        The item label.
	 * @param bool        $is_complete  Whether the item is complete.
	 * @param int         $position     The item position.
	 * @param int|null    $updated_at   The timestamp when last updated (null if never updated).
	 */
	public function __construct( $id, string $label, bool $is_complete = false, int $position = 0, ?int $updated_at = null ) {
		// Generate ID if null is passed
		if ( null === $id ) {
			$this->id = self::generate_item_id();
		} else {
			// Set ID with fallback to generated ID if validation fails.
			$id_result = $this->set_id( $id );
			if ( is_wp_error( $id_result ) ) {
				$this->id = self::generate_item_id();
			}
		}

		// Set label with fallback to 'Checklist Item' if validation fails.
		$label_result = $this->set_label( $label );
		if ( is_wp_error( $label_result ) ) {
			$this->label = esc_html__( 'Checklist Item', 'gk-gravityboard' );
		}

		$this->is_complete = $is_complete;
		$this->position    = max( 0, $position );
		$this->updated_at  = $updated_at;
	}

	/**
	 * Create from array data.
	 *
	 * @param array $data The array data.
	 * @return ChecklistItem|WP_Error
	 */
	public static function from_array( array $data ) {
		if ( ! isset( $data['label'] ) || '' === trim( $data['label'] ) ) {
			return new WP_Error( 'empty_checklist_item_label', esc_html__( 'Checklist item label is required.', 'gk-gravityboard' ) );
		}

		$item_id = isset( $data['id'] ) ? sanitize_text_field( $data['id'] ) : null;

		$item = new self(
			$item_id,
			sanitize_text_field( $data['label'] ),
			(bool) ( $data['is_complete'] ?? false ),
			(int) ( $data['position'] ?? 0 ),
			isset( $data['updated_at'] ) ? (int) $data['updated_at'] : null
		);

		// Validate the final item.
		$validation_result = $item->validate();
		if ( is_wp_error( $validation_result ) ) {
			return $validation_result;
		}

		return $item;
	}

	/**
	 * Generates a new item ID based on current microtime.
	 *
	 * @since 1.1
	 *
	 * @return string The generated item ID.
	 */
	private static function generate_item_id(): string {
		// Add a small delay between items to ensure unique IDs.
		usleep( 1000 );

		// Use microsecond precision to virtually eliminate collision risk.
		$timestamp = microtime( true ) * 1000;

		// Do not use round() here! It will cause scientific notation in JSON encoding.
		return sprintf( '%.0f', $timestamp );
	}

	/**
	 * Convert to array.
	 *
	 * @return array
	 */
	public function to_array(): array {
		$data = [
			'id'          => $this->id,
			'label'       => $this->label,
			'is_complete' => $this->is_complete,
			'position'    => $this->position,
			'updated_at'  => $this->updated_at,
		];

		return $data;
	}

	/**
	 * Get the item ID.
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Set the item ID.
	 *
	 * @param string $id The item ID.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public function set_id( string $id ) {
		if ( empty( trim( $id ) ) ) {
			return new WP_Error( 'empty_checklist_item_id', esc_html__( 'Checklist item ID cannot be empty', 'gk-gravityboard' ) );
		}
		$this->id = sanitize_text_field( $id );
		return true;
	}

	/**
	 * Get the item label.
	 *
	 * @return string
	 */
	public function get_label(): string {
		return $this->label;
	}

	/**
	 * Set the item label.
	 *
	 * @param string $label The item label.
	 *
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public function set_label( string $label ) {
		$sanitized_label = sanitize_text_field( $label );
		if ( '' === trim( $sanitized_label ) ) {
			return new WP_Error( 'empty_checklist_item_label', esc_html__( 'Checklist item label cannot be empty', 'gk-gravityboard' ) );
		}
		$this->label = $sanitized_label;
		return true;
	}

	/**
	 * Get whether the item is complete.
	 *
	 * @return bool
	 */
	public function is_complete(): bool {
		return $this->is_complete;
	}

	/**
	 * Set whether the item is complete.
	 *
	 * @param bool $is_complete Whether the item is complete.
	 */
	public function set_is_complete( bool $is_complete ): void {
		$was_complete      = $this->is_complete;
		$this->is_complete = $is_complete;

		// Update timestamp whenever completion status changes.
		if ( $is_complete !== $was_complete ) {
			$this->updated_at = time();
		}
	}

	/**
	 * Mark the item as complete.
	 */
	public function mark_complete(): void {
		$this->set_is_complete( true );
	}

	/**
	 * Mark the item as incomplete.
	 */
	public function mark_incomplete(): void {
		$this->set_is_complete( false );
	}

	/**
	 * Toggle the completion status.
	 */
	public function toggle_complete(): void {
		$this->set_is_complete( ! $this->is_complete );
	}

	/**
	 * Get the item position.
	 *
	 * @return int
	 */
	public function get_position(): int {
		return $this->position;
	}

	/**
	 * Set the item position.
	 *
	 * @param int $position The item position.
	 */
	public function set_position( int $position ): void {
		$this->position = max( 0, $position );
	}

	/**
	 * Get the last updated timestamp.
	 *
	 * @return int|null The timestamp when last updated, or null if never updated.
	 */
	public function get_updated_at(): ?int {
		return $this->updated_at;
	}

	/**
	 * Set the last updated timestamp.
	 *
	 * @param int|null $updated_at The timestamp when last updated (null if never updated).
	 */
	public function set_updated_at( ?int $updated_at ): void {
		$this->updated_at = $updated_at;
	}

	/**
	 * Update the item with new data.
	 *
	 * @param array $updates Array of updates to apply.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public function update( array $updates ) {
		foreach ( $updates as $key => $value ) {
			if ( ! in_array( $key, $this->editable_properties, true ) ) {
				return new WP_Error(
					'invalid_update_key',
					sprintf(
						/* translators: %s: Update key */
						esc_html__( 'Invalid update key: %s', 'gk-gravityboard' ),
						$key
					)
				);
			}

			$result = $this->{"set_{$key}"}( $value );

			if ( is_wp_error( $result ) ) {
				return $result;
			}
		}

		return true;
	}


	/**
	 * Validate the checklist item.
	 *
	 * @return true|WP_Error True if valid, WP_Error if not.
	 */
	public function validate() {
		if ( empty( trim( $this->id ) ) ) {
			return new WP_Error( 'empty_checklist_item_id', esc_html__( 'Checklist item ID cannot be empty', 'gk-gravityboard' ) );
		}

		if ( '' === trim( $this->label ) ) {
			return new WP_Error( 'empty_checklist_item_label', esc_html__( 'Checklist item label cannot be empty', 'gk-gravityboard' ) );
		}

		return true;
	}
}
