<?php
/**
 * GravityBoard Checklists
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Checklists;

use GF_Entry_List_Table;
use GravityKit\GravityBoard\Helpers;
use GravityKit\GravityBoard\Ajax;
use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Notifications;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

defined( 'ABSPATH' ) || die();

/**
 * Class Checklists
 *
 * Handles checklist operations for GravityBoard.
 *
 * @since 1.1
 */
class Checklists {

	/**
	 * Instance of the class.
	 *
	 * @var Checklists
	 */
	public static $instance;

	/**
	 * The meta key used to store checklists on an entry.
	 *
	 * @since 1.1
	 *
	 * @var string
	 */
	const CHECKLIST_META_KEY = '_gravityboard_checklists';

	/**
	 * Feed setting key for the checklist template.
	 *
	 * @since TODO
	 * @var string
	 */
	const CHECKLIST_TEMPLATE_SETTING = 'checklist_template';


	/**
	 * Get the instance of the class.
	 *
	 * @since 1.1
	 *
	 * @return Checklists The instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.1
	 */
	public function __construct() {
		add_filter( 'gform_entry_meta', [ $this, 'filter_entry_meta' ], 10, 2 );
		add_filter( 'gk/gravityboard/data/lanes', [ $this, 'add_checklists_to_lanes' ], 10, 3 );
		add_action( 'gk/gravityboard/ajax/register-routes', [ $this, 'register_routes' ] );

		// Hook to format checklist display in Gravity Forms entries table.
		add_filter( 'gform_entries_field_value', [ $this, 'filter_entries_field_value' ], 10, 4 );

		// Hook to format checklist data for Gravity Forms export.
		add_filter( 'gform_export_field_value', [ $this, 'filter_export_field_value' ], 10, 4 );

		// Apply checklist templates to API-created entries. 5 priority to act before default hooks.
		add_action( 'gform_post_add_entry', [ $this, 'apply_template_to_entry' ], 5, 2 );

		// Apply checklist templates to entries created via the Gravity Forms UI. 5 priority to act before default hooks.
		add_action( 'gform_entry_created', [ $this, 'apply_template_to_entry' ], 5, 2 );

		// Hook into card creation for checklist post-processing.
		add_action( 'gk/gravityboard/card/added', [ $this, 'on_card_added' ], 10, 3 );

		// Hook into card data filter to add checklist data.
		add_filter( 'gk/gravityboard/card/data', [ $this, 'filter_card_data' ], 10, 3 );
	}

	/**
	 * Filter the entry meta.
	 *
	 * @since 1.1
	 *
	 * @param array $entry_meta The entry meta.
	 *
	 * @return array The entry meta.
	 */
	public function filter_entry_meta( $entry_meta ) {
		$entry_meta[ self::CHECKLIST_META_KEY ] = [
			'label'             => esc_html__( 'GravityBoard Checklists', 'gk-gravityboard' ),
			'is_numeric'        => false,
			'is_default_column' => false,
			'filter'            => [
				'operators' => [ 'contains' ],
			],
		];

		return $entry_meta;
	}

	/**
	 * Filter the field value to properly format checklists for display in entry lists.
	 *
	 * This method is called when displaying entry lists to format the checklist
	 * field value. It adds a clickable link to the entry detail page with anchor
	 * to the checklists section, and includes item labels for search functionality.
	 *
	 * @since 1.1
	 *
	 * @param string $value The current value of the field.
	 * @param int    $form_id The ID of the form.
	 * @param string $field_id The ID of the field.
	 * @param array  $entry The entry object.
	 *
	 * @return string The filtered value of the field.
	 */
	public function filter_entries_field_value( $value, $form_id, $field_id, $entry ) {
		if ( self::CHECKLIST_META_KEY !== $field_id ) {
			return $value;
		}

		$checklists = self::get_checklists( $entry['id'] );

		$text = self::get_field_value_from_checklists( $checklists );

		if ( is_wp_error( $checklists ) ) {
			Feed::get_instance()->log_error( 'Failed to get checklists for entry ' . $entry['id'] . ': ' . $checklists->get_error_message() );
			return $text;
		}

		if ( ! empty( $checklists ) ) {
			// Extract all item labels for search purposes using the collection.
			$item_labels = $checklists->get_all_item_labels();

			// For search purposes, we include the item labels in a hidden span.
			$text .= '<span style="display:none;">' . esc_html( implode( ' ', $item_labels ) ) . '</span>';
		}

		$entry_list_table = new GF_Entry_List_Table();
		$url              = $entry_list_table->get_detail_url( $entry ) . '#' . self::CHECKLIST_META_KEY;

		return sprintf( '<a href="%s">%s</a>', $url, $text );
	}

	/**
	 * Get the field value from a checklists collection.
	 *
	 * @since 1.1
	 *
	 * @param ChecklistCollection|WP_Error $checklists The checklists collection or error.
	 *
	 * @return string The formatted field value.
	 */
	private static function get_field_value_from_checklists( $checklists ) {
		if ( is_wp_error( $checklists ) || empty( $checklists ) ) {
			return '<span class="screen-reader-text">' . esc_html__( 'No checklists', 'gk-gravityboard' ) . '</span>';
		}

		$progress = $checklists->get_progress();

		if ( $progress['total'] === 0 ) {
			return '<span class="screen-reader-text">' . esc_html__( 'No checklist items', 'gk-gravityboard' ) . '</span>';
		}

		// translators: Do not translate placeholders in square brackets. These are placeholders that will be replaced with the actual values.
		$template = esc_html__( '[completion_indicator] [completed][out_of][total] ([percentage]%)', 'gk-gravityboard' );

		$text = strtr(
			$template,
			[
				'[completion_indicator]' => $progress['completed'] > 0 ? '☑' : '☐',
				'[completed]'            => $progress['completed'],
				'[out_of]'               => sprintf( '<span class="screen-reader-text">%s</span>/', esc_html__( 'out of', 'gk-gravityboard' ) ),
				'[total]'                => $progress['total'],
				'[percentage]'           => $progress['percentage'],
			]
		);

		return $text;
	}

	/**
	 * Filter the export field value to properly format checklists for export.
	 *
	 * This method formats checklist data for export in CSV or other formats.
	 * It converts the array structure to a readable format that shows the
	 * checklist items and their completion status.
	 *
	 * @since 1.1
	 *
	 * @param string $value The current value of the field.
	 * @param int    $form_id The ID of the form.
	 * @param string $field_id The ID of the field.
	 * @param array  $entry The entry object.
	 *
	 * @return string The filtered value of the field for export.
	 */
	public function filter_export_field_value( $value, $form_id, $field_id, $entry ) {
		if ( self::CHECKLIST_META_KEY !== $field_id ) {
			return $value;
		}

		$checklists = self::get_checklists( $entry['id'] );

		if ( is_wp_error( $checklists ) || empty( $checklists ) ) {
			return '';
		}

		$progress = $checklists->get_progress();

		if ( empty( $progress['total'] ) ) {
			return '';
		}

		// Create a readable format for export.
		$export_lines = [];

		foreach ( $checklists->get_checklists() as $checklist ) {
			$checklist_title = $checklist->get_title();
			if ( ! empty( $checklist_title ) ) {
				// translators: %s is the checklist title.
				$export_lines[] = sprintf( esc_html__( 'Checklist: %s', 'gk-gravityboard' ), $checklist_title );
			}

			foreach ( $checklist->get_items() as $item ) {
				$status         = $item->is_complete() ? '☑' : '☐';
				$export_lines[] = sprintf( '%s %s', $status, $item->get_label() );
			}

			if ( count( $checklists->get_checklists() ) > 1 ) {
				$export_lines[] = ''; // Add empty line between checklists.
			}
		}

		// Join with line breaks.
		$export_text = implode( "\n", $export_lines );

		/**
		 * Filters the export text for checklists.
		 *
		 * @since 1.1
		 *
		 * @param string $export_text The export text, which is a string of checklist items and their statuses.
		 * @param ChecklistCollection $checklists The checklists collection.
		 *
		 * @return string The filtered export text.
		 */
		$export_text = apply_filters( 'gk/gravityboard/checklists/export-value', $export_text, $checklists );

		return $export_text;
	}

	/**
	 * Register REST API routes for checklists.
	 *
	 *  Checklists
	 *    • GET    /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/checklists
	 *    • PUT    /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/checklists
	 *    • POST   /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/checklists/(?P<checklist_id>[a-zA-Z0-9_-]+)/items
	 *    • PUT    /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/checklists/(?P<checklist_id>[a-zA-Z0-9_-]+)/items/(?P<item_id>[a-zA-Z0-9_-]+)
	 *    • DELETE /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/checklists/(?P<checklist_id>[a-zA-Z0-9_-]+)/items/(?P<item_id>[a-zA-Z0-9_-]+)
	 *
	 * @since 1.1
	 */
	public function register_routes() {
		// Get and update entire checklists.
		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\\d+)/cards/(?P<entry_id>\\d+)/checklists',
			[
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'rest_get_checklists' ],
					'permission_callback' => [ $this, 'can_view_checklists' ],
				],
				[
					'methods'             => 'PUT',
					'callback'            => [ $this, 'rest_update_checklists' ],
					'permission_callback' => [ $this, 'can_edit_checklists' ],
				],
			]
		);

		// Add checklist items.
		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\\d+)/cards/(?P<entry_id>\\d+)/checklists/(?P<checklist_id>[a-zA-Z0-9_-]+)/items',
			[
				'methods'             => 'POST',
				'callback'            => [ $this, 'rest_add_checklist_item' ],
				'permission_callback' => [ $this, 'can_edit_checklists' ],
			]
		);

		// Update and delete individual checklist items.
		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\\d+)/cards/(?P<entry_id>\\d+)/checklists/(?P<checklist_id>[a-zA-Z0-9_-]+)/items/(?P<item_id>[a-zA-Z0-9_.+E-]+)',
			[
				[
					'methods'             => 'PUT',
					'callback'            => [ $this, 'rest_update_checklist_item' ],
					'permission_callback' => [ $this, 'can_edit_checklists' ],
				],
				[
					'methods'             => 'DELETE',
					'callback'            => [ $this, 'rest_delete_checklist_item' ],
					'permission_callback' => [ $this, 'can_edit_checklists' ],
				],
			]
		);
	}

	/**
	 * Check if checklists are enabled for a feed.
	 *
	 * @param array $feed The feed data.
	 *
	 * @return bool True if checklists are enabled, false otherwise.
	 */
	public static function is_checklists_enabled( $feed ): bool {
		return (bool) rgars( $feed, 'meta/enable_card_checklists', false );
	}

	/**
	 * Permission check for viewing checklists.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool|WP_Error True if permission is granted, otherwise WP_Error.
	 */
	public function can_view_checklists( WP_REST_Request $request ) {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return new WP_Error( 'gk_gravityboard_feed_not_found', esc_html__( 'Feed not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}

		// Check if checklists are enabled for this feed.
		if ( ! self::is_checklists_enabled( $feed ) ) {
			return new WP_Error( 'gk_gravityboard_checklists_disabled', esc_html__( 'Checklists are not enabled for this board.', 'gk-gravityboard' ), [ 'status' => 403 ] );
		}

		// Check user permission using the correct action mapped in Helpers::get_role_setting_key_for_action().
		if ( ! Helpers::user_has_permission( 'view_checklists', $feed ) ) {
			return new WP_Error( 'gk_gravityboard_permission_denied', esc_html__( 'You do not have permission to view checklists.', 'gk-gravityboard' ), [ 'status' => 403 ] );
		}

		return true;
	}

	/**
	 * Permission check for editing checklists.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool|WP_Error True if permission is granted, otherwise WP_Error.
	 */
	public function can_edit_checklists( WP_REST_Request $request ) {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return new WP_Error( 'gk_gravityboard_feed_not_found', esc_html__( 'Feed not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}

		// Check if checklists are enabled for this feed.
		if ( ! self::is_checklists_enabled( $feed ) ) {
			return new WP_Error( 'gk_gravityboard_checklists_disabled', esc_html__( 'Checklists are not enabled for this board.', 'gk-gravityboard' ), [ 'status' => 403 ] );
		}

		// Check user permission using the correct action mapped in Helpers::get_role_setting_key_for_action().
		if ( ! Helpers::user_has_permission( 'edit_checklists', $feed ) ) {
			return new WP_Error( 'gk_gravityboard_permission_denied', esc_html__( 'You do not have permission to edit checklists.', 'gk-gravityboard' ), [ 'status' => 403 ] );
		}

		return true;
	}

	/**
	 * REST API handler for getting checklists from a card.
	 *
	 * @since 1.1
	 * @param WP_REST_Request $request The REST request.
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_get_checklists( WP_REST_Request $request ) {
		$entry    = Ajax::get_instance()->get_entry_from_request( $request );
		$entry_id = $entry['id'];

		$checklists = self::get_checklists( $entry_id );

		if ( is_wp_error( $checklists ) ) {
			return new WP_Error( 'gk_gravityboard_get_checklists_failed', $checklists->get_error_message(), [ 'status' => 500 ] );
		}

		return new WP_REST_Response(
            [
				'checklists' => $checklists->to_array(),
			]
        );
	}

	/**
	 * REST API handler for updating entire checklists for a card.
	 *
	 * @since 1.1
	 * @param WP_REST_Request $request The REST request.
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_update_checklists( WP_REST_Request $request ) {
		$entry    = Ajax::get_instance()->get_entry_from_request( $request );
		$entry_id = $entry['id'];
		$feed     = Ajax::get_instance()->get_feed_from_request( $request );

		$checklists_data = $request->get_param( 'checklists' );
		if ( ! is_array( $checklists_data ) ) {
			return new WP_Error( 'gk_gravityboard_invalid_checklists_data', esc_html__( 'Invalid checklists data.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$checklist_collection = ChecklistCollection::from_array( $checklists_data );

		$result = self::update_checklists( $entry_id, $feed, $checklist_collection );

		if ( is_wp_error( $result ) ) {
			return new WP_Error( 'gk_gravityboard_update_checklists_failed', $result->get_error_message(), [ 'status' => 500 ] );
		}

		return new WP_REST_Response(
            [
				'message'    => esc_html__( 'Checklists updated successfully.', 'gk-gravityboard' ),
				'checklists' => $checklist_collection->to_array(),
			]
        );
	}

	/**
	 * REST API handler for adding a new checklist item.
	 *
	 * @since 1.1
	 *
	 * @param WP_REST_Request $request The REST request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_add_checklist_item( WP_REST_Request $request ) {
		$entry    = Ajax::get_instance()->get_entry_from_request( $request );
		$entry_id = $entry['id'];
		$feed     = Ajax::get_instance()->get_feed_from_request( $request );

		$checklist_id = sanitize_text_field( $request->get_param( 'checklist_id' ) );
		$label        = sanitize_text_field( $request->get_param( 'label' ) );

		if ( empty( $checklist_id ) ) {
			return new WP_Error( 'gk_gravityboard_empty_checklist_id', esc_html__( 'Checklist ID is required.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		if ( empty( $label ) ) {
			return new WP_Error( 'gk_gravityboard_empty_label', esc_html__( 'Checklist item label cannot be empty.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$result = self::add_item( $entry_id, $feed, $checklist_id, $label );

		if ( is_wp_error( $result ) ) {
			return new WP_Error( 'gk_gravityboard_add_item_failed', $result->get_error_message(), [ 'status' => 500 ] );
		}

		return new WP_REST_Response(
            [
				'message'    => esc_html__( 'Checklist item added successfully.', 'gk-gravityboard' ),
				'item'       => $result['item']->to_array(),
				'checklists' => $result['checklists']->to_array(),
			]
        );
	}

	/**
	 * REST API handler for updating a checklist item.
	 *
	 * @since 1.1
	 * @param WP_REST_Request $request The REST request.
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_update_checklist_item( WP_REST_Request $request ) {
		$entry    = Ajax::get_instance()->get_entry_from_request( $request );
		$entry_id = $entry['id'];
		$feed     = Ajax::get_instance()->get_feed_from_request( $request );

		$checklist_id = sanitize_text_field( $request->get_param( 'checklist_id' ) );
		$item_id      = sanitize_text_field( $request->get_param( 'item_id' ) );
		$updates      = [];

		if ( empty( $checklist_id ) ) {
			return new WP_Error( 'gk_gravityboard_empty_checklist_id', esc_html__( 'Checklist ID is required.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		if ( empty( $item_id ) ) {
			return new WP_Error( 'gk_gravityboard_empty_item_id', esc_html__( 'Item ID is required.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		if ( $request->has_param( 'label' ) ) {
			$new_label = sanitize_text_field( $request->get_param( 'label' ) );
			if ( empty( $new_label ) ) {
				return new WP_Error( 'gk_gravityboard_empty_label', esc_html__( 'Checklist item label cannot be empty.', 'gk-gravityboard' ), [ 'status' => 400 ] );
			}
			$updates['label'] = $new_label;
		}

		if ( $request->has_param( 'is_complete' ) ) {
			$updates['is_complete'] = (bool) $request->get_param( 'is_complete' );
		}

		if ( $request->has_param( 'position' ) ) {
			$updates['position'] = (int) $request->get_param( 'position' );
		}

		$result = self::update_item( $entry_id, $feed, $checklist_id, $item_id, $updates );

		if ( is_wp_error( $result ) ) {
			return new WP_Error( 'gk_gravityboard_update_item_failed', $result->get_error_message(), [ 'status' => 500 ] );
		}

		return new WP_REST_Response(
            [
				'message'    => esc_html__( 'Checklist item updated successfully.', 'gk-gravityboard' ),
				'checklists' => $result,
			]
        );
	}

	/**
	 * REST API handler for deleting a checklist item.
	 *
	 * @since 1.1
	 * @param WP_REST_Request $request The REST request.
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_delete_checklist_item( WP_REST_Request $request ) {
		$entry    = Ajax::get_instance()->get_entry_from_request( $request );
		$entry_id = $entry['id'];
		$feed     = Ajax::get_instance()->get_feed_from_request( $request );

		$checklist_id = sanitize_text_field( $request->get_param( 'checklist_id' ) );
		$item_id      = sanitize_text_field( $request->get_param( 'item_id' ) );

		if ( empty( $checklist_id ) ) {
			return new WP_Error( 'gk_gravityboard_empty_checklist_id', esc_html__( 'Checklist ID is required.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		if ( empty( $item_id ) ) {
			return new WP_Error( 'gk_gravityboard_empty_item_id', esc_html__( 'Item ID is required.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$result = self::delete_item( $entry_id, $feed, $checklist_id, $item_id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error( 'gk_gravityboard_delete_item_failed', $result->get_error_message(), [ 'status' => 500 ] );
		}

		return new WP_REST_Response(
            [
				'message'    => esc_html__( 'Checklist item deleted successfully.', 'gk-gravityboard' ),
				'checklists' => $result,
			]
        );
	}

	/**
	 * Add checklists to the lanes.
	 *
	 * @param array $lanes The lanes data.
	 * @param array $entries The entries data.
	 * @param array $feed The feed data.
	 * @return array The lanes data.
	 */
	public function add_checklists_to_lanes( $lanes, $entries, $feed ) {
		// Only show checklists if enabled and user has permission.
		if ( ! self::is_checklists_enabled( $feed ) || ! Helpers::user_has_permission( 'view_checklists', $feed ) ) {
			return $lanes;
		}

		foreach ( $lanes as &$lane ) {
			foreach ( $lane['cards'] as &$card ) {
				$card = $this->add_checklists( $card );
			}
		}
		return $lanes;
	}

	/**
	 * Add checklists to the card.
	 *
	 * @param array $card The card data.
	 *
	 * @return array The card data.
	 */
	public function add_checklists( $card ) {
		$card['checklists'] = [];

		$entry_checklists = self::get_checklists( $card['id'] );

		if ( ! is_wp_error( $entry_checklists ) ) {
			$card['checklists'] = $entry_checklists->to_array();
		}

		return $card;
	}

	/**
	 * Get checklists for an entry.
	 *
	 * @since 1.1
	 *
	 * @param int $entry_id      The entry ID.
	 *
	 * @return ChecklistCollection|WP_Error ChecklistCollection object, or WP_Error on failure.
	 */
	public static function get_checklists( $entry_id ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID', 'gk-gravityboard' ) );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		$checklists_data = gform_get_meta( $entry_id, self::CHECKLIST_META_KEY );

		if ( ! is_array( $checklists_data ) ) {
			$checklists_data = [];
		}

		return ChecklistCollection::from_array( $checklists_data );
	}

	/**
	 * Update checklists for an entry.
	 *
	 * @since 1.1
	 *
	 * @param int                 $entry_id   The entry ID.
	 * @param array               $feed       The feed data.
	 * @param ChecklistCollection $checklists The checklist data to save.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function update_checklists( $entry_id, $feed, $checklists ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID', 'gk-gravityboard' ) );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		// Check if checklists are enabled for this feed.
		if ( ! self::is_checklists_enabled( $feed ) ) {
			return new WP_Error( 'checklists_disabled', esc_html__( 'Checklists are not enabled for this board', 'gk-gravityboard' ) );
		}

		// Check if current user has permission to update checklists.
		if ( ! Helpers::user_has_permission( 'edit_checklists', $feed, $entry['created_by'] ?? null ) ) {
			return new WP_Error( 'missing_permission', esc_html__( 'Current user does not have permission to update checklists', 'gk-gravityboard' ) );
		}

		// Update the meta with the array data.
		$result = gform_update_meta( $entry_id, self::CHECKLIST_META_KEY, $checklists->to_storage_array() );

		// Nothing changed, but there was no error. Don't trigger the actions, but return true.
		if ( 0 === $result ) {
			return true;
		}

		if ( $result ) {

			/**
			 * Runs when checklists have been updated.
			 *
			 * @since 1.1
			 *
			 * @param ChecklistCollection $checklists The checklist collection.
			 *
			 * @param array $entry The entry array.
			 * @param array $feed The feed array.
			 */
			do_action( 'gk/gravityboard/checklists/updated', $checklists, $entry, $feed );

			return true;
		}

		return new WP_Error( 'update_failed', esc_html__( 'Failed to update checklist metadata', 'gk-gravityboard' ) );
	}

	/**
	 * Add a new item to a checklist.
	 *
	 * @since 1.1
	 *
	 * @param int    $entry_id     The entry ID.
	 * @param array  $feed         The feed array.
	 * @param string $checklist_id The checklist ID.
	 * @param string $label        The item label.
	 * @return array|WP_Error Array with 'item' and 'checklists' keys on success, WP_Error on failure.
	 */
	public static function add_item( $entry_id, $feed, $checklist_id, $label ) {
		$checklists = self::get_checklists( $entry_id );

		if ( is_wp_error( $checklists ) ) {
			return $checklists;
		}

		// Find or create the checklist.
		$checklist = $checklists->find_checklist( $checklist_id );

		if ( null === $checklist ) {
			// Create new checklist if it doesn't exist.
			$checklist = new Checklist( $checklist_id, '' );
			$checklists->add_checklist( $checklist );
		}

		$new_item = new ChecklistItem(
			null,
			$label,
			false,
			$checklist->count_items()
		);

		$add_result = $checklist->add_item( $new_item );

		if ( is_wp_error( $add_result ) ) {
			return $add_result;
		}

		$update_result = self::update_checklists( $entry_id, $feed, $checklists );

		if ( is_wp_error( $update_result ) ) {
			return $update_result;
		}

		if ( $update_result ) {
			/**
			 * Runs when a checklist item has been added.
			 *
			 * @since 1.1
			 *
			 * @param array|WP_Error $entry The entry data, or a WP_Error if the entry is not found.
			 * @param ChecklistItem  $new_item The new checklist item.
			 * @param array          $feed The feed data.
			 */
			do_action( 'gk/gravityboard/checklists/item/added', Helpers::get_card( $entry_id ), $new_item, $feed );

			return [
				'item'       => $new_item,
				'checklists' => $checklists,
			];
		}

		return new WP_Error( 'item_add_failed', esc_html__( 'Failed to add checklist item', 'gk-gravityboard' ) );
	}

	/**
	 * Update a checklist item.
	 *
	 * @since 1.1
	 *
	 * @param int    $entry_id     The entry ID.
	 * @param array  $feed         The feed array.
	 * @param string $checklist_id The checklist ID.
	 * @param string $item_id      The item ID.
	 * @param array  $updates      Array of updates to apply.
	 * @return array|WP_Error Updated checklists array on success, WP_Error on failure.
	 */
	public static function update_item( $entry_id, $feed, $checklist_id, $item_id, $updates ) {
		$checklist_collection = self::get_checklists( $entry_id );

		if ( is_wp_error( $checklist_collection ) ) {
			return $checklist_collection;
		}

		// Find the specific checklist.
		$checklist = $checklist_collection->find_checklist( $checklist_id );
		if ( null === $checklist ) {
			return new WP_Error( 'checklist_not_found', esc_html__( 'Checklist not found.', 'gk-gravityboard' ) );
		}

		// Verify the item exists in the specified checklist.
		$item = $checklist->find_item( $item_id );
		if ( null === $item ) {
			return new WP_Error( 'item_not_found', esc_html__( 'Checklist item not found in the specified checklist.', 'gk-gravityboard' ) );
		}

		// Update the item using the collection.
		$success = $checklist_collection->update_item( $item_id, $updates );
		if ( ! $success ) {
			return new WP_Error( 'item_not_found', esc_html__( 'Checklist item not found.', 'gk-gravityboard' ) );
		}

		$update_result = self::update_checklists( $entry_id, $feed, $checklist_collection );

		if ( is_wp_error( $update_result ) ) {
			return $update_result;
		}

		if ( $update_result ) {
			/**
			 * Runs when a checklist item has been updated.
			 *
			 * @since 1.1
			 *
			 * @param array|WP_Error $entry The entry data, or a WP_Error if the entry is not found.
			 * @param ChecklistItem  $item The checklist item.
			 * @param ChecklistCollection $collection The updated checklists collection.
			 * @param array $feed The feed data.
			 */
			do_action( 'gk/gravityboard/checklists/item/updated', Helpers::get_card( $entry_id ), $item, $checklist_collection, $feed );

			return $checklist_collection->to_array();
		}

		return new WP_Error( 'item_update_failed', esc_html__( 'Failed to update checklist item', 'gk-gravityboard' ) );
	}

	/**
	 * Delete a checklist item.
	 *
	 * @since 1.1
	 *
	 * @param int    $entry_id     The entry ID.
	 * @param array  $feed         The feed array.
	 * @param string $checklist_id The checklist ID.
	 * @param string $item_id      The item ID.
	 * @return array|WP_Error Updated checklists array on success, WP_Error on failure.
	 */
	public static function delete_item( $entry_id, $feed, $checklist_id, $item_id ) {
		$collection = self::get_checklists( $entry_id );

		if ( is_wp_error( $collection ) ) {
			return $collection;
		}

		// Find the specific checklist.
		$checklist = $collection->find_checklist( $checklist_id );
		if ( null === $checklist ) {
			return new WP_Error( 'checklist_not_found', esc_html__( 'Checklist not found.', 'gk-gravityboard' ) );
		}

		// Verify the item exists in the specified checklist.
		$item = $checklist->find_item( $item_id );
		if ( null === $item ) {
			return new WP_Error( 'item_not_found', esc_html__( 'Checklist item not found in the specified checklist.', 'gk-gravityboard' ) );
		}

		// Remove the item using the collection.
		$success = $collection->remove_item( $item_id );
		if ( ! $success ) {
			return new WP_Error( 'item_not_found', esc_html__( 'Checklist item not found.', 'gk-gravityboard' ) );
		}

		$update_result = self::update_checklists( $entry_id, $feed, $collection );

		if ( is_wp_error( $update_result ) ) {
			return $update_result;
		}

		if ( $update_result ) {
			/**
			 * Runs when a checklist item has been deleted.
			 *
			 * @since 1.1
			 *
			 * @param array|WP_Error $entry The entry data, or a WP_Error if the entry is not found.
			 * @param ChecklistItem  $item The deleted checklist item.
			 * @param ChecklistCollection $checklists The updated checklists data.
			 * @param array $feed The feed data.
			 */
			do_action( 'gk/gravityboard/checklists/item/deleted', Helpers::get_card( $entry_id ), $item, $collection, $feed );

			return $collection->to_array();
		}

		return new WP_Error( 'item_delete_failed', esc_html__( 'Failed to delete checklist item', 'gk-gravityboard' ) );
	}

	/**
	 * Hook into card creation for checklist post-processing.
	 *
	 * This method is called after a new card (entry) has been created to initialize
	 * checklists if they are enabled for the board.
	 *
	 * @since 1.1
	 *
	 * @param array           $created_entry The created entry data.
	 * @param array           $feed          The feed data.
	 * @param WP_REST_Request $request       The REST request object.
	 */
	public function on_card_added( $created_entry, $feed, $request ) {
		// Only process if checklists are enabled for this feed.
		if ( ! self::is_checklists_enabled( $feed ) ) {
			return;
		}

		// Process complete checklist structure.
		$checklists_data = $request->get_param( 'checklists' );

		if ( empty( $checklists_data ) ) {
			return;
		}

		// Check if the user has permission to edit checklists.
		if ( ! Helpers::user_has_permission( 'edit_checklists', $feed ) ) {
			Feed::get_instance()->log_error( 'User does not have permission to edit checklists during card creation' );
			return;
		}

		if ( ! is_array( $checklists_data ) ) {
			Feed::get_instance()->log_error( 'Checklists data is not an array' );
			return;
		}

		$entry_id = $created_entry['id'];

		// Get any existing checklists (e.g., from templates that might have been applied already).
		$existing_checklists = self::get_checklists( $entry_id );
		
		// Create checklist collection from the request data.
		$new_checklist_collection = ChecklistCollection::from_array( $checklists_data );
		if ( is_wp_error( $new_checklist_collection ) ) {
			Feed::get_instance()->log_error( 'Failed to create checklists from data: ' . $new_checklist_collection->get_error_message() );
			return;
		}

		// Merge with existing checklists if any exist.
		$final_collection = null;
		if ( ! is_wp_error( $existing_checklists ) && ! $existing_checklists->is_empty() ) {
			// Combine both collections and flatten into a single checklist.
			$combined = new ChecklistCollection();
			
			// Add all existing checklists.
			foreach ( $existing_checklists->get_checklists() as $checklist ) {
				$combined->add_checklist( $checklist );
			}
			
			// Add all new checklists.
			foreach ( $new_checklist_collection->get_checklists() as $checklist ) {
				$combined->add_checklist( $checklist );
			}
			
			// Flatten everything into a single checklist.
			$flattened = $combined->flatten();
			if ( ! $flattened ) {
				// If flatten returns null, use the new collection as-is.
				$final_collection = $new_checklist_collection;
			} else {
				$final_collection = new ChecklistCollection( [ $flattened ] );
			}
		} else {
			// No existing checklists, just use the new ones.
			$final_collection = $new_checklist_collection;
		}

		// Use the centralized update method for consistency and proper permission checks.
		$result = self::update_checklists( $entry_id, $feed, $final_collection );

		if ( is_wp_error( $result ) ) {
			Feed::get_instance()->log_error( 'Failed to save checklists for new card ' . $entry_id . ': ' . $result->get_error_message() );
		}
	}

	/**
	 * Filter the card data.
	 *
	 * @since 1.1
	 *
	 * @param array $card The card data.
	 * @param array $entry The entry data.
	 * @param array $feed The feed data.
	 *
	 * @return array The filtered card data.
	 */
	public function filter_card_data( $card, $entry, $feed ) {
		// Only add checklist data if checklists are enabled for this feed.
		if ( ! self::is_checklists_enabled( $feed ) ) {
			return $card;
		}

		// Only add checklist data if the user has permission to view checklists.
		if ( ! Helpers::user_has_permission( 'view_checklists', $feed ) ) {
			return $card;
		}

		$checklists = self::get_checklists( $entry['id'] );

		if ( is_wp_error( $checklists ) ) {
			Feed::get_instance()->log_error( 'Failed to get checklists for card ' . $entry['id'] . ': ' . $checklists->get_error_message() );
			return $card;
		}

		$card['checklists'] = $checklists->to_array();

		return $card;
	}

	/**
	 * Get the checklist template from a feed.
	 *
	 * @since TODO
	 *
	 * @param array $feed The feed array.
	 *
	 * @return string The checklist template text, or empty string if not set.
	 */
	public static function get_feed_template( $feed ) {
		return rgars( $feed, 'meta/' . self::CHECKLIST_TEMPLATE_SETTING, '' );
	}

	/**
	 * Parse checklist template from text input.
	 *
	 * This method parses the checklist template text and creates a structured
	 * checklist collection. The template format is simple:
	 * - One item per line
	 * - Empty lines are ignored
	 * - No special formatting required
	 *
	 * @since TODO
	 *
	 * @param string $template_text The template text to parse.
	 *
	 * @return ChecklistCollection The parsed checklist collection.
	 */
	public static function parse_template( $template_text ) {
		$collection = new ChecklistCollection();

		if ( empty( $template_text ) ) {
			return $collection;
		}

		// Create a single checklist with an auto-generated unique ID.
		$checklist = new Checklist( null, '' );
		$collection->add_checklist( $checklist );

		$lines = explode( "\n", $template_text );
		$item_position = 0;

		foreach ( $lines as $line ) {
			$line = trim( $line );

			// Skip empty lines.
			if ( empty( $line ) ) {
				continue;
			}

			// Sanitize the line - checklists only support plain text, no HTML or Markdown.
			$sanitized_line = sanitize_text_field( $line );

			// Each non-empty line becomes a checklist item.
			$item = new ChecklistItem(
				null, // ID will be generated automatically.
				$sanitized_line, // Use the sanitized line as the item text.
				false, // Not completed by default.
				$item_position // Position.
			);

			$add_result = $checklist->add_item( $item );
			if ( is_wp_error( $add_result ) ) {
				Feed::get_instance()->log_error(
					'Failed to add template item "' . $sanitized_line . '": ' . $add_result->get_error_message()
				);
				continue;
			}
			$item_position++;
		}

		return $collection;
	}

	/**
	 * Track which entries have already had templates applied to prevent duplicates.
	 *
	 * @since TODO
	 * @var array
	 */
	private static $processed_entries = [];

	/**
	 * Apply checklist template to a new entry.
	 *
	 * This method applies the checklist template configured in the feed settings
	 * to a newly created entry. It's called via the 'gform_post_add_entry' hook
	 * which fires for ALL entry creations (form submissions, API, GravityBoard cards, etc.).
	 * 
	 * Templates will be merged with any existing checklist items rather than replacing them.
	 * This allows programmatically created checklists to be enhanced with template items.
	 *
	 * @since TODO
	 *
	 * @param array $entry The created entry data.
	 * @param array $form  The form data.
	 *
	 * @return void
	 */
	public function apply_template_to_entry( $entry, $form ) {

		// Ensure entry and form are valid.
		if ( ! is_array( $entry ) || ! isset( $entry['id'] ) || ! is_array( $form ) || ! isset( $form['id'] ) ) {
			return;
		}

		// Check if we've already processed this entry to prevent duplicate template application.
		if ( isset( self::$processed_entries[ $entry['id'] ] ) ) {
			return;
		}

		// Mark this entry as processed in memory.
		self::$processed_entries[ $entry['id'] ] = true;

		// Get existing checklists if any.
		$existing_checklists = self::get_checklists( $entry['id'] );

		if ( $existing_checklists instanceof WP_Error ) {
			return;
		}

		$active_feeds = Feed::get_instance()->get_active_feeds( $form['id'] );

		if ( empty( $active_feeds ) ) {
			return;
		}

		// Track which feeds have been applied.
		$applied_feed_ids = [];
		$items_added = 0;

		// Combine existing checklists with all template checklists.
		$combined = new ChecklistCollection();
		
		// Add existing checklists first if any.
		if ( ! $existing_checklists->is_empty() ) {
			foreach ( $existing_checklists->get_checklists() as $checklist ) {
				$combined->add_checklist( $checklist );
			}
		}

		// Process each active feed that has checklists enabled and a template configured.
		foreach ( $active_feeds as $feed ) {

			// Only process if checklists are enabled for this feed.
			if ( ! self::is_checklists_enabled( $feed ) ) {
				continue;
			}

			// Check if the entry passes the feed's conditional logic.
			if ( ! Feed::get_instance()->is_feed_condition_met( $feed, $form, $entry ) ) {
				continue;
			}

			// Get the template from feed settings.
			$template_text = self::get_feed_template( $feed );

			if ( empty( $template_text ) ) {
				continue; // No template configured for this feed.
			}

			// Parse the template (sanitization happens inside parse_template).
			$template_collection = self::parse_template( $template_text );

			if ( ! $template_collection instanceof ChecklistCollection || $template_collection->is_empty() ) {
				continue; // Template couldn't be parsed or is empty.
			}

			// Track that we've applied this feed's template.
			$applied_feed_ids[] = $feed['id'];
			
			// Count items being added.
			$items_added += $template_collection->count_items();

			// Add template checklists to the combined collection.
			foreach ( $template_collection->get_checklists() as $checklist ) {
				$combined->add_checklist( $checklist );
			}
		}

		// Save the merged checklist if we have added any template items.
		if ( $items_added > 0 && ! $combined->is_empty() ) {
			// Flatten everything into a single checklist.
			$flattened = $combined->flatten();
			if ( ! $flattened ) {
				return; // Nothing to apply.
			}
			
			$merged_collection = new ChecklistCollection( [ $flattened ] );
			$storage_array = $merged_collection->to_storage_array();
			$result = gform_update_meta( $entry['id'], self::CHECKLIST_META_KEY, $storage_array );

			// Check for actual failure (false), not 0 which means no change.
			if ( false === $result ) {
				Feed::get_instance()->log_error( sprintf(
					'Failed to apply checklist templates to entry %s from feeds: %s (added %d items)',
					$entry['id'],
					implode( ', ', $applied_feed_ids ),
					$items_added
				) );
			} else {
				/**
				 * Runs when checklist templates have been applied to an entry.
				 *
				 * @since TODO
				 *
				 * @param ChecklistCollection $merged_collection The merged template collection that was applied.
				 * @param array               $entry             The created entry data.
				 * @param array               $applied_feed_ids  Array of feed IDs that were applied.
				 * @param array               $form              The form data.
				 * @param int                 $items_added       Number of items added from templates.
				 */
				do_action( 'gk/gravityboard/checklists/templates-applied', $merged_collection, $entry, $applied_feed_ids, $form, $items_added );
			}
		}
	}

}
