<?php
/**
 * GravityBoard Checklist Entry Meta Box
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Checklists;

use GravityKit\GravityBoard\Feed;

defined( 'ABSPATH' ) || die();

/**
 * Class Entry_Meta
 *
 * Handles adding a meta box to the Gravity Forms entry detail screen
 * to display GravityBoard checklists.
 *
 * @since 1.1
 */
class Entry_Meta {

	/**
	 * Instance of the class.
	 *
	 * @var Entry_Meta
	 */
	public static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.1
	 * @return Entry_Meta The instance of the class.
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'gform_entry_detail_meta_boxes', [ $this, 'add_checklists_meta_box' ], 10, 3 );
	}

	/**
	 * Adds the checklists meta box to the entry detail screen.
	 *
	 * @since 1.1
	 *
	 * @param array $meta_boxes The existing meta boxes.
	 * @param array $entry      The current entry object.
	 * @param array $form       The current form object.
	 * @return array The modified meta boxes array.
	 */
	public function add_checklists_meta_box( $meta_boxes, $entry, $form ) {
		// Check all feeds, not just active ones. Checklists may exist on inactive feeds.
		$all_feeds = Feed::get_instance()->get_feeds( $form['id'] );

		// Only add meta box if we have at least one board.
		if ( empty( $all_feeds ) ) {
			return $meta_boxes;
		}

		// Check if checklists are enabled for at least one feed.
		$active_feeds_with_permission = self::get_active_feeds_with_permission( $all_feeds );

		if ( empty( $active_feeds_with_permission ) ) {
			return $meta_boxes;
		}

		$meta_boxes['gk_gravityboard_checklists'] = [
			'title'    => esc_html__( 'GravityBoard Checklists', 'gk-gravityboard' ),
			'callback' => [ $this, 'render_checklists_meta_box' ],
			'context'  => 'normal',
		];

		return $meta_boxes;
	}

	/**
	 * Check if there is an active feed with checklists enabled.
	 *
	 * @since 1.1
	 *
	 * @param array $feeds The list of feeds.
	 *
	 * @return array The list of active feeds with checklists enabled.
	 */
	private static function get_active_feeds_with_permission( $feeds ) {
		$active_feeds_with_permission = [];
		foreach ( $feeds as $feed ) {
			if ( ! Checklists::is_checklists_enabled( $feed ) ) {
				continue;
			}

			if ( ! Feed::get_instance()->user_has_permission( 'view_checklists', $feed ) ) {
				continue;
			}

			$active_feeds_with_permission[] = $feed;
		}

		return $active_feeds_with_permission;
	}

	/**
	 * Renders the content of the checklists meta box.
	 *
	 * @since 1.1
	 *
	 * @param array $args An array of arguments. Expected keys: 'form', 'entry', 'feeds'.
	 * @return void
	 */
	public function render_checklists_meta_box( $args ) {
		$entry = $args['entry'];
		$form  = $args['form'];

		// Get checklists for this entry.
		$checklists = Checklists::get_checklists( $entry['id'] );
		if ( is_wp_error( $checklists ) || $checklists->is_empty() ) {
			echo '<p>' . esc_html__( 'No checklists found for this entry.', 'gk-gravityboard' ) . '</p>';
			return;
		}

		// Get all feeds for this form.
		$all_feeds = Feed::get_instance()->get_feeds( $form['id'] );

		// Get the active feeds with permission.
		$active_feeds_with_permission = self::get_active_feeds_with_permission( $all_feeds );

		// We already checked this when registering the meta box, but double-check just in case.
		if ( empty( $active_feeds_with_permission ) ) {
			return;
		}

		// For viewing, any active feed with permission is fine.
		$active_feed = $active_feeds_with_permission[0];
		$can_edit    = false;
		foreach ( $active_feeds_with_permission as $feed ) {
			// For editing, we need to find the first active feed with permission.
			if ( \GravityKit\GravityBoard\Helpers::user_has_permission( 'edit_checklists', $feed ) ) {
				$active_feed = $feed;
				$can_edit    = true;
				break;
			}
		}

		echo '<div class="gk-gravityboard-checklists-metabox" id="gk-checklists-metabox-' . esc_attr( $entry['id'] ) . '">';
		echo '<div class="gk-gravityboard-board-feed-checklists" data-feed-id="' . esc_attr( $active_feed['id'] ) . '" data-entry-id="' . esc_attr( $entry['id'] ) . '">';

		// Status message area - screen reader only.
		echo '<div id="gk-checklist-status-' . esc_attr( $entry['id'] ) . '" class="gk-checklist-status-sr screen-reader-text" role="status" aria-live="polite" aria-atomic="true"></div>';

		foreach ( $checklists->get_checklists() as $checklist ) {
			self::render_single_checklist( $checklist, $entry['id'], $active_feed['id'], $can_edit );
		}

		echo '</div>';
		echo '</div>';
	}

	/**
	 * Render a single checklist.
	 *
	 * @param Checklist $checklist The checklist to render.
	 * @param int       $entry_id The entry ID.
	 * @param string    $feed_id The feed ID.
	 * @param bool      $can_edit Whether the user can edit checklists.
	 */
	public static function render_single_checklist( Checklist $checklist, $entry_id, $feed_id, $can_edit ) {
		$progress = $checklist->get_progress();

		$checklist_title = $checklist->get_title();

		// Future: When we have a checklists that have titles, the title will be displayed here.
		// For now, we'll just display the checklist title if it exists.
		if ( ! empty( $checklist_title ) ) {
			echo '<h3 class="gk-gravityboard-checklist-title">' . esc_html( $checklist_title ) . '</h3>';
		}

		// Checklist items.
		if ( ! empty( $checklist->get_items() ) ) {
			$checklist_items = '<ul class="gk-gravityboard-checklist-items">';
			foreach ( $checklist->get_items() as $item ) {
				$checklist_items .= self::render_checklist_item( $item, $entry_id, $feed_id, $checklist->get_id() );
			}
			$checklist_items .= '</ul>';
		} else {
			$checklist_items = '<p class="gk-gravityboard-no-items">' . esc_html__( 'No items in this checklist.', 'gk-gravityboard' ) . '</p>';
		}

		// Add item section for this specific checklist.
		$add_item_section = '';
		if ( $can_edit ) {
			$add_item_section = self::get_add_item_section_html( $entry_id, $feed_id, $checklist->get_id() );
		}

		$template = <<<TEMPLATE
		<div class="gk-gravityboard-checklist-container" data-checklist-id="{{checklist_id}}">
			<div class="gk-gravityboard-checklist-progress">
				<span class="gk-gravityboard-progress-text">{{progress_text}}</span>
				<div class="gk-gravityboard-progress-bar">
					<div class="gk-gravityboard-progress-fill" style="width: {{progress_percentage}}%"></div>
				</div>
			</div>
			{{checklist_items}}
			{{add_item_section}}
		</div>
TEMPLATE;

		echo strtr(
            $template,
            [
				'{{checklist_id}}'        => esc_attr( $checklist->get_id() ),
				'{{checklist_title}}'     => esc_html( $checklist->get_title() ),
				'{{checklist_items}}'     => $checklist_items, // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				'{{add_item_section}}'    => $add_item_section,
				'{{progress_percentage}}' => esc_html( $progress['percentage'] ),
				'{{progress_text}}'       => sprintf(
					// translators: 1: completed items, 2: total items, 3: percentage.
					esc_html__( '%1$d of %2$d completed (%3$d%%)', 'gk-gravityboard' ),
					esc_html( $progress['completed'] ),
					esc_html( $progress['total'] ),
					esc_html( $progress['percentage'] )
				),
			]
        ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Get the "Add an item" section HTML for a specific checklist.
	 *
	 * @param int    $entry_id The entry ID.
	 * @param string $feed_id The feed ID.
	 * @param string $checklist_id The checklist ID.
	 * @return string The HTML for the add item section.
	 */
	private static function get_add_item_section_html( $entry_id, $feed_id, $checklist_id ) {
		ob_start();
		?>
		<div class="gk-gravityboard-add-item-section" data-entry-id="<?php echo esc_attr( $entry_id ); ?>" data-feed-id="<?php echo esc_attr( $feed_id ); ?>" data-checklist-id="<?php echo esc_attr( $checklist_id ); ?>">
			<div class="gk-gravityboard-add-item-form" style="display: none;">
				<div class="gk-gravityboard-add-item-input-group">
					<input
						type="text"
						class="gk-gravityboard-add-item-input"
						placeholder="<?php echo esc_attr__( 'Add checklist item…', 'gk-gravityboard' ); ?>"
						maxlength="255"
					>
					<div class="gk-gravityboard-add-item-actions">
						<button type="button" class="button button-primary button-small gk-gravityboard-add-item-btn">
							<?php esc_html_e( 'Add', 'gk-gravityboard' ); ?>
						</button>
						<button type="button" class="button button-small gk-gravityboard-cancel-add-btn">
							<?php esc_html_e( 'Cancel', 'gk-gravityboard' ); ?>
						</button>
					</div>
				</div>
			</div>
			<button type="button" class="button button-secondary gk-gravityboard-show-add-btn">
				<span class="dashicons dashicons-plus-alt" style="margin-right: 5px;"></span>
				<?php esc_html_e( 'Add an item', 'gk-gravityboard' ); ?>
			</button>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render a single checklist item with interactivity.
	 *
	 * @param ChecklistItem $item The checklist item.
	 * @param int           $entry_id The entry ID.
	 * @param string        $feed_id The feed ID.
	 * @param string        $checklist_id The checklist ID.
	 * @return string       The rendered item HTML.
	 */
	private static function render_checklist_item( ChecklistItem $item, $entry_id, $feed_id, $checklist_id ) {
		$item_template = <<<TEMPLATE
		<li class="gk-gravityboard-checklist-item" data-item-id="[item_id]">
			<label class="gk-gravityboard-item-display">
				<input type="checkbox"
					   class="gk-gravityboard-item-checkbox"
					   data-entry-id="[entry_id]"
					   data-feed-id="[feed_id]"
					   data-checklist-id="[checklist_id]"
					   data-item-id="[item_id]"
					   [checked]>
				<span class="gk-gravityboard-item-text [completed]">[label]</span>
			</label>
		</li>
TEMPLATE;

		return strtr(
            $item_template,
            [
				'[item_id]'      => esc_attr( $item->get_id() ),
				'[entry_id]'     => esc_attr( $entry_id ),
				'[feed_id]'      => esc_attr( $feed_id ),
				'[checklist_id]' => esc_attr( $checklist_id ),
				'[checked]'      => checked( $item->is_complete(), true, false ),
				'[completed]'    => $item->is_complete() ? 'completed' : '',
				'[label]'        => esc_html( $item->get_label() ),
			]
        );
	}
}