<?php
/**
 * ErrorCollection class for managing multiple WP_Error objects.
 *
 * @package GravityKit\GravityBoard
 *
 * @since 1.1
 */

namespace GravityKit\GravityBoard;

use ArrayAccess;
use Countable;
use Iterator;
use WP_Error;

/**
 * Collection class for managing multiple WP_Error objects.
 *
 * @since 1.1
 */
class ErrorCollection implements ArrayAccess, Countable, Iterator {

	/**
	 * Array of WP_Error objects.
	 *
	 * @var WP_Error[]
	 */
	private $errors = [];

	/**
	 * Current position for Iterator.
	 *
	 * @var int
	 */
	private $position = 0;

	/**
	 * Add a WP_Error to the collection.
	 *
	 * @param WP_Error $error The error to add.
	 *
	 * @return void
	 */
	public function add( WP_Error $error ) {
		$this->errors[] = $error;
	}

	/**
	 * Check if the collection has any errors.
	 *
	 * @return bool True if there are errors, false otherwise.
	 */
	public function has_errors() {
		return ! empty( $this->errors );
	}

	/**
	 * Get all errors as an array.
	 *
	 * @return WP_Error[]
	 */
	public function get_errors() {
		return $this->errors;
	}

	/**
	 * Clear all errors from the collection.
	 *
	 * @return void
	 */
	public function clear() {
		$this->errors   = [];
		$this->position = 0;
	}

	/**
	 * Get the first error in the collection.
	 *
	 * @return WP_Error|null
	 */
	public function get_first_error() {
		return ! empty( $this->errors ) ? $this->errors[0] : null;
	}

	/**
	 * Get error data in structured format for API responses.
	 *
	 * @return array Structured error data with field names and messages.
	 */
	public function get_error_data() {
		$error_data = [
			'errors' => [],
		];

		foreach ( $this->errors as $error ) {
			if ( is_wp_error( $error ) ) {
				$error_info             = $error->get_error_data();
				$error_data['errors'][] = [
					'field_name' => $error_info['field_name'] ?? '',
					'message'    => $error->get_error_message(),
				];
			}
		}

		return $error_data;
	}

	/**
	 * Convert the error collection to a single WP_Error object.
	 *
	 * @param string $error_code The error code to use for the consolidated error. Default 'validation_failed'.
	 * @param int    $status     The HTTP status code. Default 400.
	 *
	 * @return WP_Error|null Returns null if no errors exist.
	 */
	public function as_wp_error( $error_code = 'validation_failed', $status = 400 ) {
		if ( ! $this->has_errors() ) {
			return null;
		}

		$error_data  = $this->get_error_data();
		$error_count = count( $error_data['errors'] );

		// translators: %d is the number of fields that failed validation.
		$message = _n(
			'Validation failed for %d field.',
			'Validation failed for %d fields.',
			$error_count,
			'gk-gravityboard'
		);

		if ( $error_count > 1 ) {
			$message = sprintf( $message, $error_count );
		}

		return new WP_Error(
			$error_code,
			esc_html( $message ),
			array_merge( [ 'status' => $status ], $error_data )
		);
	}

	/**
	 * Get all error messages as a flat array.
	 *
	 * @return string[]
	 */
	public function get_error_messages() {
		$messages = [];

		foreach ( $this->errors as $error ) {
			$error_messages = $error->get_error_messages();
			$messages       = array_merge( $messages, $error_messages );
		}

		return $messages;
	}

	/**
	 * Get all error codes as a flat array.
	 *
	 * @return string[]
	 */
	public function get_error_codes() {
		$codes = [];

		foreach ( $this->errors as $error ) {
			$error_codes = $error->get_error_codes();
			$codes       = array_merge( $codes, $error_codes );
		}

		return array_unique( $codes );
	}

	/**
	 * Check if offset exists.
	 *
	 * @param mixed $offset The offset to check.
	 *
	 * @return bool
	 */
	#[\ReturnTypeWillChange]
	public function offsetExists( $offset ) {
		return isset( $this->errors[ $offset ] );
	}

	/**
	 * Get value at offset.
	 *
	 * @param mixed $offset The offset to retrieve.
	 *
	 * @return WP_Error|null
	 */
	#[\ReturnTypeWillChange]
	public function offsetGet( $offset ) {
		return $this->errors[ $offset ] ?? null;
	}

	/**
	 * Set value at offset.
	 *
	 * @param mixed $offset The offset to set.
	 * @param mixed $value  The value to set (must be WP_Error).
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function offsetSet( $offset, $value ) {
		if ( ! $value instanceof WP_Error ) {
			return;
		}

		if ( is_null( $offset ) ) {
			$this->errors[] = $value;
		} else {
			$this->errors[ $offset ] = $value;
		}
	}

	/**
	 * Unset value at offset.
	 *
	 * @param mixed $offset The offset to unset.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function offsetUnset( $offset ) {
		unset( $this->errors[ $offset ] );
		$this->errors = array_values( $this->errors ); // Re-index array.
	}

	/**
	 * Count errors in the collection.
	 *
	 * @return int
	 */
	#[\ReturnTypeWillChange]
	public function count() {
		return count( $this->errors );
	}

	/**
	 * Rewind iterator to the first element.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function rewind() {
		$this->position = 0;
	}

	/**
	 * Get current element.
	 *
	 * @return WP_Error|null
	 */
	#[\ReturnTypeWillChange]
	public function current() {
		return $this->errors[ $this->position ] ?? null;
	}

	/**
	 * Get current key.
	 *
	 * @return int
	 */
	#[\ReturnTypeWillChange]
	public function key() {
		return $this->position;
	}

	/**
	 * Move to next element.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function next() {
		++$this->position;
	}

	/**
	 * Check if current position is valid.
	 *
	 * @return bool
	 */
	#[\ReturnTypeWillChange]
	public function valid() {
		return isset( $this->errors[ $this->position ] );
	}
}
