<?php
/**
 * Registers the GravityBoard feed with Gravity Forms.
 *
 * @since 1.0
 *
 * @package GravityKit\GravityBoard
 */

namespace GravityKit\GravityBoard;

use Exception;
use GFForms;
use GFFeedAddOn;
use GFAPI;
use GFCommon;
use GravityKit\GravityBoard\QueryFilters\QueryFilters;
use WP_Error;
use Gravity_Forms\Gravity_Forms\Settings\Fields\Generic_Map;
use Gravity_Forms\Gravity_Forms\Settings\Settings;
use GravityKit\GravityBoard\Admin\Board_View;
use GravityKit\GravityBoard\BoardsOverview\Boards_Overview;
use GravityKit\GravityBoard\Checklists\Checklists;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'GFForms' ) ) {
	exit;
}

if ( ! class_exists( 'GFFeedAddOn' ) ) {
	GFForms::include_feed_addon_framework();
}

/**
 * Class Feed
 *
 * @package GravityKit\GravityBoard
 */
class Feed extends GFFeedAddOn {

	/**
	 * Version number.
	 *
	 * @var string
	 */
	protected $_version = '1.0';

	/**
	 * Minimum Gravity Forms version.
	 *
	 * @var string
	 */
	protected $_min_gravityforms_version = '2.5';

	/**
	 * Slug for the add-on.
	 *
	 * @var string
	 */
	protected $_slug = 'gravityboard';

	/**
	 * Path to the plugin file.
	 *
	 * @var string
	 */
	protected $_path = 'gravityboard/gravityboard.php';

	/**
	 * Full path to the plugin file.
	 *
	 * @var string
	 */
	protected $_full_path = GRAVITYBOARD_FILE;

	/**
	 * Title of the add-on.
	 *
	 * @var string
	 */
	protected $_title = 'Gravity Forms Kanban Board Add-On';

	/**
	 * Short title of the add-on.
	 *
	 * @var string
	 */
	protected $_short_title = 'GravityBoard';

	/**
	 * @var string|array A string or an array of capabilities or roles that have access to the form settings
	 */
	protected $_capabilities_form_settings = 'gravityforms_edit_forms';

	const ENTRY_META_POSITION_KEY = 'gravityboard_position_%s';

	const MAX_ENTRIES_PER_BOARD = 1000;

	/**
	 * Instance of the class.
	 *
	 * @var Feed|null
	 */
	private static $_instance = null;

	/**
	 * Cached list of users
	 *
	 * @since 1.0
	 * @var array
	 */
	private static $users_cache = [];

	/**
	 * Get singleton instance of the plugin
	 *
	 * @since 1.0
	 *
	 * @return Feed Instance of the feed class.
	 */
	public static function get_instance() {

		if ( null === self::$_instance ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Enqueue media assets on the feed settings page.
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	public function feed_settings_init() {

		parent::feed_settings_init();

		// Hook the method to enqueue media assets.
		add_action( 'admin_enqueue_scripts', 'wp_enqueue_media' );

		if ( $this->is_feed_edit_page() && rgget( 'subview' ) === $this->_slug ) {
			add_filter( 'gform_feed_settings_before_fields', [ $this, 'add_feed_settings_before_fields' ] );
		}
	}

	/**
	 * Add content before feed settings fields.
	 *
	 * @since 1.0
	 *
	 * @param string $html Existing HTML content.
	 * @return string Modified HTML content.
	 */
	public function add_feed_settings_before_fields( $html ) {
		ob_start();
		?>
		<div class="gravityboard-feed-settings-before-fields">
			<div class="gravityboard-feed-settings-before-fields-icon">
				<svg width="64" height="64" viewBox="0 0 80 80" fill="none" xmlns="http://www.w3.org/2000/svg">
					<g clip-path="url(#clip0_7850_1254)">
						<rect width="100" height="100" fill="#0079BF" />
						<path fill-rule="evenodd" clip-rule="evenodd" d="M37.7971 54.6666V56.5C37.7971 59.5376 35.3049 62 32.2691 62H22.5281C19.4521 62 17 59.5376 17 56.5L17 29.0001V25.3333V23.5001C17 20.4625 19.4521 18.0001 22.5281 18.0001L32.2691 18.0001C35.3049 18.0001 37.7971 20.4625 37.7971 23.5001V51.0001V54.6666ZM32.2691 21.6667C33.2691 21.6667 34.1115 22.4875 34.1115 23.5001V56.5C34.1115 57.5125 33.2691 58.3334 32.2691 58.3334H22.5281C21.488 58.3334 20.6856 57.5125 20.6856 56.5L20.6856 23.5001C20.6856 22.4875 21.488 21.6667 22.5281 21.6667H32.2691Z" fill="white" />
						<path fill-rule="evenodd" clip-rule="evenodd" d="M43 46.8333L43 45C43 41.9624 45.4922 39.5 48.5281 39.5L58.2691 39.5C61.345 39.5 63.7971 41.9624 63.7971 45L63.7971 44L63.7971 47.6667L63.7971 56.5C63.7971 59.5376 61.345 62 58.2691 62L48.5281 62C45.4922 62 43 59.5376 43 56.5L43 57.4999L43 46.8333ZM48.5281 58.3333C47.528 58.3333 46.6856 57.5126 46.6856 56.5L46.6856 45C46.6856 43.9875 47.528 43.1666 48.5281 43.1666L58.2691 43.1666C59.3092 43.1666 60.1115 43.9875 60.1115 45L60.1115 56.5C60.1115 57.5126 59.3092 58.3333 58.2691 58.3333L48.5281 58.3333Z" fill="white" />
						<path fill-rule="evenodd" clip-rule="evenodd" d="M63.7971 28.1667V30C63.7971 33.0376 61.3049 35.5 58.2691 35.5H48.5281C45.4521 35.5 43 33.0376 43 30V31V27.3333V23.5C43 20.4624 45.4521 18 48.5281 18L58.2691 18C61.3049 18 63.7971 20.4624 63.7971 23.5V22.5001V28.1667ZM58.2691 21.6667C59.2691 21.6667 60.1115 22.4874 60.1115 23.5V30C60.1115 31.0125 59.2691 31.8334 58.2691 31.8334H48.5281C47.488 31.8334 46.6856 31.0125 46.6856 30V23.5C46.6856 22.4874 47.488 21.6667 48.5281 21.6667H58.2691Z" fill="white" />
					</g>
					<defs>
						<clipPath id="clip0_7850_1254">
							<rect width="80" height="80" rx="8" fill="white" />
						</clipPath>
					</defs>
				</svg>
			</div>
			<div class="gravityboard-feed-settings-before-fields-content">
				<h3><?php esc_html_e( 'GravityBoard Settings', 'gk-gravityboard' ); ?></h3>
				<p><?php esc_html_e( 'GravityBoard is a powerful tool for managing your Gravity Forms entries in a Kanban board format.', 'gk-gravityboard' ); ?> <a href="https://docs.gravitykit.com/category/1049-gravityboard" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Read the documentation to learn more.', 'gk-gravityboard' ); ?></a></p>
				<?php
				$feed_id   = $this->get_current_feed_id();
				$board_url = $feed_id ? self::get_view_feed_url( $feed_id ) : '';
				?>
				<?php if ( $feed_id ) { ?>
				<div>
					<a
						aria-label="<?php esc_attr_e( 'View Board', 'gk-gravityboard' ); ?>"
						href="<?php echo esc_url( $board_url ); ?>"
						class="preview-form gform-button gform-button--white gform-button--icon-leading"
					>
						<i class="gform-button__icon gform-common-icon gform-common-icon--eye"></i>
						<?php esc_html_e( 'View Board', 'gk-gravityboard' ); ?>
					</a>
				</div>
				<?php } else { ?>
				<div>
					<a
						aria-label="<?php esc_attr_e( 'Getting Started with GravityBoard', 'gk-gravityboard' ); ?>"
						href="<?php echo esc_url( Helpers::get_getting_started_url( 'new-feed-header' ) ); ?>"
						class="preview-form gform-button gform-button--white gform-button--icon-leading"
						target="_blank"
						rel="noopener noreferrer"
					>
						<i class="gform-button__icon gform-common-icon gform-common-icon--information-circle"></i>
						<?php esc_html_e( 'Getting Started with GravityBoard', 'gk-gravityboard' ); ?>
						<span class="screen-reader-text"> <?php esc_html_e( '(This link opens in a new window.)', 'gk-gravityboard' ); ?></span>
					</a>
				</div>
				<?php } ?>
			</div>
		</div>
		<?php
		$html .= ob_get_clean();
		return $html;
	}

	/**
	 * Allow duplicate feeds
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed to check.
	 * @return bool True if the feed can be duplicated, false otherwise.
	 */
	public function can_duplicate_feed( $feed ) {
		return true;
	}

	/**
	 * Get users that can be assigned based on configured roles
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed configuration.
	 * @return array Array of users with id, text, avatar, and email
	 */
	public function get_assignable_users( $feed ) {

		$allowed_roles = rgars( $feed, 'meta/assignee_roles', [] );

		if ( empty( $allowed_roles ) ) {
			return [];
		}

		$cache_key = md5( wp_json_encode( $allowed_roles ) );
		if ( isset( self::$users_cache[ $cache_key ] ) ) {
			return self::$users_cache[ $cache_key ];
		}

		$users      = [];
		$query_args = [
			'number'  => -1,
			'orderby' => 'display_name',
			'order'   => 'ASC',
			'fields'  => [ 'ID', 'display_name', 'user_email', 'user_login' ],
		];
		if ( ! empty( $allowed_roles ) && [ 'any' ] !== $allowed_roles ) {
			$query_args['role__in'] = $allowed_roles;
		}

		$user_query = new \WP_User_Query( $query_args );

		if ( ! empty( $user_query->results ) ) {
			foreach ( $user_query->results as $user ) {

				$display_name = $user->display_name ? $user->display_name : $user->user_login;

				$users[] = [
					'id'     => $user->ID,
					'text'   => (string) $display_name,
					'avatar' => get_avatar_url( $user->ID, [ 'size' => 64 ] ),
					'email'  => $user->user_email,
				];
			}
		}

		self::$users_cache[ $cache_key ] = $users;

		return $users;
	}

	/**
	 * Get WordPress user roles for settings dropdowns
	 *
	 * @since 1.0
	 *
	 * @return array Array of role choices formatted for Gravity Forms settings
	 */
	private function get_role_choices() {
		$choices = [
			[
				'label'   => esc_html__( 'Access Control', 'gk-gravityboard' ),
				'choices' => [
					[
						'label' => esc_html__( 'Everyone (Including Logged-Out Users)', 'gk-gravityboard' ),
						'value' => 'enabled',
					],
					[
						'label' => esc_html__( 'All Logged-In Users', 'gk-gravityboard' ),
						'value' => 'read',
					],
				],
			],
		];

		$caps = [
			'gravityforms_edit_forms'       => esc_html__( 'Edit Forms', 'gk-gravityboard' ),
			'gravityforms_delete_forms'     => esc_html__( 'Delete Forms', 'gk-gravityboard' ),
			'gravityforms_create_form'      => esc_html__( 'Create Forms', 'gk-gravityboard' ),
			'gravityforms_view_entries'     => esc_html__( 'View Entries', 'gk-gravityboard' ),
			'gravityforms_edit_entries'     => esc_html__( 'Edit Entries', 'gk-gravityboard' ),
			'gravityforms_delete_entries'   => esc_html__( 'Delete Entries', 'gk-gravityboard' ),
			'gravityforms_edit_entry_notes' => esc_html__( 'Manage Entry Notes', 'gk-gravityboard' ),
			'gravityforms_view_settings'    => esc_html__( 'View Settings', 'gk-gravityboard' ),
			'gravityforms_edit_settings'    => esc_html__( 'Edit Settings', 'gk-gravityboard' ),
			'gravityforms_export_entries'   => esc_html__( 'Export Entries', 'gk-gravityboard' ),
		];

		foreach ( $caps as $cap => $label ) {
			$caps_choices[] = [
				// translators: %s: Gravity Forms capability label.
				'label' => sprintf( esc_html__( 'Gravity Forms: %s', 'gk-gravityboard' ), $label ),
				'value' => $cap,
			];
		}

		$choices[] = [
			'label'   => esc_html__( 'Gravity Forms Capabilities', 'gk-gravityboard' ),
			'choices' => $caps_choices,
		];

		// Get all WordPress roles.
		$roles = wp_roles()->roles;

		foreach ( $roles as $role_key => $role ) {
			$roles_choices[] = [
				'label' => $role['name'],
				'value' => $role_key,
			];
		}

		$choices[] = [
			'label'   => esc_html__( 'WordPress Roles', 'gk-gravityboard' ),
			'choices' => $roles_choices,
		];

		return $choices;
	}

	/**
	 * Get the default assignee role, properly formatted.
	 *
	 * Gravity Forms expects the _label_, not the value, for the default value. That is translated.
	 * So we need to make sure we return the correct, translated, label for the administrator role.
	 *
	 * @since 1.0
	 *
	 * @return array Array of default assignee roles. Default is the administrator role.
	 */
	private function get_default_assignee_roles() {
		$roles = wp_roles()->get_names();

		if ( isset( $roles['administrator'] ) ) {
			return [ $roles['administrator'] ];
		}

		return [];
	}

	/**
	 * Get WordPress user roles for settings dropdowns in a flat array
	 *
	 * @since 1.0
	 *
	 * @return array Array of role choices formatted for Gravity Forms settings
	 */
	private function get_role_choices_flat() {
		$choices = [
			[
				'label' => esc_html__( 'All User Roles', 'gk-gravityboard' ),
				'value' => 'any',
			],
		];
		$roles   = wp_roles()->get_names();

		foreach ( $roles as $role_key => $role_name ) {
			$choices[] = [
				'label' => $role_name,
				'name'  => $role_key,
			];
		}

		return $choices;
	}

	/**
	 * Get the field types that can be used as lanes
	 *
	 * @since 1.0
	 *
	 * @return array The field types that can be used as lanes.
	 */
	private function get_lane_field_types() {
		return [
			'select' => esc_html__( 'Select', 'gk-gravityboard' ),
			'radio'  => esc_html__( 'Radio', 'gk-gravityboard' ),
		];
	}

	/**
	 * Get the fields for the feed settings
	 *
	 * @since 1.0
	 *
	 * @return array The fields for the feed settings
	 */
	public function feed_settings_fields() {
		$lane_field_types = $this->get_lane_field_types();
		$lane_fields      = $this->get_fields_by_type( array_keys( $lane_field_types ) );

		if ( 1 >= count( $lane_fields ) ) {
			return [
				[
					'title'  => esc_html__( 'Missing a required field type', 'gk-gravityboard' ),
					'fields' => [
						[
							'label' => '<h3>' . esc_html__( 'This form is missing a field required by GravityBoard.', 'gk-gravityboard' ) . '</h4>',
							'type'  => 'html',
							'name'  => 'gravityboard_not_supported',
							'html'  =>
							'<div>' .
								'<p class="onboarding-text">' . esc_html__( 'GravityBoard uses choice-based fields to define the lanes that an entry is in (for example, "Backlog", "In Progress", "Done"). This form does not contain any of these fields.', 'gk-gravityboard' ) . '</p>' .
								// translators: Do not translate [url], [/url], or [fields]. They are placeholders for the edit form link and the list of fields.
								'<p class="onboarding-text">' . strtr(
                                    esc_html__( 'Please [url]edit the form[/url] to add one of the following fields: [fields].', 'gk-gravityboard' ),
                                    [
										'[url]'    => '<a href="' . esc_url( remove_query_arg( [ 'view', 'subview', 'fid' ] ) ) . '">',
										'[/url]'   => '</a>',
										'[fields]' => '<strong>' . implode( '</strong>, <strong>', array_values( $lane_field_types ) ) . '</strong>',
									]
                                ) . '</p>' .
								'</div>',
						],
					],
				],
			];
		}

		$settings = [
			[
				'title'    => esc_html__( 'Configure Board', 'gk-gravityboard' ),
				'sections' => [
					[
						'fields' => [
							[
								'label'       => esc_html__( 'Board Name', 'gk-gravityboard' ),
								'type'        => 'text',
								'name'        => 'feed_name',
								'description' => esc_html__( 'Enter a name to identify this board. If left blank, a name will be created automatically. This is publicly visible.', 'gk-gravityboard' ),
								'class'       => 'medium',
								'required'    => false,
								'placeholder' => $this->get_default_feed_name(),
							],
							[
								'type'          => 'hidden',
								'name'          => 'feed_date_created',
								'default_value' => gmdate( 'Y-m-d H:i:s' ),
							],
							[
								'type'          => 'hidden',
								'name'          => 'feed_created_by',
								'default_value' => get_current_user_id(),
							],
						],
					],
					[
						'title'  => esc_html__( 'Field Mapping', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'            => 'card_fields',
								'type'            => 'field_map',
								'description'     => '<h3>' . esc_html__( 'Select the fields to show on each card.', 'gk-gravityboard' ) . '</h3><p>' . esc_html__( 'Choose the fields to display for each card in the board.', 'gk-gravityboard' ) . '</p>',
								'disabled_custom' => true,
								'field_map'       => [
									[
										'name'       => 'lane_field',
										'label'      => esc_html__( 'Lane Field', 'gk-gravityboard' ),
										'required'   => true,
										'desc'       => esc_html__( 'Select the field to be used as lanes in the Kanban board.', 'gk-gravityboard' ),
										'field_type' => array_keys( $lane_field_types ),
									],
									[
										'name'          => 'title',
										'label'         => esc_html__( 'Card Title', 'gk-gravityboard' ),
										'required'      => true,
										'default_value' => $this->get_first_field_by_type( 'text' ),
										'field_type'    => [ 'text', 'email', 'website' ],
									],
									[
										'name'          => 'description',
										'label'         => esc_html__( 'Card Description', 'gk-gravityboard' ),
										'required'      => false,
										'default_value' => $this->get_first_field_by_type( 'textarea' ),
										'field_type'    => [ 'textarea', 'text', 'email', 'website' ],
									],
									[
										'name'          => 'label',
										'label'         => esc_html__( 'Label', 'gk-gravityboard' ),
										'required'      => false,
										'default_value' => '',
										'field_type'    => [ 'text', 'email', 'website', 'select', 'radio' ],
									],
									[
										'name'          => 'due_date',
										'label'         => esc_html__( 'Due Date', 'gk-gravityboard' ),
										'required'      => false,
										'default_value' => '',
										'field_type'    => 'date',
									],
								],
							],
						],
					],
					[
						'title'  => esc_html__( 'Entry Notes', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'enable_entry_notes',
								'label'         => esc_html__( 'Enable Entry Notes', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'Allow viewing and adding Gravity Forms Entry Notes in the card details modal. This is similar to allowing comments on cards.', 'gk-gravityboard' ),
								'default_value' => '1',
							],
						],
					],
					[
						'title'  => esc_html__( 'Assignees', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'enable_assignees',
								'label'         => esc_html__( 'Enable Assignees', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'Allow users to be assigned to cards.', 'gk-gravityboard' ),
								'default_value' => '1',
							],
							[
								'name'          => 'assignee_roles[]',
								'label'         => esc_html__( 'Available User Roles', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles will be available when assigning users to cards.', 'gk-gravityboard' ),
								'default_value' => $this->get_default_assignee_roles(),
								'choices'       => $this->get_role_choices_flat(),
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_assignees',
											'value' => '1',
										],
									],
								],
							],
						],
					],
					[
						'title'  => esc_html__( 'Checklists', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'enable_card_checklists',
								'label'         => esc_html__( 'Enable Card Checklists', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'Allow users to add checklists and checklist items to cards on this board.', 'gk-gravityboard' ),
								'default_value' => '0',
							],
						],
					],
					[
						'title'  => esc_html__( 'Attachments', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'enable_card_attachments',
								'label'         => esc_html__( 'Enable Card Attachments', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'Allow users to attach files to cards on this board.', 'gk-gravityboard' ),
								'default_value' => '0',
							],
						],
					],
				],
			],
			[
				'title'    => esc_html__( 'Notifications', 'gk-gravityboard' ),
				'sections' => [
					[
						'title'  => esc_html__( 'Form Notifications', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'trigger_notifications',
								'label'         => esc_html__( 'Trigger Form Notifications', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'When cards are added, updated, or deleted, trigger any form notifications that would normally be sent by Gravity Forms.', 'gk-gravityboard' ),
								'default_value' => '1',
							],
						],
					],
					[
						'title'      => esc_html__( 'Mention Notifications', 'gk-gravityboard' ),
						'fields'     => [
							[
								'name'          => 'enable_mention_notifications',
								'label'         => esc_html__( 'Enable Mention Notifications', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'Send an email notification when a user is mentioned in an entry note.', 'gk-gravityboard' ),
								'default_value' => '1',
							],
							[
								'name'          => 'mention_notification_subject',
								'label'         => esc_html__( 'Mention Notification Subject', 'gk-gravityboard' ),
								'type'          => 'text',
								'class'         => 'large',
								'description'   => sprintf(
									// translators: %s: List of available merge tags.
									esc_html__( 'Enter the subject for the mention notification email. Available merge tags: %s', 'gk-gravityboard' ),
									'<code>{notifier_name}</code>, <code>{mentioned_user_name}</code>, <code>{site_title}</code>, <code>{admin_email}</code>, <code>{board_name}</code>, <code>{entry_id}</code>, <code>{form_id}</code>'
								),
								'default_value' => esc_html__( '[GravityBoard] You were mentioned by {notifier_name} on {board_name}', 'gk-gravityboard' ),
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_mention_notifications',
											'value' => '1',
										],
									],
								],
							],
							[
								'name'          => 'mention_notification_body',
								'label'         => esc_html__( 'Mention Notification Body', 'gk-gravityboard' ),
								'type'          => 'textarea',
								'class'         => 'large merge-tag-support mt-wp_editor',
								'use_editor'    => true,
								'description'   => sprintf(
									// translators: %s: List of available merge tags.
									esc_html__( 'Enter the body for the mention notification email. Available merge tags: %s', 'gk-gravityboard' ),
									'<code>{notifier_name}</code>, <code>{mentioned_user_name}</code>, <code>{note_excerpt}</code>, <code>{entry_details_url}</code>, <code>{entry_id}</code>, <code>{form_id}</code>, <code>{board_name}</code>, <code>{site_title}</code>, <code>{admin_email}</code>'
								),
								'default_value' => implode(
									"\n",
									[
										// translators: %s: Mentioned user's display name.
										sprintf( esc_html__( 'Hi %s,', 'gk-gravityboard' ), '{mentioned_user_name}' ),
										'',
										// translators: %1$s Notifier's name, %2$s Board name.
										sprintf( esc_html__( '%1$s mentioned you in a note on board "%2$s":', 'gk-gravityboard' ), '{notifier_name}', '{board_name}' ),
										'',
										'> {note_excerpt}',
										'',
										// translators: %1$s: Entry ID, %2$s: Form ID.
										sprintf( esc_html__( 'This was regarding entry #%1$s (form #%2$s).', 'gk-gravityboard' ), '{entry_id}', '{form_id}' ),
										// translators: %s: Entry details URL.
										sprintf( esc_html__( 'You can view the entry details here: %s', 'gk-gravityboard' ), '{entry_details_url}' ),
										'',
										esc_html__( 'Thank you,', 'gk-gravityboard' ),
										esc_html__( 'The GravityBoard System', 'gk-gravityboard' ),
									]
								),
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_mention_notifications',
											'value' => '1',
										],
									],
								],
							],
						],
						'dependency' => [
							'live'   => true,
							'fields' => [
								[
									'field'  => 'enable_entry_notes',
									'values' => [ '1' ],
								],
							],
						],
					],
					[
						'title'      => esc_html__( 'Mention Notifications', 'gk-gravityboard' ),
						'fields'     => [
							[
								'name'  => 'mention_notifications_requires_notes',
								'label' => esc_html__( 'Mentions require Entry Notes to be enabled.', 'gk-gravityboard' ),
								'type'  => 'html',
								'html'  => esc_html__( 'Mentions allow you to @mention users when leaving notes on cards. This requires Entry Notes to be enabled in the Configure Board section.', 'gk-gravityboard' ),
							],
						],
						'dependency' => [
							'live'   => true,
							'fields' => [
								[
									'field'  => 'enable_entry_notes',
									'values' => [ '', '0', false ],
								],
							],
						],
					],
				],
			],
			[
				'title'    => esc_html__( 'Permissions', 'gk-gravityboard' ),
				'sections' => [
					[
						'title'  => esc_html__( 'Board Permissions', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'view_board_role',
								'label'         => esc_html__( 'Who Can View Board', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => sprintf(
									// translators: %1$s: Capability name. %2$s: Capability name.
									esc_html__( 'Select which user roles can view the board. Users with any of the selected roles will have access. Leave empty to disable access for everyone. Note: Users who have full access to Gravity Forms or are site admins will always be able to view the board (specifically, users with the %1$s or %2$s capability).', 'gk-gravityboard' ),
									'<code>manage_options</code>',
									'<code>gform_full_access</code>'
								),
								'default_value' => [ 'gravityforms_view_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
						],
					],
					[
						'title'  => esc_html__( 'Card Permissions', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'card_add_role',
								'label'         => esc_html__( 'Who Can Add Cards', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can add cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
							[
								'name'          => 'card_edit_role',
								'label'         => esc_html__( 'Who Can Edit Cards', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can edit cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
							[
								'name'          => 'card_delete_role',
								'label'         => esc_html__( 'Who Can Delete Cards', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can delete cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_delete_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
						],
					],
					[
						'title'  => esc_html__( 'Lane Permissions', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'lane_modify_role',
								'label'         => esc_html__( 'Who Can Modify Lanes', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can add, reorder, and delete lanes. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_forms' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
						],
					],
					[
						'title'  => esc_html__( 'Assignee Permissions', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'       => 'assignees_disabled',
								// translators: %s: The feature that is disabled.
								'label'      => sprintf( esc_html__( '%s are disabled.', 'gk-gravityboard' ), esc_html__( 'Assignees', 'gk-gravityboard' ) ),
								'type'       => 'html',
								// translators: %1$s: The feature that is disabled.
								'html'       => sprintf( esc_html__( 'To configure permissions for %1$s, please enable %1$s in the Configure Board tab.', 'gk-gravityboard' ), esc_html__( 'Assignees', 'gk-gravityboard' ) ),
								'dependency' => [
									'live'   => true,
									'fields' => [
										[
											'field'  => 'enable_assignees',
											'values' => [ '0', false, '' ],
										],
									],
								],
							],
							[
								'name'          => 'view_assignees_role',
								'label'         => esc_html__( 'View Assignees', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can view the assignees. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_assignees',
											'value' => '1',
										],
									],
								],
							],
							[
								'name'          => 'assign_users_role',
								'label'         => esc_html__( 'Assign Users', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can assign users to cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_assignees',
											'value' => '1',
										],
									],
								],
							],
						],
					],
					[
						'title'  => esc_html__( 'Entry Note Permissions', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'       => 'entry_note_disabled',
								// translators: %s: The feature that is disabled.
								'label'      => sprintf( esc_html__( '%s are disabled.', 'gk-gravityboard' ), esc_html__( 'Entry Notes', 'gk-gravityboard' ) ),
								'type'       => 'html',
								// translators: %1$s: The feature that is disabled.
								'html'       => sprintf( esc_html__( 'To configure permissions for %1$s, please enable %1$s in the Configure Board tab.', 'gk-gravityboard' ), esc_html__( 'Entry Notes', 'gk-gravityboard' ) ),
								'dependency' => [
									'live'   => true,
									'fields' => [
										[
											'field'  => 'enable_entry_notes',
											'values' => [ '0', false, '' ],
										],
									],
								],
							],
							[
								'name'          => 'entry_note_view_role',
								'label'         => esc_html__( 'Who Can View Entry Notes', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can view entry notes (comments) on cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_view_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_entry_notes',
											'value' => '1',
										],
									],
								],
							],
							[
								'name'          => 'entry_note_add_role',
								'label'         => esc_html__( 'Who Can Add Entry Notes', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can add entry notes (comments) to cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_entry_notes',
											'value' => '1',
										],
									],
								],
							],
							[
								'name'          => 'entry_note_manage_role',
								'label'         => esc_html__( 'Who Can Edit and Delete Entry Notes', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can edit or delete entry notes on cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entry_notes' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
								'dependency'    => [
									'live'   => true,
									'fields' => [
										[
											'field' => 'enable_entry_notes',
											'value' => '1',
										],
									],
								],
							],
						],
					],
					[
						'title'      => esc_html__( 'Checklist Permissions', 'gk-gravityboard' ),
						'dependency' => [
							'live'   => true,
							'fields' => [
								[
									'field' => 'enable_card_checklists',
									'value' => '1',
								],
							],
						],
						'fields'     => [
							[
								'name'          => 'view_checklists_role',
								'label'         => esc_html__( 'Who Can View Checklists', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can view card checklists. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_view_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
							[
								'name'          => 'edit_checklists_role',
								'label'         => esc_html__( 'Who Can Edit Checklists', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can add, edit, and delete checklist items. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
						],
					],
					[
						'title'      => esc_html__( 'Attachment Permissions', 'gk-gravityboard' ),
						'dependency' => [
							'live'   => true,
							'fields' => [
								[
									'field' => 'enable_card_attachments',
									'value' => '1',
								],
							],
						],
						'fields'     => [
							[
								'name'          => 'view_attachments_role',
								'label'         => esc_html__( 'Who Can View Attachments', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can view card attachments. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_view_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
							[
								'name'          => 'add_attachments_role',
								'label'         => esc_html__( 'Who Can Add Attachments', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can add attachments to cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_edit_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
							[
								'name'          => 'delete_attachments_role',
								'label'         => esc_html__( 'Who Can Delete Attachments', 'gk-gravityboard' ),
								'type'          => 'select',
								'multiple'      => true,
								'enhanced_ui'   => true,
								'description'   => esc_html__( 'Select which user roles can delete attachments from cards. Users with any of the selected roles will have access.', 'gk-gravityboard' ),
								'default_value' => [ 'gravityforms_delete_entries' ],
								'choices'       => $this->get_role_choices(),
								'validation_callback' => [ $this, 'validate_multi_select_permission' ],
							],
						],
					],
				],
			],
			[
				'title'    => esc_html__( 'Appearance', 'gk-gravityboard' ),
				'sections' => [
					[
						'title'  => esc_html__( 'Board Size', 'gk-gravityboard' ),
						'fields' => [
							[
								'label'               => esc_html__( 'Minimum Height', 'gk-gravityboard' ),
								'type'                => 'text',
								'name'                => 'board_min_height',
								'description'         => esc_html__( 'Set the minimum height for the board. Supports all CSS length units (px, %, vh, vw, em, rem, etc.) and keywords (auto, max-content, min-content, etc.).', 'gk-gravityboard' ),
								'class'               => 'css-dimension-field',
								'placeholder'         => '400px',
								'default_value'       => '',
								'validation_callback' => array( $this, 'validate_css_dimension_field' ),
								'validation_pattern'  => $this->get_field_validation_config()['board_min_height']['pattern'],
								'validation_message'  => $this->get_field_validation_config()['board_min_height']['message'],
							],
						],
					],
					[
						'title'  => esc_html__( 'Board Background', 'gk-gravityboard' ),
						'fields' => [
							[
								'label'         => esc_html__( 'Board Background Color', 'gk-gravityboard' ),
								'type'          => 'text',
								'name'          => 'board_background_color',
								'description'   => esc_html__( 'Sets the background color for the board. This is overridden if a background image is set.', 'gk-gravityboard' ),
								'class'         => 'small gf-color-picker',
								'placeholder'   => '',
								'default_value' => '#0079bf',
							],
							[
								'name'          => 'board_background_image_id',
								'label'         => '[Hidden] Background image storage',
								'type'          => 'hidden',
								'default_value' => '',
							],
							[
								'label' => esc_html__( 'Board Background Image', 'gk-gravityboard' ),
								'type'  => 'html',
								'name'  => 'board_background_image_display',
								'html'  => function () {
									$settings = $this->get_current_settings();

									// Get the ID from the actual hidden setting field value.
									$image_id  = rgars( $settings, 'board_background_image_id' );
									$image_url = $image_id ? wp_get_attachment_image_url( $image_id, 'medium' ) : '';

									// Preview div - MUST keep the ID for JS targeting.
									$html = '<div id="gravityboard-background-image-preview" style="margin-bottom: 10px; max-width: 266px; max-height: 200px; min-height: 200px;' . ( $image_url ? 'background-image: url(' . esc_url( $image_url ) . '); background-size: cover; background-position: center; border: 1px solid #ccc;' : 'display: none;' ) . '"></div>';

									// Upload button - MUST keep the ID for JS targeting.
									$html .= '<button type="button" class="button" id="gravityboard-upload-background-image-button">' . esc_html__( 'Upload Image', 'gk-gravityboard' ) . '</button> ';

									// Remove button (initially hidden if no image) - MUST keep the ID for JS targeting.
									$html .= '<button type="button" class="button button-link-delete" id="gravityboard-remove-background-image-button" style="' . ( ! $image_id ? 'display:none;' : '' ) . '">' . esc_html__( 'Remove Image', 'gk-gravityboard' ) . '</button>';

									$html .= '<p class="description">' . esc_html__( 'Select an image for the board background. This will override the background color setting.', 'gk-gravityboard' ) . '</p>';

									return $html;
								},
							],
						],
					],
					[
						'title'  => esc_html__( 'Content Rendering', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'enable_markdown',
								'label'         => esc_html__( 'Render Markdown', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'When enabled, card descriptions will be rendered as Markdown, allowing for formatted text, links, lists, and more. When disabled, only plain text with clickable links will be shown.', 'gk-gravityboard' ),
								'default_value' => '1',
							],
						],
					],
					[
						'title'       => esc_html__( 'Lane Background', 'gk-gravityboard' ),
						'description' => esc_html__( 'Customize the background color for each lane on your board. Leave a color blank to use the default lane appearance.', 'gk-gravityboard' ),
						'fields'      => $this->get_lane_background_color_fields(),
					],
				],
			],
			[
				'title'    => esc_html__( 'Advanced', 'gk-gravityboard' ),
				'sections' => [
					[
						'title'  => esc_html__( 'Feed Conditions', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'        => 'conditional_logic',
								'full_screen' => false,
								'label'       => esc_html__( 'Conditional Logic', 'gk-gravityboard' ),
								'tooltip'     => 'export_conditional_logic',
								'type'        => 'html',
								'html'        => sprintf(
									'<span class="gform-settings-description" id="description-%s">%s</span><div id="gk-query-filters"></div>',
									'conditional_logic',
									esc_html__(
										'Limit the displayed entries. Only entries that match these conditions will be displayed on the board.',
										'gk-gravityboard'
									)
								),
							],
						],
					],
					[
						'title'  => esc_html__( 'Entry Deletion', 'gk-gravityboard' ),
						'fields' => [
							[
								'label'         => esc_html__( 'Entry Deletion', 'gk-gravityboard' ),
								'type'          => 'radio',
								'name'          => 'deletion_mode',
								'horizontal'    => true,
								'description'   => esc_html__( 'What happens when a card is deleted from the board?', 'gk-gravityboard' ),
								'default_value' => 'trash',
								'choices'       => [
									[
										'label' => esc_html__( 'Move to Trash', 'gk-gravityboard' ),
										'value' => 'trash',
									],
									[
										'label' => esc_html__( 'Delete Permanently', 'gk-gravityboard' ),
										'value' => 'delete',
									],
								],
							],
						],
					],
					[
						'title'  => esc_html__( 'Attachments', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'allow_all_attachment_types',
								'label'         => esc_html__( 'Allow All File Types', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'When enabled, allows uploading files of any type, bypassing WordPress file type restrictions. When disabled, only file types allowed by WordPress will be permitted. Use with caution as this may pose security risks.', 'gk-gravityboard' ),
								'default_value' => '0',
							],
						],
					],
					[
						'title'      => esc_html__( 'Checklists', 'gk-gravityboard' ),
						'dependency' => [
							'live'   => true,
							'fields' => [
								[
									'field' => 'enable_card_checklists',
									'value' => '1',
								],
							],
						],
						'fields'     => [
							[
								'name'          => 'sort_completed_checklist_items',
								'label'         => esc_html__( 'Move Completed Items to Bottom', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'Completed checklist items will automatically move below active items, maintaining the order in which they were completed.', 'gk-gravityboard' ),
								'default_value' => '0',
							],
							[
								'name'        => Checklists::CHECKLIST_TEMPLATE_SETTING,
								'label'       => esc_html__( 'Checklist Template', 'gk-gravityboard' ),
								'type'        => 'textarea',
								'class'       => 'medium',
								'description' => esc_html__( 'Enter checklist items to automatically add to new entries. One item per line.', 'gk-gravityboard' ),
								'placeholder' => esc_html__( "Example:\nReview requirements\nComplete design\nTest functionality\nCode review\nDocumentation updated", 'gk-gravityboard' ),
								'default_value' => '',
								'rows'        => 5,
							],
						],
					],
					[
						'title'  => esc_html__( 'Debug Mode', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'          => 'enable_debug_mode',
								'label'         => esc_html__( 'Enable Debug Mode', 'gk-gravityboard' ),
								'type'          => 'toggle',
								'description'   => esc_html__( 'When enabled, detailed debug information will be logged to the browser console. This should only be enabled for troubleshooting and then disabled.', 'gk-gravityboard' ),
								'default_value' => '0',
							],
						],
					],
				],
			],
		];

		return $settings;
	}

	/**
	 * Returns calendar icon for the form settings menu.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_menu_icon() {
		return '<svg width="18" height="18" viewBox="0 0 47 44" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
<path fill-rule="evenodd" clip-rule="evenodd" d="M20.7971 36.6666V38.5C20.7971 41.5376 18.3049 44 15.2691 44H5.52806C2.45215 44 0 41.5376 0 38.5L0 11.0001L0 7.33332L0 5.50006C0 2.46246 2.45215 6.10352e-05 5.52806 6.10352e-05L15.2691 6.10352e-05C18.3049 6.10352e-05 20.7971 2.46246 20.7971 5.50006L20.7971 33.0001V36.6666ZM15.2691 3.66673C16.2691 3.66673 17.1115 4.48745 17.1115 5.50006L17.1115 38.5C17.1115 39.5125 16.2691 40.3334 15.2691 40.3334H5.52806C4.48796 40.3334 3.68563 39.5125 3.68563 38.5L3.68563 5.50006C3.68563 4.48745 4.48796 3.66673 5.52806 3.66673L15.2691 3.66673Z" fill="currentColor"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M26 28.8333L26 27C26 23.9624 28.4922 21.5 31.5281 21.5L41.2691 21.5C44.345 21.5 46.7971 23.9624 46.7971 27L46.7971 26L46.7971 29.6667L46.7971 38.5C46.7971 41.5376 44.345 44 41.2691 44L31.5281 44C28.4922 44 26 41.5376 26 38.5L26 39.4999L26 28.8333ZM31.5281 40.3333C30.528 40.3333 29.6856 39.5126 29.6856 38.5L29.6856 27C29.6856 25.9875 30.528 25.1666 31.5281 25.1666L41.2691 25.1666C42.3092 25.1666 43.1115 25.9875 43.1115 27L43.1115 38.5C43.1115 39.5126 42.3092 40.3333 41.2691 40.3333L31.5281 40.3333Z" fill="currentColor"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M46.7971 10.1667V12C46.7971 15.0376 44.3049 17.5 41.2691 17.5L31.5281 17.5C28.4521 17.5 26 15.0376 26 12V13L26 9.33326V5.5C26 2.4624 28.4521 0 31.5281 0L41.2691 0C44.3049 0 46.7971 2.4624 46.7971 5.5V4.50008V10.1667ZM41.2691 3.66667C42.2691 3.66667 43.1115 4.48739 43.1115 5.5V12C43.1115 13.0125 42.2691 13.8334 41.2691 13.8334L31.5281 13.8334C30.488 13.8334 29.6856 13.0125 29.6856 12V5.5C29.6856 4.48739 30.488 3.66667 31.5281 3.66667L41.2691 3.66667Z" fill="currentColor"/>
</svg>';
	}

	/**
	 * Get the choice based fields
	 *
	 * @since 1.0
	 *
	 * @param array|string $required_types The field types to get. Multiple field types as an array or a single type in a string.
	 * @param array|string $excluded_types The field types to exclude. Multiple field types as an array or a single type in a string.
	 *
	 * @return array The choice based fields
	 */
	private function get_fields_by_type( $required_types = [], $excluded_types = [] ) {
		$generic_map = new Generic_Map( [], new Settings() );
		return $generic_map->get_value_choices( $required_types, $excluded_types );
	}

	/**
	 * Get the edit feed URL
	 *
	 * @since 1.0
	 *
	 * @param int|string $feed_id The ID of the feed (or string placeholder, e.g. '%d').
	 * @param int|string $form_id The ID of the form (or string placeholder, e.g. '%d').
	 *
	 * @return string The edit feed URL.
	 */
	public static function get_edit_feed_url( $feed_id, $form_id = null ) {
		if ( ! $form_id && is_numeric( $feed_id ) ) {
			$form_id = rgars( self::get_instance()->get_feed( $feed_id ), 'form_id' );
		}

		$url = sprintf( 'admin.php?page=gf_edit_forms&view=settings&subview=%s&id=%s&fid=%s', self::get_instance()->get_slug(), $form_id, $feed_id );

		return admin_url( $url );
	}

	/**
	 * Get the view feed URL
	 *
	 * @since 1.0
	 *
	 * @param int|string $feed_id The ID of the feed (or string placeholder, e.g. '%d').
	 *
	 * @return string The view feed URL.
	 */
	public static function get_view_feed_url( $feed_id ) {
		return sprintf( 'admin.php?page=' . Board_View::ADMIN_VIEW_PAGE_SLUG . '&feed_id=%s', $feed_id );
	}

	/**
	 * Get the view boards URL
	 *
	 * @since 1.0
	 *
	 * @return string The view boards URL.
	 */
	public static function get_view_boards_url() {
		return sprintf( 'admin.php?page=' . Boards_Overview::PAGE_ID );
	}

	/**
	 * Get the feed, locally cached.
	 *
	 * @since 1.0
	 *
	 * @param int $feed_id The ID of the feed.
	 *
	 * @return array|false The feed array, or false if the feed does not exist.
	 */
	public function get_feed( $feed_id ) {
		static $feed = [];

		if ( ! isset( $feed[ $feed_id ] ) ) {
			$feed[ $feed_id ] = parent::get_feed( $feed_id );
		}

		return $feed[ $feed_id ];
	}

	/**
	 * Get the form for a feed.
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed array.
	 *
	 * @return array|WP_Error The form data or a WP_Error if the form is not found.
	 */
	public function get_form( $feed ) {
		if ( empty( $feed['id'] ) ) {
			return new WP_Error( 'invalid_feed_id', esc_html__( 'Invalid feed ID.', 'gk-gravityboard' ) );
		}

		$form_id = rgars( $feed, 'form_id', 0 );

		if ( empty( $form_id ) ) {
			return new WP_Error( 'invalid_form_id', esc_html__( 'Invalid form ID.', 'gk-gravityboard' ) );
		}

		if ( is_wp_error( $form_id ) ) {
			return $form_id;
		}

		return Helpers::get_form( $form_id );
	}

	/**
	 * Get the settings for a feed.
	 *
	 * @since 1.0
	 *
	 * @param int $feed_id The ID of the feed.
	 *
	 * @return array The board settings.
	 */
	public function get_settings_for_script( $feed_id ) {

		// Get the feed to access its settings.
		$feed = $this->get_feed( $feed_id );

		if ( empty( $feed ) ) {
			return new WP_Error( 'no_feed', esc_html__( 'No feed found.', 'gk-gravityboard' ) );
		}

		// Remove conditional logic for security; it may expose private business logic.
		unset(
			$feed['feed_condition_conditional_logic_object'],
			$feed['conditional_logic'],
		);

		// Get field types for the mapped fields.
		$field_types = $this->get_field_types( $feed, $feed['form_id'] );

		// Get available users for this board based on role settings.
		$available_users = $this->get_assignable_users( $feed );

		// Create settings object for React app - This is the original $board_settings.
		$board_settings = [
			'feed_name'       => $this->get_board_name( $feed ),
			'mapped_fields'   => $this->get_mapped_fields( $feed ),
			'field_types'     => $field_types,
			'current_feed_id' => $feed_id,
			'form_id'         => $feed['form_id'],
			'admin_url'       => admin_url(),
			'nonce'           => wp_create_nonce( Ajax::NONCE_ACTION ),
			'debug_mode'      => Helpers::is_toggle_enabled( $feed, 'meta/enable_debug_mode' ),
			'rest_namespace'  => Ajax::REST_NAMESPACE,
			'rest_base_url'   => rest_url( Ajax::REST_NAMESPACE ),
			'permissions'     => [
				'can_add_card'                   => $this->user_has_permission( 'add_card', $feed ),
				'can_delete_card'                => $this->user_has_permission( 'delete_card', $feed ),
				'can_update_card'                => $this->user_has_permission( 'edit_card', $feed ),
				'can_modify_lane'                => $this->user_has_permission( 'modify_lane', $feed ),
				'can_view_assignees'             => $this->user_has_permission( 'view_assignees', $feed ),
				'can_assign_users'               => $this->user_has_permission( 'assign_users', $feed ),
				'can_view_entry_notes'           => $this->user_has_permission( 'view_entry_notes', $feed ),
				'can_add_entry_notes'            => $this->user_has_permission( 'add_entry_notes', $feed ),
				'can_manage_entry_notes'         => $this->user_has_permission( 'manage_entry_notes', $feed ),
				'can_view_checklists'            => $this->user_has_permission( 'view_checklists', $feed ),
				'can_edit_checklists'            => $this->user_has_permission( 'edit_checklists', $feed ),
				'can_view_attachments'           => $this->user_has_permission( 'view_attachments', $feed ),
				'can_add_attachments'            => $this->user_has_permission( 'add_attachments', $feed ),
				'can_delete_attachments'         => $this->user_has_permission( 'delete_attachments', $feed ),
				'can_view_gravity_forms_entries' => GFCommon::current_user_can_any( 'gravityforms_view_entries' ),
			],
			'features'        => [
				'enable_assignees'        => Helpers::is_toggle_enabled( $feed, 'meta/enable_assignees' ),
				'enable_entry_notes'      => Helpers::is_toggle_enabled( $feed, 'meta/enable_entry_notes' ),
				'enable_card_checklists'  => Helpers::is_toggle_enabled( $feed, 'meta/enable_card_checklists' ),
				'enable_card_attachments' => Helpers::is_toggle_enabled( $feed, 'meta/enable_card_attachments' ),
				'enable_markdown'         => Helpers::is_toggle_enabled( $feed, 'meta/enable_markdown', true ),
			],
			'settings'        => [
				'checklists' => [
					'sort_completed_items' => Helpers::is_toggle_enabled( $feed, 'meta/sort_completed_checklist_items' ),
				],
				'auto_focus' => is_admin(), // Auto-focus cards by default only in admin area.
			],
			'users'           => $available_users,
			'lane_colors'     => rgars( $feed, 'meta/lane_colors', [] ),
			'use_full_height' => 0,
		];

		/**
		 * Filter the board settings.
		 *
		 * @since 1.0
		 *
		 * @param array $board_settings The board settings.
		 * @param int $feed_id The ID of the feed.
		 *
		 * @return array The board settings.
		 */
		$board_settings = apply_filters( 'gk/gravityboard/board-settings', $board_settings, $feed_id );

		return $board_settings;
	}

	/**
	 * Get the feed name
	 *
	 * TODO: Once we require Gravity Forms 2.9.9, we can remove this method and use $this->get_feed_name() instead.
	 *
	 * @since 1.0
	 *
	 * @param array  $feed The feed array.
	 * @param string $key The key to get the feed name from.
	 *
	 * @return string The feed name.
	 */
	public function get_feed_name( $feed, $key = '' ) {
		if ( method_exists( GFAPI::class, 'get_feed_name' ) ) {
			return GFAPI::get_feed_name( $feed, $key );
		}

		if ( empty( $key ) ) {
			$key = ! empty( $feed['meta']['feedName'] ) ? 'feedName' : 'feed_name';
		}

		return rgars( $feed, 'meta/' . $key, '' );
	}

	/**
	 * Get the board name for a feed
	 *
	 * @since 1.0
	 *
	 * @param array      $feed The feed array.
	 * @param array|null $form The form array.
	 *
	 * @return string The board name.
	 */
	public function get_board_name( $feed, $form = null ) {
		$feed_name = $this->get_feed_name( $feed );
		$form      = Helpers::get_form( $feed['form_id'] );

		if ( empty( $feed_name ) && ! is_wp_error( $form ) && isset( $feed['id'] ) ) {
			$feed_name = sprintf(
				// translators: %1$s is the form title, %2$d is the board ID.
				esc_html__( '%1$s Board #%2$d', 'gk-gravityboard' ),
				$form['title'],
				$feed['id']
			);
		}

		return $feed_name;
	}

	/**
	 * Get field types for the mapped fields
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed array.
	 * @param int   $form_id The form ID.
	 *
	 * @return array The field types and configurations
	 */
	private function get_field_types( $feed, $form_id ) {
		$field_types   = [];
		$mapped_fields = $this->get_mapped_fields( $feed );

		foreach ( $mapped_fields as $key => $field_id ) {
			if ( ! empty( $field_id ) ) {
				$field = GFAPI::get_field( $form_id, $field_id );

				if ( $field ) {
					// Create field configuration with type.
					$field_config = [
						'type' => $field->type,
					];

					// Add required status from Gravity Forms field.
					if ( ! empty( $field->isRequired ) ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
						$field_config['required'] = true;
					}

					// Add choices for fields that have options.
					if ( in_array( $field->type, [ 'radio', 'select', 'checkbox', 'multiselect' ], true ) && ! empty( $field->choices ) ) {
						$field_config['choices'] = $field->choices;
					}

					// Add placeholder if available.
					if ( ! empty( $field->placeholder ) ) {
						$field_config['placeholder'] = $field->placeholder;
					}

					// Add maxLength for text and textarea fields that have character limits.
					if ( ! empty( $field->maxLength ) && is_numeric( $field->maxLength ) ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
						$field_config['maxLength'] = absint( $field->maxLength ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
					}

					// Store the field configuration.
					$field_types[ $key ] = $field_config;
				} else {
					// Handle entry properties and entry meta that are not form fields.
					$field_config = $this->get_entry_property_field_type( $field_id, $form_id );
					if ( $field_config ) {
						$field_types[ $key ] = $field_config;
					}
				}
			}
		}

		return $field_types;
	}

	/**
	 * Get field type configuration for entry properties and entry meta
	 *
	 * @since 1.0
	 *
	 * @param string $field_id The field ID (could be entry property or meta key).
	 * @param int    $form_id The form ID.
	 *
	 * @return array|null The field configuration or null if not found.
	 */
	private function get_entry_property_field_type( $field_id, $form_id ) {
		// Handle standard entry properties.
		$entry_properties = [
			'id'             => [ 'type' => 'number' ],
			'date_created'   => [ 'type' => 'date' ],
			'date_updated'   => [ 'type' => 'date' ],
			'ip'             => [ 'type' => 'text' ],
			'source_url'     => [ 'type' => 'url' ],
			'form_title'     => [ 'type' => 'text' ],
			'payment_status' => [ 'type' => 'text' ],
			'payment_date'   => [ 'type' => 'date' ],
			'payment_amount' => [ 'type' => 'text' ],
			'transaction_id' => [ 'type' => 'text' ],
			'created_by'     => [ 'type' => 'text' ],
		];

		if ( isset( $entry_properties[ $field_id ] ) ) {
			return $entry_properties[ $field_id ];
		}

		return null;
	}

	/**
	 * Get the mapped fields for the feed
	 *
	 * @since 1.0
	 * @internal Do not rely on this method. It's used internally by the plugin.
	 *
	 * @param array $feed The feed array.
	 *
	 * @return array The mapped fields
	 */
	public function get_mapped_fields( $feed ) {
		$mapped_fields = [];
		foreach ( $feed['meta'] as $key => $value ) {
			if ( strpos( $key, 'card_fields_' ) === 0 ) {
				$mapped_fields[ str_replace( 'card_fields_', '', $key ) ] = $value;
			}
		}

		return $mapped_fields;
	}

	/**
	 * Get a specific field from the feed by key.
	 *
	 * @since 1.0
	 *
	 * @param array  $feed The feed array.
	 * @param string $key The field key to retrieve.
	 *
	 * @return string The field value.
	 */
	public static function get_feed_field_by_key( $feed, $key ) {
		return rgars( $feed, 'meta/card_fields_' . $key, '' );
	}

	/**
	 * Get the entry meta position key for a given feed ID.
	 *
	 * @param int $feed_id The ID of the feed.
	 *
	 * @return string The entry meta position key.
	 */
	public static function get_entry_meta_position_key( $feed_id ) {
		return sprintf( self::ENTRY_META_POSITION_KEY, $feed_id );
	}

	/**
	 * Get a feed setting value for an entry.
	 *
	 * @since 1.0
	 *
	 * @param array  $feed The feed array.
	 * @param string $key The setting key.
	 * @param array  $entry The entry array.
	 * @param string $default_value The default value to return if not set.
	 *
	 * @return string The setting value.
	 */
	public function get_feed_setting( $feed, $key, $entry, $default_value = '' ) {
		$field_id = self::get_feed_field_by_key( $feed, $key );

		return html_entity_decode( rgar( $entry, $field_id, $default_value ), ENT_QUOTES | ENT_HTML5, 'UTF-8' );
	}

	/**
	 * Set the title for the feed settings
	 *
	 * @return string
	 */
	public function feed_settings_title() {
		return esc_html__( 'GravityBoard Settings', 'gk-gravityboard' );
	}

	/**
	 * Set the title for the feed list
	 *
	 * @return string
	 */
	public function feed_list_title() {
		return esc_html__( 'GravityBoard Boards', 'gk-gravityboard' );
	}

	/**
	 * Check if the current user can create a feed
	 *
	 * @return bool
	 */
	public function can_create_feed() {
		return GFCommon::current_user_can_any( 'gravityforms_edit_forms' );
	}

	/**
	 * Set the columns for the feed list
	 *
	 * @return array
	 */
	public function feed_list_columns() {
		return array(
			'feedName'   => esc_html__( 'Name', 'gk-gravityboard' ),
			'shortcode'  => sprintf(
				'%s <small>(%s)</small>',
				esc_html__( 'Shortcode', 'gk-gravityboard' ),
				esc_html__( 'Click to copy', 'gk-gravityboard' )
			),
			'view_board' => esc_html__( 'View Board', 'gk-gravityboard' ),
		);
	}

	/**
	 * Custom column for View Board action in the feed list.
	 *
	 * @param array $feed The feed being displayed.
	 *
	 * @return string
	 */
	public function get_column_value_view_board( $feed ) {
		$view_url = self::get_view_feed_url( $feed['id'] );

		return sprintf(
			'<a href="%s" class="button" title="%s"><i class="gform-button__icon gform-common-icon gform-common-icon--eye"></i> %s</a>',
			esc_url( $view_url ),
			esc_attr__( 'View this board in the admin', 'gk-gravityboard' ),
			esc_html__( 'View Board', 'gk-gravityboard' )
		);
	}

	/**
	 * Custom column for the Feed ID in the feed list
	 *
	 * @param array $feed The feed being displayed.
	 * @return string
	 */
	public function get_column_value_shortcode( $feed ) {
		return '<div class="gk-gravityboard-shortcode">
            <input title="' . esc_html__( 'Click to copy', 'gk-gravityboard' ) . '" class="code shortcode widefat" readonly value="' . esc_attr( '[gravityboard id=\'' . $feed['id'] . '\']' ) . '" />
            <div class="copied">' . esc_html__( 'Copied!', 'gk-gravityboard' ) . '</div>
        </div>';
	}

	/**
	 * Custom column for the Feed Name in the feed list.
	 *
	 * @param array $feed The feed being displayed.
	 *
	 * @return string
	 */
	public function get_column_value_feedName( $feed ) {
		$feed_name = rgars( $feed, 'meta/feed_name', '' );

		// If feed_name is empty, create a descriptive default name.
		if ( ! empty( $feed_name ) ) {
			return $feed_name;
		}

		$form       = GFAPI::get_form( $feed['form_id'] );
		$form_title = $form ? $form['title'] : esc_html__( 'Unknown Form', 'gk-gravityboard' );
		$feed_name  = sprintf(
			// translators: %1$s is Form Title, %2$d is Feed ID.
			esc_html__( '%1$s Board #%2$d', 'gk-gravityboard' ),
			$form_title,
			$feed['id']
		);

		return $feed_name;
	}

	/**
	 * Check if the current user has permission to perform an action.
	 *
	 * @since 1.0
	 *
	 * @param string     $action The action to check (add_card, edit_card, delete_card, add_lane, edit_lane, delete_lane).
	 * @param array|bool $feed The feed containing the settings.
	 *
	 * @return bool Whether the user has permission.
	 */
	public function user_has_permission( $action, $feed ) {
		return Helpers::user_has_permission( $action, $feed );
	}

	/**
	 * Prefixes field name for use in input elements according to the GF version.
	 *
	 * @since $ver$
	 *
	 * @param string $name Field name.
	 *
	 * @return string Prefixed field name.
	 */
	protected function prefix_field_name( string $name = '' ): string {
		$prefix = $this->is_gravityforms_supported( '2.5-beta' ) ? '_gform_setting_' : '_gaddon_setting_';

		return $prefix . $name;
	}

	/**
	 * Returns the conditional logic.
	 *
	 * @since $ver$
	 *
	 * @param array $feed_settings The feed settings to use.
	 *
	 * @return "null"|array The conditional logic.
	 */
	public static function get_conditional_logic( array $feed_settings ) {
		$conditional_logic = rgar( $feed_settings, 'conditional_logic', 'null' );
		if ( 'null' === $conditional_logic ) {
			$conditional_logic = rgars(
				$feed_settings,
				'feed_condition_conditional_logic_object/conditionalLogic',
				'null'
			);

			if ( ! empty( $conditional_logic ) && 'null' !== $conditional_logic ) {
				$conditional_logic = QueryFilters::create()->convert_gf_conditional_logic( $conditional_logic );
			}
		}

		if ( is_array( $conditional_logic ) ) {
			$conditional_logic['version'] ??= 2;
		}

		return $conditional_logic;
	}
	/**
	 * Enqueues the scripts for query filters.
	 *
	 * @return void
	 */
	protected function enqueue_query_filters_js(): void {
		$feed_settings = $this->get_current_settings();
		$query_filters = QueryFilters::create();

		if ( $this->is_feed_edit_page() ) {
			$form = $this->get_current_form();
			if ( $form ) {
				try {
					$query_filters = $query_filters->with_form( $form );
				} catch ( Exception $exception ) {
					$this->log_error( $exception->getMessage() );

					return;
				}
			}

			$conditional_logic = static::get_conditional_logic( $feed_settings );
			$query_filters->enqueue_scripts(
				[
					'input_element_name' => $this->prefix_field_name( 'conditional_logic' ),
					'conditions'         => $conditional_logic,
				]
			);
		}
	}
	/**
	 * Define the scripts for the GravityBoard add-on.
	 *
	 * @since 1.0
	 *
	 * @return array The script definitions.
	 */
	public function scripts(): array {
		$this->enqueue_query_filters_js();

		$feed_settings_file_path          = 'assets/js/admin-feed-settings.js';
		$entry_detail_mentions_file_path  = 'assets/js/entry-detail-mentions.js';
		$admin_feed_list_file_path        = 'assets/js/admin-feed-list.js';
		$entry_detail_checklist_file_path = 'assets/js/entry-detail-checklist.js';

		$scripts = [
			[
				'handle'    => 'gravityboard_admin_feed_settings',
				'src'       => plugins_url( $feed_settings_file_path, GRAVITYBOARD_FILE ),
				'deps'      => [ 'jquery', 'wp-a11y', 'clipboard', 'wp-color-picker' ],
				'version'   => $this->get_asset_version( $feed_settings_file_path ),
				'in_footer' => true,
				'enqueue'   => [
					[ $this, 'is_feed_edit_page' ],
				],
				'strings'   => [
					'image_selected'        => esc_html__( 'Background image selected.', 'gk-gravityboard' ),
					'image_removed'         => esc_html__( 'Background image removed.', 'gk-gravityboard' ),
					'unsaved_changes'       => esc_html__( 'You have unsaved changes. Are you sure you want to leave this page?', 'gk-gravityboard' ),
					'field_validation'      => $this->get_field_validation_config(),
					'disabled_role_message' => esc_html__( '🚫 This functionality is disabled. Select roles to enable.', 'gk-gravityboard' ),
				],
			],
			[
				'handle'    => 'gravityboard_entry_detail_mentions',
				'src'       => plugins_url( $entry_detail_mentions_file_path, GRAVITYBOARD_FILE ),
				'deps'      => [],
				'version'   => $this->get_asset_version( $entry_detail_mentions_file_path ),
				'in_footer' => true,
				'enqueue'   => [
					[ 'admin_page' => [ 'entry_view' ] ],
				],
			],
			[
				'handle'    => 'gravityboard_entry_detail_checklist',
				'src'       => plugins_url( $entry_detail_checklist_file_path, GRAVITYBOARD_FILE ),
				'deps'      => [ 'jquery' ],
				'version'   => $this->get_asset_version( $entry_detail_checklist_file_path ),
				'in_footer' => true,
				'enqueue'   => [
					[ 'admin_page' => [ 'entry_view' ] ],
				],
				'strings'   => [
					'rest_url'                   => esc_url_raw( rest_url( 'gravityboard/v1/' ) ),
					'rest_nonce'                 => wp_create_nonce( 'wp_rest' ),
					'updatingChecklistItem'      => esc_html__( 'Updating checklist item…', 'gk-gravityboard' ),
					'checklistItemUpdated'       => esc_html__( 'Checklist item updated successfully.', 'gk-gravityboard' ),
					'addingChecklistItem'        => esc_html__( 'Adding checklist item…', 'gk-gravityboard' ),
					'checklistItemAdded'         => esc_html__( 'Checklist item added successfully.', 'gk-gravityboard' ),
					'errorUpdatingChecklistItem' => esc_html__( 'Error updating checklist item.', 'gk-gravityboard' ),
					'errorAddingChecklistItem'   => esc_html__( 'Error adding checklist item.', 'gk-gravityboard' ),
					// translators: Do not translate {completed}, {total}, {percentage}. These are placeholders for the completed items, total items, and percentage of completion.
					'progressText'               => esc_html__( '{completed} of {total} completed ({percentage}%)', 'gk-gravityboard' ),
				],
			],
			[
				'handle'    => 'gravityboard-admin-feed-list',
				'src'       => plugins_url( $admin_feed_list_file_path, GRAVITYBOARD_FILE ),
				'deps'      => [ 'jquery', 'wp-a11y', 'clipboard' ],
				'version'   => $this->get_asset_version( $admin_feed_list_file_path ),
				'in_footer' => true,
				'enqueue'   => [
					[ $this, 'is_feed_list_page' ],
				],
			],
		];

		return array_merge( parent::scripts(), $scripts );
	}

	/**
	 * Get the version of an asset file.
	 *
	 * @since 1.0
	 *
	 * @param string $file_path The path to the asset file.
	 *
	 * @return string The version of the asset file.
	 */
	private function get_asset_version( $file_path ) {
		return file_exists( plugin_dir_path( GRAVITYBOARD_FILE ) . $file_path ) ? filemtime( plugin_dir_path( GRAVITYBOARD_FILE ) . $file_path ) : $this->_version;
	}

	/**
	 * Define the styles for the GravityBoard add-on.
	 *
	 * @since 1.0
	 *
	 * @return array The style definitions.
	 */
	public function styles() {

		$entry_detail_mentions_file_path = 'assets/css/entry-detail-mentions.css';
		$admin_feed_list_css_path        = 'assets/css/admin-feed-list.css';
		$admin_feed_settings_css_path    = 'assets/css/admin-feed-settings.css';
		$entry_detail_checklist_css_path = 'assets/css/entry-detail-checklist.css';

		$plugin_styles = [
			[
				'handle'  => 'gravityboard_entry_detail_mentions',
				'src'     => plugins_url( $entry_detail_mentions_file_path, GRAVITYBOARD_FILE ),
				'version' => $this->get_asset_version( $entry_detail_mentions_file_path ),
				'enqueue' => [
					[ 'admin_page' => [ 'entry_view' ] ],
				],
			],
			[
				'handle'  => 'gravityboard_entry_detail_checklist',
				'src'     => plugins_url( $entry_detail_checklist_css_path, GRAVITYBOARD_FILE ),
				'version' => $this->get_asset_version( $entry_detail_checklist_css_path ),
				'enqueue' => [
					[ 'admin_page' => [ 'entry_view' ] ],
				],
			],
			[
				'handle'  => 'gravityboard-admin-feed-list',
				'src'     => plugins_url( $admin_feed_list_css_path, GRAVITYBOARD_FILE ),
				'version' => $this->get_asset_version( $admin_feed_list_css_path ),
				'enqueue' => [
					[ $this, 'is_feed_list_page' ],
				],
			],
			[
				'handle'  => 'gravityboard-admin-feed-settings',
				'src'     => plugins_url( $admin_feed_settings_css_path, GRAVITYBOARD_FILE ),
				'version' => $this->get_asset_version( $admin_feed_settings_css_path ),
				'enqueue' => [
					[ $this, 'is_feed_edit_page' ],
				],
			],
			[
				'handle'  => 'wp-color-picker',
				'enqueue' => [
					[ $this, 'is_feed_edit_page' ],
				],
			],
		];

		QueryFilters::enqueue_styles();

		return array_merge( parent::styles(), $plugin_styles );
	}

	/**
	 * Process feed settings before they're saved to ensure proper data format.
	 *
	 * @since 1.0
	 *
	 * @param mixed $feed_settings The feed settings to process.
	 *
	 * @return mixed The processed feed settings.
	 */
	public function process_feed_settings( $feed_settings ) {
		// Make sure assignee_roles is always an array.
		if ( isset( $feed_settings['assignee_roles'] ) && ! is_array( $feed_settings['assignee_roles'] ) ) {
			$feed_settings['assignee_roles'] = empty( $feed_settings['assignee_roles'] ) ? array() : array( $feed_settings['assignee_roles'] );
		}

		foreach ( $feed_settings['lane_colors'] as $lane_name => $lane_color ) {
			$sanitized_lane_color = sanitize_hex_color( $lane_color );

			if ( '' === $sanitized_lane_color ) {
				$sanitized_lane_color = '#f1f2f4';
			}

			$feed_settings['lane_colors'][ $lane_name ] = $sanitized_lane_color;
		}

		return $feed_settings;
	}

	/**
	 * Override the save_feed_settings method to process settings before saving.
	 *
	 * @since 1.0
	 *
	 * @param mixed $feed_id The ID of the feed being saved.
	 * @param int   $form_id The ID of the form to which the feed belongs.
	 * @param mixed $settings The settings to be saved.
	 * @return int|bool The ID of the saved feed or false on failure.
	 */
	public function save_feed_settings( $feed_id, $form_id, $settings ) {

		// Process settings before saving.
		$settings = $this->process_feed_settings( $settings );

		// Call parent method to save.
		return parent::save_feed_settings( $feed_id, $form_id, $settings );
	}

	/**
	 * Validates settings before they're saved.
	 *
	 * @since 1.0
	 *
	 * @param array $fields The fields to be validated.
	 * @param array $settings The settings to be validated.
	 * @return array The validated settings.
	 */
	public function validate_settings( $fields, $settings ) {
		// Return settings with validation applied as needed.
		$is_valid = parent::validate_settings( $fields, $settings );

		// Ensure assignee_roles is an array.
		if ( isset( $settings['assignee_roles'] ) && ! is_array( $settings['assignee_roles'] ) && ! empty( $settings['assignee_roles'] ) ) {
			$settings['assignee_roles'] = array( $settings['assignee_roles'] );
		}

		return $is_valid;
	}

	/**
	 * Validate CSS dimension field.
	 *
	 * @since 1.0
	 *
	 * @param array $field The field being validated.
	 * @param mixed $field_setting The field value.
	 *
	 * @return void
	 */
	public function validate_css_dimension_field( $field, $field_setting ) {
		// Skip validation if field is empty and not required.
		if ( empty( $field_setting ) && ! rgar( $field, 'required' ) ) {
			return;
		}

		// Check if required field is empty.
		if ( rgar( $field, 'required' ) && empty( $field_setting ) ) {
			$this->set_field_error( $field, rgar( $field, 'error_message', esc_html__( 'This field is required.', 'gk-gravityboard' ) ) );
			return;
		}

		$value             = trim( $field_setting );
		$validation_config = $this->get_field_validation_config()['board_min_height'];
		$pattern           = rgar( $field, 'validation_pattern', $validation_config['pattern'] );

		// Validate CSS dimension format using the pattern from field definition.
		if ( ! preg_match( '/' . $pattern . '/i', $value ) ) {
			$error_message = rgar( $field, 'validation_message', $validation_config['message'] );
			$this->set_field_error( $field, $error_message );
		}
	}

	/**
	 * Get a feed setting value, ensuring it's of the correct data type.
	 *
	 * @since 1.0
	 *
	 * @param array  $feed The feed object.
	 * @param string $setting_name The setting name.
	 * @param mixed  $default_value Optional. The default value to return if the setting doesn't exist.
	 * @param bool   $force_array Optional. If true, forces the return value to be an array.
	 *
	 * @return mixed The setting value.
	 */
	public function get_feed_setting_value( $feed, $setting_name, $default_value = '', $force_array = false ) {
		$value = rgars( $feed, 'meta/' . $setting_name, $default_value );

		// Ensure certain settings are always arrays.
		if ( $force_array || 'assignee_roles' === $setting_name ) {
			if ( ! is_array( $value ) ) {
				$value = empty( $value ) ? [] : [ $value ];
			}
		}

		return $value;
	}

	/**
	 * Returns the ID of the lane field from a feed.
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed.
	 *
	 * @return int The ID of the lane field. 0 if not found.
	 */
	public function get_lane_field_id( $feed ) {
		return (int) rgars( $feed, 'meta/card_fields_lane_field', 0 );
	}

	/**
	 * Returns the lane field from a feed.
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed.
	 *
	 * @return GF_Field|null The lane field or null if not found.
	 */
	public function get_lane_field( $feed ) {

		$lane_field_id = $this->get_lane_field_id( $feed );

		if ( empty( $lane_field_id ) ) {
			return null;
		}

		$form  = GFAPI::get_form( $feed['form_id'] );
		$field = GFAPI::get_field( $form, $lane_field_id );

		return $field;
	}

	/**
	 * Generates the field definitions for lane background color pickers.
	 *
	 * @since 1.0
	 * @return array An array of field definitions for Gravity Forms settings.
	 */
	private function get_lane_background_color_fields() {
		$form_id = $this->get_current_form();

		$current_feed_object = $this->get_current_feed();
		$saved_lane_colors   = $this->get_feed_setting_value( $current_feed_object, 'lane_colors', [] );

		if ( ! $form_id ) {
			return [
				[
					'type' => 'html',
					'name' => 'lane_colors_no_form_id',
					'html' => '<p>' . esc_html__( 'Form ID could not be determined. Please ensure you are editing a feed associated with a form.', 'gk-gravityboard' ) . '</p>',
				],
			];
		}

		$lane_field_id = $this->get_feed_setting_value( $current_feed_object, 'card_fields_lane_field', 0 );

		if ( empty( $lane_field_id ) ) {
			return [
				[
					'type' => 'html',
					'name' => 'lane_colors_no_lane_field_selected',
					'html' => '<p>' . esc_html__( 'To configure lane colors, please first select and save a "Lane Field" in the "Field Mapping" section above.', 'gk-gravityboard' ) . '</p>',
				],
			];
		}

		$lane_gf_field = GFAPI::get_field( $form_id, $lane_field_id );

		if ( ! $lane_gf_field || ! property_exists( $lane_gf_field, 'choices' ) || empty( $lane_gf_field->choices ) ) {
			return [
				[
					'type' => 'html',
					'name' => 'lane_colors_lane_field_no_choices',
					'html' => '<p>' . esc_html__( 'The selected "Lane Field" either does not exist or has no choices defined. Please check the form and field settings.', 'gk-gravityboard' ) . '</p>',
				],
			];
		}

		$choices = $lane_gf_field->choices;

		$fields = [];
		foreach ( $choices as $choice ) {
			if ( ! is_array( $choice ) || ! isset( $choice['value'] ) || ! isset( $choice['text'] ) ) {
				continue;
			}

			$field_name = sprintf( 'lane_colors[%s]', esc_attr( $choice['value'] ) );

			$fields[] = [
				// translators: %s: Lane name.
				'label'         => sprintf( esc_html__( 'Color for Lane: %s', 'gk-gravityboard' ), esc_html( $choice['text'] ) ),
				'type'          => 'text',
				'name'          => $field_name,
				'class'         => 'small gf-color-picker',
				'value'         => $saved_lane_colors[ $choice['value'] ] ?? '',
				'default_value' => '#f1f2f4',
			];
		}

		if ( empty( $fields ) ) {
			return [
				[
					'type' => 'html',
					'name' => 'lane_colors_no_valid_choices_found',
					'html' => '<p>' . esc_html__( 'No valid choices were found for the selected "Lane Field" to configure colors.', 'gk-gravityboard' ) . '</p>',
				],
			];
		}

		return $fields;
	}

	/**
	 * Get the color for a lane.
	 *
	 * @param array  $feed  The feed data.
	 * @param string $lane_value The value of the lane.
	 * @return string|null The color hex code or null if not set.
	 */
	public function get_lane_color( array $feed, string $lane_value ): ?string {
		$lane_colors = rgars( $feed, 'meta/lane_colors', [] );
		return rgar( $lane_colors, $lane_value, '#f1f2f4' );
	}

	/**
	 * Returns the short code for the current feed.
	 *
	 * @since 1.9.0
	 *
	 * @param array $feed The feed object.
	 *
	 * @return string The short code.
	 */
	protected function get_shortcode_for_feed( array $feed = [] ): string {
		if ( ! $feed ) {
			$feed = $this->get_current_feed();

			if ( ! is_array( $feed ) ) {
				return '';
			}

			// Make sure we have the posted values as well.
			$feed['meta'] = $this->get_current_settings();
		}

		$atts = [ sprintf( '%s="%d"', 'id', rgar( $feed, 'id' ) ) ];

		return sprintf( '[%s %s]', 'gravityboard', implode( ' ', $atts ) );
	}

	/**
	 * Get field validation configuration.
	 *
	 * @since 1.0
	 *
	 * @return array Field validation configuration.
	 */
	private function get_field_validation_config() {
		return [
			'board_min_height' => [
				'pattern' => '^(?:-?\d*\.?\d+(?:%|px|em|rem|ex|ch|lh|rlh|vh|vw|vmin|vmax|vb|vi|svh|svw|lvh|lvw|dvh|dvw|cm|mm|in|pt|pc|q)|0|auto|max-content|min-content|fit-content|inherit|initial|unset|revert|revert-layer)$',
				'message' => esc_html__( 'Please enter a valid CSS dimension (e.g., 400px, 50vh, 100%, auto, max-content)', 'gk-gravityboard' ),
			],
		];
	}

	/**
	 * Validate multi-select permission field.
	 *
	 * Ensures at least one permission is selected and prevents conflicting selections.
	 *
	 * @since 1.0
	 *
	 * @param array $field The field configuration.
	 * @param mixed $value The field value to validate.
	 * @return true|WP_Error True if valid, WP_Error if invalid.
	 */
	public function validate_multi_select_permission( $field, $value ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundBeforeLastUsed
		// Allow empty values - they mean "no access" (disabled).
		if ( empty( $value ) ) {
			return true;
		}

		// Ensure value is array.
		$values = is_array( $value ) ? $value : [ $value ];

		// Check for conflicting selections.
		if ( in_array( 'enabled', $values, true ) && count( $values ) > 1 ) {
			return new WP_Error( 'conflicting_permission', esc_html__( 'Cannot select "Everyone (Including Logged-Out Users)" with specific roles.', 'gk-gravityboard' ) );
		}

		return true;
	}

	/**
	 * Migrate single-value permissions to array format for multi-select support.
	 *
	 * This method converts existing single-value permission settings to array format
	 * while maintaining backward compatibility.
	 *
	 * @since 1.0
	 *
	 * @return int Number of feeds migrated.
	 */
	public function migrate_single_to_multi_permissions() {
		$feeds = $this->get_feeds();
		$migrated_count = 0;

		$permission_keys = [
			'view_board_role', 'card_add_role', 'card_edit_role',
			'card_delete_role', 'lane_modify_role', 'view_assignees_role',
			'assign_users_role', 'entry_note_view_role', 'entry_note_add_role',
			'entry_note_manage_role', 'view_checklists_role', 'edit_checklists_role',
			'view_attachments_role', 'add_attachments_role', 'delete_attachments_role'
		];

		foreach ( $feeds as $feed ) {
			$updated = false;

			foreach ( $permission_keys as $key ) {
				$value = rgars( $feed, "meta/{$key}" );
				if ( ! empty( $value ) && is_string( $value ) ) {
					$feed['meta'][ $key ] = [ $value ];
					$updated = true;
				}
			}

			if ( $updated ) {
				$this->update_feed_meta( $feed['id'], $feed['meta'] );
				$migrated_count++;
			}
		}

		// Set migration flag.
		update_option( 'gravityboard_permissions_migrated', true );
		update_option( 'gravityboard_permissions_migration_date', current_time( 'mysql' ) );

		return $migrated_count;
	}

	/**
	 * Check if permissions migration has been completed.
	 *
	 * @since 1.0
	 *
	 * @return bool True if migration is complete.
	 */
	public function is_permissions_migration_complete() {
		return (bool) get_option( 'gravityboard_permissions_migrated', false );
	}

	/**
	 * Rollback permissions migration (convert arrays back to single values).
	 *
	 * This method provides a way to rollback the migration if needed.
	 * It converts the first value in each permission array back to a single value.
	 *
	 * @since 1.0
	 *
	 * @return int Number of feeds rolled back.
	 */
	public function rollback_permissions_migration() {
		$feeds = $this->get_feeds();
		$rollback_count = 0;

		$permission_keys = [
			'view_board_role', 'card_add_role', 'card_edit_role',
			'card_delete_role', 'lane_modify_role', 'view_assignees_role',
			'assign_users_role', 'entry_note_view_role', 'entry_note_add_role',
			'entry_note_manage_role', 'view_checklists_role', 'edit_checklists_role',
			'view_attachments_role', 'add_attachments_role', 'delete_attachments_role'
		];

		foreach ( $feeds as $feed ) {
			$updated = false;

			foreach ( $permission_keys as $key ) {
				$value = rgars( $feed, "meta/{$key}" );
				if ( is_array( $value ) && ! empty( $value ) ) {
					// Take the first value in the array as the single value.
					$feed['meta'][ $key ] = $value[0];
					$updated = true;
				}
			}

			if ( $updated ) {
				$this->update_feed_meta( $feed['id'], $feed['meta'] );
				$rollback_count++;
			}
		}

		// Remove migration flags.
		delete_option( 'gravityboard_permissions_migrated' );
		delete_option( 'gravityboard_permissions_migration_date' );

		return $rollback_count;
	}
}
