<?php
/**
 * GravityBoard Helpers
 *
 * @package GravityKit
 * @subpackage GravityBoard
 * @since 1.0
 */

namespace GravityKit\GravityBoard;

use GFCommon;
use GravityKit\GravityBoard\Assignees\Assignees;
use GFFormsModel;
use WP_Error;

class Helpers {

	const MENTION_REGEX = '/@\[(.+?)\]\(user:(\d+)\)/';

	/**
	 * Normalize permission value to array format for backward compatibility.
	 *
	 * @since TODO
	 *
	 * @param mixed $value Single value or array of permission values.
	 *
	 * @return array Normalized array of permission values.
	 */
	private static function normalize_permission_value( $value ) {
		if ( is_string( $value ) ) {
			// Handle legacy 'disabled' string value.
			if ( $value === 'disabled' ) {
				return [];
			}
			return [ $value ]; // Convert single to array
		}

		if ( is_array( $value ) ) {
			// Filter out 'disabled' from arrays for backward compatibility.
			// If empty, permissions are disabled.
			return array_filter( $value, function( $role ) {
				return $role !== 'disabled';
			});
		}

		return [];
	}

	/**
	 * Check if the current user has permission to perform an action.
	 * Now supports multi-select permissions while maintaining backward compatibility.
	 *
	 * @since 1.0
	 *
	 * @param string     $action The action to check (add_card, edit_card, delete_card, add_lane, edit_lane, delete_lane).
	 * @param array|bool $feed The feed containing the settings.
	 * @param int|null   $user_id The user ID to check permissions for. If null, the current user will be used.
	 *
	 * @return bool Whether the user has permission.
	 */
	public static function user_has_permission( $action, $feed, $user_id = null ) {
		if ( empty( $feed ) || ! is_array( $feed ) || empty( $feed['is_active'] ) ) {
			return false;
		}

		// Get the setting key for this action's role.
		$role_setting_key = self::get_role_setting_key_for_action( $action );

		if ( empty( $role_setting_key ) ) {
			return false;
		}

		// Get selected roles (now supports arrays).
		$selected_roles = rgars( $feed, "meta/{$role_setting_key}", [] );
		$selected_roles = self::normalize_permission_value( $selected_roles );

		// Check for global enabled state.
		if ( in_array( 'enabled', $selected_roles, true ) ) {
			return true;
		}

		$user_id = is_null( $user_id ) ? get_current_user_id() : (int) $user_id;

		// Site admin or full GF access always has permission.
		if ( user_can( $user_id, 'manage_options' ) || user_can( $user_id, 'gform_full_access' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			return true;
		}

		// If no roles are selected, access is disabled.
		if ( empty( $selected_roles ) ) {
			return false;
		}

		// Check each role/capability.
		foreach ( $selected_roles as $role ) {
			if ( $role === 'enabled' ) {
				continue; // Already handled above.
			}

			if ( 0 === strpos( $role, 'gravityforms_' ) || 0 === strpos( $role, 'gform_' ) ) {
				if ( user_can( $user_id, $role ) || user_can( $user_id, 'gform_full_access' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
					return true;
				}
			} else {
				if ( user_can( $user_id, $role ) ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Get the settings key for the role associated with an action.
	 *
	 * @param string $action The action to check.
	 * @return string The settings key.
	 */
	public static function get_role_setting_key_for_action( $action ) {
		switch ( $action ) {
			case 'fetch_entries':
			case 'view_board':
				return 'view_board_role';
			case 'add_card':
				return 'card_add_role';
			case 'update_card':
			case 'edit_card':
			case 'move_card':
				return 'card_edit_role';
			case 'delete_card':
				return 'card_delete_role';
			case 'add_lane':
			case 'move_lane':
			case 'modify_lane':
			case 'update_lane':
			case 'delete_lane':
				return 'lane_modify_role';
			case 'fetch_entry_notes':
			case 'view_entry_notes':
				return 'entry_note_view_role';
			case 'add_entry_note':
			case 'add_entry_notes':
				return 'entry_note_add_role';
			case 'assign_users':
				return 'assign_users_role';
			case 'view_assignees':
				return 'view_assignees_role';
			case 'edit_entry_note':
			case 'delete_entry_note':
			case 'manage_entry_notes':
				return 'entry_note_manage_role';
			case 'view_attachments':
				return 'view_attachments_role';
			case 'upload_attachment':
			case 'add_attachments':
				return 'add_attachments_role';
			case 'delete_attachment':
			case 'delete_attachments':
				return 'delete_attachments_role';
			case 'view_checklists':
				return 'view_checklists_role';
			case 'edit_checklists':
			case 'add_checklist_item':
			case 'update_checklist_item':
			case 'delete_checklist_item':
				return 'edit_checklists_role';
			default:
				return '';
		}
	}

	/**
	 * Format notes for display.
	 *
	 * @since 1.0
	 *
	 * @param array $notes The notes to format.
	 *
	 * @return array The formatted notes.
	 */
	public static function format_notes_for_display( $notes ) {
		$formatted_notes = [];
		foreach ( $notes as $note ) {
			$formatted_notes[] = self::format_note_for_display( $note );
		}
		return $formatted_notes;
	}

	/**
	 * Format a single note for display.
	 *
	 * @since 1.0
	 *
	 * @param object $note The note to format.
	 *
	 * @return array The formatted note.
	 */
	public static function format_note_for_display( $note ) {
		$user = get_user_by( 'email', $note->user_email );
		return [
			'id'           => $note->id,
			'user_name'    => $user ? esc_html( $user->display_name ) : esc_html( $note->user_name ),
			'user_email'   => isset( $note->user_email ) ? esc_html( $note->user_email ) : '',
			'date_created' => $note->date_created,
			'note_type'    => $note->note_type,
			'sub_type'     => $note->sub_type,
			'timestamp'    => strtotime( $note->date_created ),
			'user_avatar'  => $user ? get_avatar_url( $user->ID, [ 'size' => 64 ] ) : '',
			'date'         => self::format_date_diff( $note->date_created ),
			'content'      => self::render_mentions_as_html( $note->value ),
			'content_raw'  => $note->value,
			'value'        => $note->value,
		];
	}

	/**
	 * Check if the current user can perform a specific action on any of the active feeds.
	 * This is more performant than checking each feed individually in a loop for each action.
	 *
	 * @since 1.0.0
	 * @since 1.1   $feeds can be a WP_Error as well as an array.
	 *
	 * @param string         $action The action to check (e.g., 'add_card', 'view_board').
	 * @param array|WP_Error $feeds  Array of feed objects or a WP_Error if there are no feeds.
	 *
	 * @return bool True if the user has permission for the action on at least one feed, false otherwise.
	 */
	public static function can_current_user_perform_action_on_any_feed( string $action, $feeds ): bool {

		if ( is_wp_error( $feeds ) || empty( $feeds ) ) {
			return false;
		}

		foreach ( $feeds as $feed ) {
			if ( ! self::user_has_permission( $action, $feed ) ) {
				continue;
			}

			return true;
		}

		return false;
	}

	/**
	 * Parses mentions from note content.
	 *
	 * Extracts user IDs from @[DisplayName](UserID) markup.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content The note content.
	 * @return int[] Array of mentioned user IDs.
	 */
	public static function parse_mentions( $content ) {

		preg_match_all( self::MENTION_REGEX, $content, $matches );

		if ( empty( $matches[1] ) ) {
			return [];
		}

		return Assignees::sanitize_assignees( $matches[2] );
	}

	/**
	 * Get new mentions from old and new note contents.
	 *
	 * @since 1.0.0
	 *
	 * @param string $old_content The old content, possibly containing mentions.
	 * @param string $new_content The new content, possibly containing mentions.
	 *
	 * @return int[] Array of mentioned user IDs.
	 */
	public static function get_new_mentions( $old_content, $new_content ) {
		$old_mentioned_ids = self::parse_mentions( $old_content );
		$new_mentioned_ids = self::parse_mentions( $new_content );
		return array_diff( $new_mentioned_ids, $old_mentioned_ids );
	}

	/**
	 * Renders mentions in note content as HTML.
	 *
	 * Transforms @[DisplayName](UserID) into styled spans.
	 * Also applies wpautop.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content The raw note content.
	 * @return string The note content with mentions rendered as HTML.
	 */
	public static function render_mentions_as_html( $content ) {
		if ( empty( $content ) ) {
			return '';
		}

		$html_content = preg_replace_callback(
			self::MENTION_REGEX,
			function ( $matches ) {
				$display_name = esc_html( $matches[1] );
				$user_id      = (int) $matches[2];
				return sprintf(
					'<span class="gk-gravityboard-mention" data-user-id="%d">@%s</span>',
					$user_id,
					$display_name
				);
			},
			$content
		);

		return nl2br( wp_kses_post( $html_content ) );
	}

	/**
	 * Safely delete a file using WordPress Filesystem API.
	 *
	 * @since 1.1
	 *
	 * @param string $file_path The full path to the file to delete.
	 * @return bool|WP_Error True if file was deleted successfully, false otherwise.
	 */
	public static function delete_file( $file_path ) {
		wp_delete_file_from_directory( $file_path, wp_get_upload_dir()['basedir'] );

		// Check if file exists.
		if ( ! file_exists( $file_path ) ) {
			return new WP_Error( 'file_not_found', esc_html__( 'The attachment file was not found.', 'gk-gravityboard' ) );
		}

		// Initialize WordPress Filesystem.
		global $wp_filesystem;

		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		// Initialize WP_Filesystem.
		if ( ! WP_Filesystem() ) {
			return false;
		}

		// Delete the file using WordPress Filesystem API.
		return $wp_filesystem->delete( $file_path, false, 'f' );
	}

	/**
	 * Get a card by its ID.
	 *
	 * @since 1.0
	 *
	 * @internal This is a helper method for internal use only.
	 *
	 * @param int $card_id The ID of the card to get.
	 *
	 * @return array|WP_Error The card data or a WP_Error if the card is not found.
	 */
	public static function get_card( $card_id ) {
		return \GFAPI::get_entry( $card_id );
	}

	/**
	 * Get a form by its ID, caching the result.
	 *
	 * @since 1.0
	 *
	 * @internal This is a helper method for internal use only.
	 *
	 * @param int $form_id The ID of the form to get.
	 *
	 * @return array|WP_Error The form data or a WP_Error if the form is not found.
	 */
	public static function get_form( $form_id ) {

		if ( empty( $form_id ) ) {
			return new WP_Error( 'invalid_form_id', esc_html__( 'Invalid form ID.', 'gk-gravityboard' ) );
		}

		$form = \GFAPI::get_form( $form_id );

		if ( empty( $form ) ) {
			return new WP_Error( 'form_not_found', esc_html__( 'Form not found.', 'gk-gravityboard' ) );
		}

		return $form;
	}

	/**
	 * Get entries by lane.
	 *
	 * @since 1.0
	 *
	 * @param int    $form_id The ID of the form.
	 * @param int    $lane_field_id The ID of the lane field.
	 * @param string $lane_value The value of the lane.
	 *
	 * @return array|WP_Error Either an array of the Entry objects or a WP_Error instance.
	 */
	public static function get_entries_by_lane( $form_id, $lane_field_id, $lane_value ) {

		// Move entries.
		$search = [
			'status'        => 'active',
			'field_filters' => [
				[
					'key'   => $lane_field_id,
					'value' => $lane_value,
				],
			],
		];

		$entries = \GFAPI::get_entries( $form_id, $search );

		return $entries;
	}

	/**
	 * Check if a lane has entries.
	 *
	 * @since $ver$
	 *
	 * @param int    $form_id The ID of the form.
	 * @param int    $lane_field_id The ID of the lane field.
	 * @param string $lane_value The value of the lane.
	 *
	 * @return bool Whether the lane has entries.
	 */
	public static function has_entries_in_lane( int $form_id, int $lane_field_id, string $lane_value ): bool {
		$search = [
			'status'        => 'active',
			'field_filters' => [
				[
					'key'   => $lane_field_id,
					'value' => $lane_value,
				],
			],
		];

		return \GFAPI::count_entries( $form_id, $search ) > 0;
	}
	/**
	 * Format date difference.
	 *
	 * @since 1.0
	 *
	 * @param string $from The date to compare from.
	 * @param string $to The date to compare to.
	 *
	 * @return string The formatted date difference.
	 */
	public static function format_date_diff( $from, $to = null ) {
		if ( is_null( $to ) ) {
			$to = time();
		}

		// Convert $from to timestamp if it's a date string.
		$from_timestamp = is_numeric( $from ) ? $from : strtotime( $from );

		$time_diff = human_time_diff( $from_timestamp, $to );

		// translators: %s: Human-readable time difference (e.g., "5 minutes", "2 hours").
		return sprintf( esc_html__( '%s ago', 'gk-gravityboard' ), $time_diff );
	}

	/**
	 * Get Help Scout beacon suggestions for the boards overview page.
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	public static function get_beacon_suggestions() {

		return [
			[
				'text' => esc_html__( 'Getting Started', 'gk-gravityboard' ),
				'url'  => self::get_getting_started_url( 'beacon-suggestions' ),
			],
			[
				'text' => esc_html__( 'GravityBoard Documentation', 'gk-gravityboard' ),
				'url'  => 'https://docs.gravitykit.com/category/1049-gravityboard',
			],
		];
	}

	/**
	 * Get the getting started URL with UTM parameters.
	 *
	 * @since 1.0
	 *
	 * @param string $utm_content The UTM content value to add to the URL.
	 *
	 * @return string The getting started URL.
	 */
	public static function get_getting_started_url( $utm_content = 'beacon' ) {
		return add_query_arg(
            [
				'utm_source'   => 'plugin',
				'utm_medium'   => 'inapp',
				'utm_campaign' => 'boards-overview',
				'utm_content'  => $utm_content,
			],
            'https://docs.gravitykit.com/article/1050-getting-started-with-gravityboard'
        );
	}

	/**
	 * Get the lane ID from the lane value.
	 *
	 * @since 1.1
	 *
	 * @param array{id: string, title: string, value: string, cards: array{}, color: string}[] $lanes The lanes.
	 * @param string                                                                           $lane_value The lane value.
	 * @return string The lane ID.
	 */
	public static function get_lane_id_from_value( $lanes, $lane_value ) {

		// Lanes are case-insensitive.
		$lane_values = array_map( 'strtolower', wp_list_pluck( $lanes, 'value', 'id' ) );
		$lane_id     = array_search( strtolower( $lane_value ), $lane_values, true );

		// If the lane value is not found, return the default lane ID.
		return false === $lane_id ? '-1' : (string) $lane_id;
	}

	/**
	 * Validate a field value using Gravity Forms validation.
	 *
	 * @since 1.1
	 *
	 * @param string $field_id The field ID to validate.
	 * @param mixed  $value The value to validate.
	 * @param array  $form The form object.
	 * @param array  $entry Optional. Entry data for context.
	 * @return WP_Error|true WP_Error if validation fails, true if valid.
	 */
	public static function validate_field_value( $field_id, $value, $form, $entry = [] ) {
		$field = \GFAPI::get_field( $form, $field_id );
		if ( ! $field ) {
			return new WP_Error( 'field_not_found', __( 'Field not found.', 'gk-gravityboard' ) );
		}

		// Check required fields.
		if ( $field->isRequired && self::is_field_empty( $value ) ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			return new WP_Error( 'required_field', $field->errorMessage ?: __( 'This field is required.', 'gk-gravityboard' ) ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		}

		// Check duplicates.
		if ( $field->noDuplicates && ! self::is_field_empty( $value ) ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			$old_value = rgar( $entry, $field_id );
			if ( $old_value !== $value && \GFFormsModel::is_duplicate( $form['id'], $field, $value ) ) {
				// translators: %s is the value of the field.
				return new WP_Error( 'duplicate_value', sprintf( __( "This field requires a unique entry and '%s' has already been used.", 'gk-gravityboard' ), $value ) );
			}
		}

		// Use Gravity Forms validation.
		$field->failed_validation  = false;
		$field->validation_message = '';
		$field->formId             = $form['id']; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase

		$field->validate( $value, $form );

		if ( $field->failed_validation ) {
			return new WP_Error( 'validation_failed', $field->validation_message ?: __( 'Invalid value.', 'gk-gravityboard' ) );
		}

		return true;
	}

	/**
	 * Check if a field value is empty.
	 *
	 * @since 1.1
	 *
	 * @param mixed $value The value to check.
	 * @return bool True if empty, false otherwise.
	 */
	public static function is_field_empty( $value ) {
		if ( is_array( $value ) ) {
			foreach ( $value as $v ) {
				if ( '' !== trim( (string) $v ) ) {
					return false;
				}
			}
			return true;
		}

		return '' === trim( (string) $value );
	}

	/**
	 * Validate multiple field values at once.
	 *
	 * @since 1.1
	 *
	 * @internal This is a helper method for internal use only.
	 *
	 * @param array $field_values Array of field_id => value pairs.
	 * @param array $form The form object.
	 * @param array $entry Optional. Entry data for context.
	 * @param array $field_mappings Optional. Array mapping field_id to field_key for better error handling.
	 *
	 * @return ErrorCollection|true ErrorCollection if validation fails, true if all valid.
	 */
	public static function validate_fields( $field_values, $form, $entry = [], $field_mappings = [] ) {
		$validation_errors = new ErrorCollection();
		foreach ( $field_values as $field_id => $value ) {
			if ( ! $field_id ) {
				continue; // Skip unmapped fields
			}

			$result = self::validate_field_value( $field_id, $value, $form, $entry );
			if ( is_wp_error( $result ) ) {
				// Get field information for better frontend handling.
				$field_key = $field_mappings[ (string) $field_id ] ?? null;

				$result->add_data( [ 'field_name' => $field_key ] );
				$validation_errors->add( $result );
			}
		}

		if ( $validation_errors->has_errors() ) {
			return $validation_errors;
		}

		return true;
	}

	/**
	 * Sanitize field value using Gravity Forms field sanitization.
	 *
	 * @since 1.1
	 *
	 * @param mixed  $value The value to sanitize.
	 * @param array  $form The form object.
	 * @param string $field_id The field ID.
	 * @return string The sanitized value.
	 */
	public static function sanitize_field_value( $value, $form, $field_id ) {
		$field = \GFAPI::get_field( $form, $field_id );
		if ( ! $field ) {
			return sanitize_text_field( $value );
		}

		// Use Gravity Forms' built-in sanitization.
		return $field->sanitize_entry_value( $value, $form['id'] );
	}

	/**
	 * Check if a toggle field is enabled.
	 *
	 * @since 1.1
	 *
	 * @param array  $feed The feed.
	 * @param string $key The key of the toggle field.
	 * @param bool   $default_value The default value if the key is not found.
	 *
	 * @return bool True if the toggle field is enabled, false otherwise.
	 */
	public static function is_toggle_enabled( $feed, $key, $default_value = false ) {

		$value = rgars( $feed, $key, null );

		if ( is_null( $value ) ) {
			return $default_value;
		}

		return (bool) $value;
	}
}
