<?php
/**
 * GravityView integration.
 *
 * @package GravityKit\GravityBoard\Integrations
 * @since   1.1
 */

namespace GravityKit\GravityBoard\Integrations;

use GravityKit\GravityBoard\Feed;
use GVCommon;
use GravityView_Edit_Entry;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The GravityView integration class.
 *
 * @since 1.1
 */
class GravityView {
	/**
	 * The instance of this class.
	 *
	 * @var GravityView|null
	 */
	private static $instance = null;

	/**
	 * Returns the instance of this class.
	 *
	 * @return GravityView|null
	 */
	public static function get_instance() {
		if ( ! self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * GravityView constructor.
	 *
	 * @since 1.1
	 */
	private function __construct() {
		$this->add_hooks();
	}

	/**
	 * Adds the hooks for this integration.
	 *
	 * @return void
	 */
	private function add_hooks() {
		if ( ! class_exists( 'GravityView_Plugin' ) ) {
			return;
		}

		add_filter( 'gform_gravityboard_feed_settings_fields', [ $this, 'add_feed_settings_fields' ], 10, 2 );
		add_filter( 'gk/gravityboard/board-settings', [ $this, 'add_board_settings' ], 10, 2 );
		add_filter( 'gk/gravityboard/card/data', [ $this, 'add_gravityview_urls_to_card' ], 10, 3 );
	}

	/**
	 * Adds the GravityView settings to the feed settings.
	 *
	 * @param array $settings The feed settings.
	 * @param Feed  $addon    The Feed addon instance.
	 *
	 * @return array The modified feed settings.
	 */
	public function add_feed_settings_fields( $settings, $addon ) {
		if ( ! class_exists( 'GV\View' ) ) {
			return $settings;
		}

		// Check if the form is not supported by GravityBoard.
		foreach ( $settings as $section ) {
			if ( isset( $section['fields'] ) ) {
				foreach ( $section['fields'] as $field ) {
					if ( isset( $field['name'] ) && 'gravityboard_not_supported' === $field['name'] ) {
						return $settings;
					}
				}
			}
		}

		$form    = $addon->get_current_form();
		$form_id = $form['id'];

		$views = \GVCommon::get_connected_views(
            $form_id,
            [
				'orderby' => 'post_title',
				'order'   => 'ASC',
			],
			false
        );

		if ( ! empty( $views ) ) {
			$choices = [
				[
					'label' => esc_html__( '— Select a View —', 'gk-gravityboard' ),
					'value' => '',
				],
			];

			foreach ( $views as $view ) {
				if ( ! $view->post_title ) {
					$view_title = esc_html_x( '(Untitled)', 'The label for a View that has no title', 'gk-gravityboard' );
				} else {
					$view_title = esc_html( $view->post_title );
				}

				$choices[] = [
					// translators: %1$s is the view title, %2$d is the view ID.
					'label' => sprintf( esc_html__( '%1$s (#%2$d)', 'gk-gravityboard' ), $view_title, $view->ID ),
					'value' => $view->ID,
				];
			}

			$gravityview_settings = [
				'title'    => esc_html__( 'GravityView', 'gk-gravityboard' ),
				'sections' => [
					[
						'title'  => esc_html__( 'Connect a View', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'        => 'gravityview_view_id',
								'label'       => esc_html__( 'Select a View', 'gk-gravityboard' ),
								'type'        => 'select',
								'description' => esc_html__( 'Select a GravityView View to link to this board. This will add "View in GravityView" and "Edit in GravityView" links to each card.', 'gk-gravityboard' ),
								'choices'     => $choices,
							],
							[
								'name'        => 'gravityview_page_id',
								'label'       => esc_html__( 'Embed Page ID (Optional)', 'gk-gravityboard' ),
								'type'        => 'text',
								'input_type'  => 'number',
								'description' => esc_html__( 'Enter the ID of the page where the selected GravityView View is embedded in order to generate correct entry links.', 'gk-gravityboard' ),
								'class'       => 'small',
								'dependency'  => [
									'live'   => true,
									'fields' => [
										[
											'field'    => 'gravityview_view_id',
											'value'    => '',
											'operator' => 'isnot',
										],
									],
								],
							],
						],
					],
				],
			];
		} else {
			// No views exist - show instructions to create one.
			$create_view_url = admin_url( 'post-new.php?post_type=gravityview&form_id=' . $form_id );

			$gravityview_settings = [
				'title'    => esc_html__( 'GravityView', 'gk-gravityboard' ),
				'sections' => [
					[
						'title'  => esc_html__( 'No GravityView Views found for this form.', 'gk-gravityboard' ),
						'fields' => [
							[
								'name'  => 'gravityview_create_instructions',
								'label' => '',
								'type'  => 'html',
								'html'  => strtr(
									'<span class="gform-settings-description">{description}</span>
									<a href="{url}" class="button button-primary">{label}</a>',
									[
										'{description}' => esc_html__( 'GravityView allows you to display Gravity Forms entries on the frontend of your site in custom layouts, including dynamic tables, lists, and more.', 'gk-gravityboard' ),
										'{url}'         => esc_url( $create_view_url ),
										'{label}'       => esc_html__( 'Create a View for this Form', 'gk-gravityboard' ),
									]
								),
							],
						],
					],
				],
			];
		}

		// Insert the GravityView settings after the "Configure Board" section.
		array_splice( $settings, 1, 0, [ $gravityview_settings ] );

		return $settings;
	}

	/**
	 * Adds the GravityView settings to the board settings script.
	 *
	 * @param array $board_settings The board settings.
	 * @param int   $feed_id        The feed ID.
	 *
	 * @return array The modified board settings.
	 */
	public function add_board_settings( $board_settings, $feed_id ) {

		if ( ! class_exists( 'GV\View' ) ) {
			return $board_settings;
		}

		$feed = Feed::get_instance()->get_feed( $feed_id );

		$view_id = rgars( $feed, 'meta/gravityview_view_id' );
		$page_id = rgars( $feed, 'meta/gravityview_page_id' );

		$board_settings['gravityview_view_id'] = $view_id;
		$board_settings['gravityview_page_id'] = $page_id;

		// Ensure features array exists before setting GravityView integration status.
		if ( ! isset( $board_settings['features'] ) ) {
			$board_settings['features'] = [];
		}

		$board_settings['features']['enable_gravityview'] = ! empty( $view_id );

		return $board_settings;
	}

	/**
	 * Adds GravityView view and edit URLs to the card data.
	 *
	 * @param array $card  The card data.
	 * @param array $entry The entry data.
	 * @param array $feed  The feed data.
	 *
	 * @return array The modified card data.
	 */
	public function add_gravityview_urls_to_card( $card, $entry, $feed ) {
		$view_id = rgars( $feed, 'meta/gravityview_view_id' );
		$page_id = rgars( $feed, 'meta/gravityview_page_id' );

		if ( ! $view_id || ! class_exists( 'GravityView_API' ) || ! class_exists( 'GravityView_Edit_Entry' ) ) {
			return $card;
		}

		$can_read_view  = GVCommon::has_cap( 'read_gravityview', $view_id );
		$can_edit_entry = GravityView_Edit_Entry::check_user_cap_edit_entry( $entry, $view_id );

		if ( ! $can_read_view && ! $can_edit_entry ) {
			return $card;
		}

		// Use the page ID if provided, otherwise use the view ID.
		$post_id = ! empty( $page_id ) ? $page_id : $view_id;

		$card['gv_view_url'] = $can_read_view ? \GravityView_API::entry_link( $entry, $post_id ) : '';
		$card['gv_edit_url'] = $can_edit_entry ? \GravityView_Edit_Entry::get_edit_link( $entry, $view_id, $post_id ) : '';

		return $card;
	}
}

GravityView::get_instance();
