<?php
/**
 * GravityBoard Lane
 *
 * @package GravityBoard
 * @since   1.0
 */

namespace GravityKit\GravityBoard;

use GFAPI;
use WP_Error;
use WP_REST_Request;
use GravityKit\GravityBoard\Cards\CardCollection;

defined( 'ABSPATH' ) || exit;

/**
 * Class Lane
 *
 * Represents a single lane in a GravityBoard with its properties and operations.
 *
 * @since 1.0.0
 */
class Lane {

	/**
	 * Lane ID.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	private string $id;

	/**
	 * Lane title.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	private string $title;

	/**
	 * Lane value.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	private string $value;

	/**
	 * Lane cards.
	 *
	 * @since 1.0
	 *
	 * @var CardCollection
	 */
	private CardCollection $cards;

	/**
	 * Lane color.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	private string $color;

	/**
	 * Feed data.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	private array $feed;

	/**
	 * Form data.
	 *
	 * @since 1.0
	 *
	 * @var array
	 */
	private array $form;

	/**
	 * Lane field.
	 *
	 * @since 1.0
	 *
	 * @var \GF_Field|WP_Error
	 */
	private $lane_field;

	/**
	 * Lane constructor.
	 *
	 * @since 1.0
	 *
	 * @param string              $id     The lane ID.
	 * @param string              $title  The lane title.
	 * @param string              $value  The lane value.
	 * @param string              $color  The lane color.
	 * @param CardCollection|null $cards  The lane cards.
	 * @param array               $feed   The feed data (optional, for operations).
	 * @param array               $form   The form data (optional, for operations).
	 */
	public function __construct(
		string $id,
		string $title,
		string $value,
		string $color = '',
		CardCollection $cards = null,
		array $feed = [],
		array $form = []
	) {
		$this->id    = $id;
		$this->title = $title;
		$this->value = $value;
		$this->color = $color;
		$this->cards = $cards ?? new CardCollection();
		$this->feed  = $feed;
		$this->form  = $form;

		if ( ! empty( $form ) ) {
			$this->lane_field = $this->get_lane_field();
		}
	}

	/**
	 * Get lane ID.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Get lane title.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_title(): string {
		return $this->title;
	}

	/**
	 * Set lane title.
	 *
	 * @since 1.0
	 *
	 * @param string $title The lane title.
	 */
	public function set_title( string $title ): void {
		$this->title = $title;
	}

	/**
	 * Get lane value.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_value(): string {
		return $this->value;
	}

	/**
	 * Set lane value.
	 *
	 * @since 1.0
	 *
	 * @param string $value The lane value.
	 */
	public function set_value( string $value ): void {
		$this->value = $value;
	}

	/**
	 * Get lane color.
	 *
	 * @since 1.0
	 *
	 * @return string
	 */
	public function get_color(): string {
		return $this->color;
	}

	/**
	 * Set lane color.
	 *
	 * @since 1.0
	 *
	 * @param string $color The lane color.
	 */
	public function set_color( string $color ): void {
		$this->color = $color;
	}

	/**
	 * Get lane cards.
	 *
	 * @since 1.0
	 *
	 * @return CardCollection
	 */
	public function get_cards(): CardCollection {
		return $this->cards;
	}

	/**
	 * Set lane cards.
	 *
	 * @since 1.0
	 *
	 * @param CardCollection $cards The lane cards.
	 */
	public function set_cards( CardCollection $cards ): void {
		$this->cards = $cards;
	}

	/**
	 * Add a card to the lane.
	 *
	 * @since 1.0
	 *
	 * @param Card $card The card to add.
	 */
	public function add_card( Card $card ): void {
		$this->cards->add_card( $card );
	}

	/**
	 * Convert lane to array format.
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	public function to_array(): array {
		return [
			'id'    => $this->id,
			'title' => $this->title,
			'value' => $this->value,
			'cards' => $this->cards->to_array(),
			'color' => $this->color,
		];
	}

	/**
	 * Create a lane from request data.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 * @param array           $feed    The feed data.
	 *
	 * @return array|WP_Error Lane data on success, WP_Error on failure.
	 */
	public static function create_from_request( WP_REST_Request $request, array $feed ) {
		$form = Helpers::get_form( $feed['form_id'] );
		if ( is_wp_error( $form ) ) {
			return $form;
		}

		// Create a temporary lane for operations.
		$lane = new self( '', '', '', '', new CardCollection(), $feed, $form );
		return $lane->add_lane( $request );
	}

	/**
	 * Update a lane from request data.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 * @param array           $feed    The feed data.
	 * @param int             $lane_id The lane ID.
	 *
	 * @return array|WP_Error Update result on success, WP_Error on failure.
	 */
	public static function update_from_request( WP_REST_Request $request, array $feed, int $lane_id ) {
		$form = Helpers::get_form( $feed['form_id'] );
		if ( is_wp_error( $form ) ) {
			return $form;
		}

		// Create a temporary lane for operations.
		$lane = new self( '', '', '', '', new CardCollection(), $feed, $form );
		return $lane->update_lane( $request, $lane_id );
	}

	/**
	 * Delete a lane.
	 *
	 * @since 1.0
	 *
	 * @param array $feed    The feed data.
	 * @param int   $lane_id The lane ID.
	 *
	 * @return array|WP_Error Delete result on success, WP_Error on failure.
	 */
	public static function delete( array $feed, int $lane_id ) {
		$form = Helpers::get_form( $feed['form_id'] );
		if ( is_wp_error( $form ) ) {
			return $form;
		}

		// Create a temporary lane for operations.
		$lane = new self( '', '', '', '', new CardCollection(), $feed, $form );
		return $lane->delete_lane( $lane_id );
	}

	/**
	 * Move a lane.
	 *
	 * @since 1.0
	 *
	 * @param array $feed        The feed data.
	 * @param int   $lane_id     The lane ID.
	 * @param int   $new_position The new position.
	 *
	 * @return array|WP_Error Move result on success, WP_Error on failure.
	 */
	public static function move( array $feed, int $lane_id, int $new_position ) {
		$form = Helpers::get_form( $feed['form_id'] );
		if ( is_wp_error( $form ) ) {
			return $form;
		}

		// Create a temporary lane for operations.
		$lane = new self( '', '', '', '', new CardCollection(), $feed, $form );
		return $lane->move_lane( $lane_id, $new_position );
	}

	/**
	 * Build lanes data from field.
	 *
	 * @since 1.0
	 *
	 * @param \GF_Field $lane_field The lane field.
	 * @param array     $feed       The feed data.
	 *
	 * @return array The lanes data.
	 */
	public static function build_lanes_from_field( $lane_field, array $feed ): array {
		$lanes = [];
		foreach ( $lane_field->choices as $key => $choice ) {
			$lane                   = new self(
				(string) $key,
				$choice['text'],
				$choice['value'],
				Feed::get_instance()->get_lane_color( $feed, $choice['value'] ),
				new CardCollection()
			);
			$lanes[ (string) $key ] = $lane->to_array();
		}

		return $lanes;
	}

	/**
	 * Get the lane field.
	 *
	 * @since 1.0
	 *
	 * @return \GF_Field|WP_Error The lane field or error.
	 */
	private function get_lane_field() {
		$lane_field_id = Feed::get_instance()->get_lane_field_id( $this->feed );
		$lane_field    = GFAPI::get_field( $this->form, $lane_field_id );

		if ( ! $lane_field ) {
			return new WP_Error( 'lane_field_not_found', __( 'Lane field not found.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		return $lane_field;
	}

	/**
	 * Add a new lane.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 *
	 * @return array|WP_Error Lane data on success, WP_Error on failure.
	 */
	private function add_lane( WP_REST_Request $request ) {
		if ( is_wp_error( $this->lane_field ) ) {
			return $this->lane_field;
		}

		$lane_title = sanitize_text_field( $request->get_param( 'title' ) );
		if ( '' === $lane_title ) {
			return new WP_Error( 'empty_title', __( 'Lane title cannot be empty.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		// Check for duplicates.
		foreach ( $this->lane_field->choices as $choice ) {
			if ( strtolower( $choice['text'] ) === strtolower( $lane_title ) ) {
				return new WP_Error( 'duplicate_lane', __( 'A lane with this name already exists.', 'gk-gravityboard' ), [ 'status' => 409 ] );
			}
		}

		$new_lane_id     = count( $this->lane_field->choices );
		$new_lane_choice = [
			'text'       => $lane_title,
			'value'      => $lane_title,
			'isSelected' => false,
			'price'      => null,
		];

		$this->lane_field->choices[] = $new_lane_choice;

		$result = $this->save_form();
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		/**
		 * Runs when a lane has been added.
		 *
		 * @since 1.0
		 *
		 * @param int   $new_lane_id     The ID of the new lane.
		 * @param array $new_lane_choice The new lane choice.
		 * @param array $feed            The feed data.
		 */
		do_action( 'gk/gravityboard/lane/added', $new_lane_id, $new_lane_choice, $this->feed );

		return [
			'id'    => (string) $new_lane_id,
			'title' => $lane_title,
			'cards' => [],
		];
	}

	/**
	 * Update a lane.
	 *
	 * @since 1.0
	 *
	 * @param WP_REST_Request $request The REST request.
	 * @param int             $lane_id The lane ID.
	 *
	 * @return array|WP_Error Update result on success, WP_Error on failure.
	 */
	private function update_lane( WP_REST_Request $request, int $lane_id ) {
		if ( is_wp_error( $this->lane_field ) ) {
			return $this->lane_field;
		}

		$new_title = sanitize_text_field( $request->get_param( 'title' ) );
		if ( '' === trim( $new_title ) ) {
			return new WP_Error( 'empty_title', __( 'Lane title cannot be empty.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		if ( ! isset( $this->lane_field->choices[ $lane_id ] ) ) {
			return new WP_Error( 'lane_not_found', __( 'Lane not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}

		// Check for duplicates.
		foreach ( $this->lane_field->choices as $key => $choice ) {
			if ( $key !== $lane_id && strtolower( $choice['text'] ) === strtolower( $new_title ) ) {
				return new WP_Error( 'duplicate_lane', __( 'A lane with this name already exists.', 'gk-gravityboard' ), [ 'status' => 409 ] );
			}
		}

		// Don't update the value, just the text.
		// Otherwise, cards will be detached from the lane.
		$this->lane_field->choices[ $lane_id ]['text'] = $new_title;

		$result = $this->save_form();
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		/**
		 * Runs when a lane has been edited.
		 *
		 * @since 1.0
		 *
		 * @param array $form          The form data.
		 * @param int   $lane_field_id The ID of the lane field.
		 * @param array $feed          The feed data.
		 */
		do_action( 'gk/gravityboard/lane/edited', $this->form, $this->lane_field->id, $this->feed );

		return [ 'success' => true ];
	}

	/**
	 * Delete a lane.
	 *
	 * @since 1.0
	 *
	 * @param int $lane_id The lane ID.
	 *
	 * @return array|WP_Error Delete result on success, WP_Error on failure.
	 */
	private function delete_lane( int $lane_id ) {
		if ( is_wp_error( $this->lane_field ) ) {
			return $this->lane_field;
		}

		if ( ! isset( $this->lane_field->choices[ $lane_id ] ) ) {
			return new WP_Error( 'lane_not_found', __( 'Lane not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}

		$lane_value  = (string) $this->lane_field->choices[ $lane_id ]['value'];
		$has_entries = Helpers::has_entries_in_lane(
			(int) $this->form['id'],
			(int) $this->lane_field->id,
			$lane_value,
		);

		if ( $has_entries ) {
			return new WP_Error(
				'lane_has_cards',
				__( 'This lane has cards. Please move cards to another lane before deleting.', 'gk-gravityboard' ),
				[ 'status' => 409 ]
			);
		}

		// Remove choice and reindex.
		unset( $this->lane_field->choices[ $lane_id ] );
		$this->lane_field->choices = array_values( $this->lane_field->choices );

		$result = $this->save_form();
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		/**
		 * Runs when a lane has been deleted.
		 *
		 * @since 1.0
		 *
		 * @param array $form          The form data.
		 * @param int   $lane_field_id The ID of the lane field.
		 * @param array $feed          The feed data.
		 * @param int   $lane_id       The ID of the lane that was deleted.
		 */
		do_action( 'gk/gravityboard/lane/deleted', $this->form, $this->lane_field->id, $this->feed, $lane_id );

		return [ 'deleted' => true ];
	}

	/**
	 * Move a lane.
	 *
	 * @since 1.0
	 *
	 * @param int $lane_id     The lane ID.
	 * @param int $new_position The new position.
	 *
	 * @return array|WP_Error Move result on success, WP_Error on failure.
	 */
	private function move_lane( int $lane_id, int $new_position ) {
		if ( is_wp_error( $this->lane_field ) ) {
			return $this->lane_field;
		}

		if ( ! isset( $this->lane_field->choices[ $lane_id ] ) ) {
			return new WP_Error( 'lane_not_found', __( 'Lane not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}

		// Remove the choice and insert it at the new position.
		$choice = array_splice( $this->lane_field->choices, $lane_id, 1 );
		array_splice( $this->lane_field->choices, $new_position, 0, $choice );

		$result = $this->save_form();
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		/**
		 * Runs when a lane has been moved.
		 *
		 * @since 1.0
		 *
		 * @param array $form          The form data.
		 * @param int   $lane_field_id The ID of the lane field.
		 * @param array $feed          The feed data.
		 * @param int   $new_position  The new position of the lane.
		 */
		do_action( 'gk/gravityboard/lane/moved', $this->form, $this->lane_field->id, $this->feed, $new_position );

		return [ 'success' => true ];
	}

	/**
	 * Save the form with updated lane field.
	 *
	 * @since 1.0
	 *
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	private function save_form() {
		$update = GFAPI::update_form( $this->form );
		if ( is_wp_error( $update ) ) {
			return $update;
		}

		return true;
	}
}
