<?php
/**
 * GravityBoard Attachments
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Notes;

defined( 'ABSPATH' ) || die();

use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Ajax;
use GravityKit\GravityBoard\Helpers;
use GravityKit\GravityBoard\Notifications;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Class Ajax
 *
 * Handles AJAX requests for GravityBoard operations.
 *
 * @since 1.0
 */
class Notes {


	/**
	 * The type of note used by GravityBoard.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	const NOTE_TYPE = 'gravityboard';

	/**
	 * The sub-type of note used by GravityBoard.
	 *
	 * @since 1.0
	 *
	 * @var string
	 */
	const NOTE_SUB_TYPE = 'gravityboard_feed_%d';

	/**
	 * Instance of the class.
	 *
	 * @var Notes
	 */
	public static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.0
	 *
	 * @return Notes The instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0
	 */
	public function __construct() {
		add_filter( 'gk/gravityboard/data/lanes', [ $this, 'add_note_counts_to_lanes' ], 10, 3 );
		add_action( 'gk/gravityboard/ajax/register-routes', [ $this, 'register_routes' ] );
	}

	/**
	 * Register REST routes for notes.
	 *
	 * @since 1.0
	 */
	public function register_routes() {
		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/notes',
			[
				'methods'             => 'GET',
				'callback'            => [ $this, 'fetch_entry_notes' ],
				'args'                => [
					'entry_id' => [
						'type'     => 'integer',
						'required' => true,
					],
					'feed_id'  => [
						'type'     => 'integer',
						'required' => true,
					],
				],
				'permission_callback' => [ $this, 'can_fetch_entry_notes' ],
			]
		);

		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/notes',
			[
				'methods'             => 'POST',
				'callback'            => [ $this, 'add_entry_note' ],
				'permission_callback' => [ $this, 'can_add_entry_note' ],
			]
		);

		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/notes/(?P<note_id>\d+)',
			[
				'methods'             => 'PUT',
				'callback'            => [ $this, 'edit_entry_note' ],
				'permission_callback' => [ $this, 'can_edit_entry_note' ],
			]
		);

		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/notes/(?P<note_id>\d+)',
			[
				'methods'             => 'DELETE',
				'callback'            => [ $this, 'delete_entry_note' ],
				'permission_callback' => [ $this, 'can_delete_entry_note' ],
			]
		);
	}

	/**
	 * Handles fetching entry notes via REST API.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response The REST response object.
	 */
	public function fetch_entry_notes( WP_REST_Request $request ) {
		$entry    = Ajax::get_instance()->get_entry_from_request( $request );
		$feed     = Ajax::get_instance()->get_feed_from_request( $request );
		$entry_id = $entry['id'];

		if ( ! $feed || is_wp_error( $feed ) ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Feed not found or error retrieving feed.', 'gk-gravityboard' ) ], 404 );
		}

		if ( ! Feed::get_instance()->user_has_permission( 'view_entry_notes', $feed ) ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'You do not have permission to view entry notes for this feed.', 'gk-gravityboard' ) ], 403 );
		}

		if ( empty( $feed['meta']['enable_entry_notes'] ) || '1' !== $feed['meta']['enable_entry_notes'] ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Entry notes are not enabled for this board.', 'gk-gravityboard' ) ], 400 );
		}

		$notes           = self::get_notes( $entry_id );
		$notes_count     = is_array( $notes ) ? count( $notes ) : 0;
		$formatted_notes = Helpers::format_notes_for_display( $notes );

		return new WP_REST_Response(
			[
				'notes'       => $formatted_notes,
				'notes_count' => $notes_count,
			],
			200
		);
	}

	/**
	 * Check if the current user has permission to fetch entry notes.
	 *
	 * @since 1.0
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_fetch_entry_notes( WP_REST_Request $request ): bool {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return false;
		}
		return Feed::get_instance()->user_has_permission( 'view_entry_notes', $feed );
	}

	/**
	 * Check if the current user has permission to add an entry note.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_add_entry_note( WP_REST_Request $request ): bool {
		try {
			$feed = Ajax::get_instance()->get_feed_from_request( $request );
			if ( ! $feed || is_wp_error( $feed ) ) {
				return false;
			}
			return Feed::get_instance()->user_has_permission( 'add_entry_notes', $feed );
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * Check if the current user has permission to edit an entry note.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_edit_entry_note( WP_REST_Request $request ): bool {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return false;
		}
		return Feed::get_instance()->user_has_permission( 'manage_entry_notes', $feed );
	}

	/**
	 * Check if the current user has permission to delete an entry note.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_delete_entry_note( WP_REST_Request $request ): bool {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return false;
		}
		return Feed::get_instance()->user_has_permission( 'manage_entry_notes', $feed );
	}

	/**
	 * Handles adding an entry note via REST API.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response The REST response object.
	 */
	public function add_entry_note( WP_REST_Request $request ): WP_REST_Response {
		$feed_id  = (int) $request->get_param( 'feed_id' );
		$entry_id = (int) $request->get_param( 'entry_id' );

		$params           = $request->get_params();
		$note_content_raw = isset( $params['note'] ) ? wp_kses_post( wp_unslash( $params['note'] ) ) : '';
		$site_path        = isset( $params['site_path'] ) ? sanitize_text_field( wp_unslash( $params['site_path'] ) ) : '';

		$feed = Feed::get_instance()->get_feed( $feed_id );
		if ( is_wp_error( $feed ) || ! $feed ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Feed not found.', 'gk-gravityboard' ) ], 404 );
		}

		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) || ! $entry ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Entry not found.', 'gk-gravityboard' ) ], 404 );
		}

		if ( empty( trim( $note_content_raw ) ) ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Note content cannot be empty.', 'gk-gravityboard' ) ], 400 );
		}

		$current_user = wp_get_current_user();
		$user_id      = $current_user->ID;
		$user_name    = $current_user->display_name;

		$mentioned_user_ids = Helpers::parse_mentions( $note_content_raw );

		$note_id = self::add_note( $entry_id, $user_id, $user_name, $note_content_raw, self::NOTE_TYPE, $feed['id'] );

		if ( is_wp_error( $note_id ) ) {
			/** @var \WP_Error $error_object */
			$error_object = $note_id;
			return new WP_REST_Response( [ 'message' => $error_object->get_error_message() ], 500 );
		}

		if ( $note_id ) {
			if ( ! empty( $mentioned_user_ids ) ) {
				Notifications::send_mention_notifications( $mentioned_user_ids, $entry_id, (int) $note_id, $note_content_raw, $feed, $site_path );
			}

			$new_note_object    = \GFAPI::get_note( $note_id );
			$response_note_data = Helpers::format_note_for_display( $new_note_object );

			return new WP_REST_Response(
				[
					'message'     => esc_html__( 'Note added successfully.', 'gk-gravityboard' ),
					'note'        => $response_note_data,
					'notes_count' => self::get_entry_notes_counts( [ $entry_id ], $feed['id'] )[ $entry_id ] ?? 0,
				],
				200
			);
		} else {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Failed to add note.', 'gk-gravityboard' ) ], 500 );
		}
	}

	/**
	 * Handles editing an entry note via REST API.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response The REST response object.
	 */
	public function edit_entry_note( WP_REST_Request $request ): WP_REST_Response {
		$feed_id  = (int) $request->get_param( 'feed_id' );
		$entry_id = (int) $request->get_param( 'entry_id' );
		$note_id  = (int) $request->get_param( 'note_id' );

		$params               = $request->get_params();
		$new_note_content_raw = isset( $params['note_content'] ) ? wp_kses_post( wp_unslash( $params['note_content'] ) ) : '';
		$site_path            = isset( $params['site_path'] ) ? sanitize_text_field( wp_unslash( $params['site_path'] ) ) : '';

		$feed = Feed::get_instance()->get_feed( $feed_id );
		if ( is_wp_error( $feed ) || ! $feed ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Feed not found.', 'gk-gravityboard' ) ], 404 );
		}

		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) || ! $entry ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Entry not found.', 'gk-gravityboard' ) ], 404 );
		}

		if ( empty( $note_id ) ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Missing note ID for editing.', 'gk-gravityboard' ) ], 400 );
		}

		if ( empty( trim( $new_note_content_raw ) ) ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Note content cannot be empty.', 'gk-gravityboard' ) ], 400 );
		}

		$existing_note = \GFAPI::get_note( $note_id );

		if ( ! $existing_note || is_wp_error( $existing_note ) ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Original note not found.', 'gk-gravityboard' ) ], 404 );
		}

		if ( (int) $existing_note->entry_id !== $entry_id ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Note does not belong to the specified entry.', 'gk-gravityboard' ) ], 400 );
		}

		$updated = \GFAPI::update_note(
			[
				'id'    => $note_id,
				'value' => $new_note_content_raw,
			]
		);

		if ( is_wp_error( $updated ) ) {
			return new WP_REST_Response( [ 'message' => $updated->get_error_message() ], 500 );
		}

		if ( ! $updated ) {
			return new WP_REST_Response( [ 'message' => esc_html__( 'Failed to update note.', 'gk-gravityboard' ) ], 500 );
		}

		$newly_added_mention_ids = Helpers::get_new_mentions( $existing_note->value, $new_note_content_raw );

		if ( ! empty( $newly_added_mention_ids ) ) {
			Notifications::send_mention_notifications( $newly_added_mention_ids, $entry_id, $note_id, $new_note_content_raw, $feed, $site_path );
		}

		$updated_note_object = \GFAPI::get_note( $note_id );
		$response_note_data  = Helpers::format_note_for_display( $updated_note_object );

		return new WP_REST_Response(
			[
				'message' => esc_html__( 'Note updated successfully.', 'gk-gravityboard' ),
				'note'    => $response_note_data,
			],
			200
		);
	}

	/**
	 * Add notes to the card.
	 *
	 * @since 1.0
	 *
	 * @param array $lanes The lanes data.
	 * @param array $entries The entries data.
	 * @param array $feed The feed data.
	 * @return array The lanes data.
	 */
	public function add_note_counts_to_lanes( $lanes, $entries, $feed ) {

		// Only show notes if enabled and user has permission.
		$enable_notes   = (bool) rgars( $feed, 'meta/enable_entry_notes', false );
		$can_view_notes = Feed::get_instance()->user_has_permission( 'view_entry_notes', $feed );

		if ( ! $enable_notes || ! $can_view_notes ) {
			return $lanes;
		}

		$note_counts = self::get_entry_notes_counts( array_column( $entries, 'id' ), $feed['id'] );

		foreach ( $lanes as &$lane ) {
			foreach ( $lane['cards'] as &$card ) {
				$card['notes_count'] = rgar( $note_counts, $card['id'], 0 );
			}
		}

		return $lanes;
	}

	/**
	 * Delete an entry note via REST API.
	 *
	 * @since 1.0
	 * @param WP_REST_Request $request The REST request object.
	 */
	public function delete_entry_note( WP_REST_Request $request ) {

		$entry   = Ajax::get_instance()->get_entry_from_request( $request );
		$feed    = Ajax::get_instance()->get_feed_from_request( $request );
		$note_id = $request->get_param( 'note_id' );

		if ( empty( $note_id ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Note ID is required', 'gk-gravityboard' ) ] );
		}

		if ( empty( $feed['meta']['enable_entry_notes'] ) || '1' !== $feed['meta']['enable_entry_notes'] ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Entry notes are not enabled for this board', 'gk-gravityboard' ) ] );
		}

		if ( ! Feed::get_instance()->user_has_permission( 'manage_entry_notes', $feed ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'You do not have permission to delete entry notes', 'gk-gravityboard' ) ] );
		}

		$deleted = \GFAPI::delete_note( $note_id );

		if ( is_wp_error( $deleted ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to delete note', 'gk-gravityboard' ) . ': ' . $deleted->get_error_message() ] );
		}

		if ( ! $deleted ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to delete note.', 'gk-gravityboard' ) ] );
		}

		$all_notes   = self::get_notes( $entry['id'] );
		$notes_count = $all_notes ? count( $all_notes ) : 0;

		wp_send_json_success(
			[
				'notes_count' => $notes_count,
			]
		);
	}

	/**
	 * Get notes for an entry. Replacement for GFAPI::get_notes to ensure our formatting.
	 *
	 * @since 1.0.0
	 *
	 * @param int $entry_id The entry ID.
	 * @return array Array of formatted notes.
	 */
	public static function get_notes( $entry_id ) {
		$notes = \GFAPI::get_notes(
			[
				'entry_id'  => $entry_id,
				'note_type' => self::NOTE_TYPE,
			]
		);

		if ( empty( $notes ) ) {
			return [];
		}

		$notes = array_filter(
			$notes,
			function ( $note ) {
				return ! empty( $note->id );
			}
		);

		return $notes;
	}

	/**
	 * Add a note to an entry.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $entry_id The entry ID.
	 * @param int    $user_id The user ID.
	 * @param string $user_name The user name.
	 * @param string $note The note content.
	 * @param string $note_type The note type.
	 * @param int    $feed_id The feed ID.
	 * @return array Array of formatted notes.
	 */
	public static function add_note( $entry_id, $user_id, $user_name, $note, $note_type = self::NOTE_TYPE, $feed_id = null ) {
		$sub_type = self::get_note_sub_type( $feed_id );
		return \GFAPI::add_note( $entry_id, $user_id, $user_name, $note, $note_type, $sub_type );
	}

	/**
	 * Get the sub-type for a note.
	 *
	 * @since 1.0.0
	 *
	 * @param int $feed_id The feed ID.
	 *
	 * @return string|null The sub-type.
	 */
	private static function get_note_sub_type( $feed_id ) {
		return $feed_id ? 'gravityboard_feed_' . $feed_id : null;
	}

	/**
	 * Get note counts for multiple entries in a single query.
	 *
	 * @since 1.0
	 *
	 * @param array $entry_ids Array of entry IDs to get note counts for.
	 * @param int   $feed_id The feed ID.
	 * @return array Associative array of entry_id => count.
	 */
	public static function get_entry_notes_counts( $entry_ids, $feed_id ) {
		global $wpdb;

		if ( empty( $entry_ids ) ) {
			return [];
		}

		static $note_counts = [];

		if ( ! empty( $note_counts[ $feed_id ] ) ) {
			// Check if all requested entry_ids are in the cache for this feed_id.
			$all_present = true;
			foreach ( $entry_ids as $eid ) {
				if ( ! isset( $note_counts[ $feed_id ][ $eid ] ) ) {
					$all_present = false;
					break;
				}
			}
			if ( $all_present ) {
				return $note_counts[ $feed_id ];
			}
		}

		// Sanitize the entry IDs array.
		$entry_ids = array_map( 'intval', $entry_ids );

		// Get the notes table name.
		$notes_table = \GFFormsModel::get_entry_notes_table_name();

		/**
		 * Filter whether to filter notes by feed.
		 *
		 * @since 1.0
		 *
		 * @param bool $filter_notes_by_feed Whether to filter notes by feed.
		 * @param int $feed_id The feed ID.
		 */
		$filter_notes_by_board = apply_filters( 'gk/gravityboard/notes/filter-by-board', false, $feed_id );

		$and_sub_type = '';
		if ( $filter_notes_by_board ) {
			$sub_type = self::get_note_sub_type( $feed_id );
			if ( $sub_type ) {
				$and_sub_type = $wpdb->prepare( ' AND sub_type = %s', $sub_type );
			}
		}

		// Convert array to comma-separated string for IN clause - we've already sanitized above.
		$entry_ids_str = implode( ',', $entry_ids );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- This is a prepared statement.
		$query = $wpdb->prepare(
			"SELECT entry_id, COUNT(*) as note_count
			FROM {$notes_table}
			WHERE entry_id IN ({$entry_ids_str})
			AND note_type = %s
			{$and_sub_type}
			GROUP BY entry_id",
			self::NOTE_TYPE
		);

		// @phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$results = $wpdb->get_results( $query );

		// Initialize counts for all requested entry_ids to 0.
		$counts = array_fill_keys( $entry_ids, 0 );

		// Convert results to associative array.
		foreach ( $results as $result ) {
			$counts[ $result->entry_id ] = (int) $result->note_count;
		}

		if ( ! isset( $note_counts[ $feed_id ] ) ) {
			$note_counts[ $feed_id ] = [];
		}
		// Merge new counts with existing cached counts for the feed_id.
		$note_counts[ $feed_id ] = array_merge( $note_counts[ $feed_id ], $counts );

		return $counts;
	}
}
