<?php
/**
 * Handles notifications integration with Gravity Forms.
 *
 * @since 1.0
 *
 * @package GravityKit\GravityBoard
 */

namespace GravityKit\GravityBoard;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Assignees\Assignees;
use GravityKit\GravityBoard\Helpers;
use GFCommon;
use GFAPI;

/**
 * Handles notifications integration with Gravity Forms.
 *
 * @since 1.0
 *
 * @package GravityKit\GravityBoard
 */
class Notifications {


	/**
	 * Instance of this class.
	 *
	 * @since 1.0
	 * @var Notifications
	 */
	private static $instance = null;

	/**
	 * The current feed being processed.
	 *
	 * @since 1.0
	 * @var array
	 */
	private static $current_feed = [];

	/**
	 * The current entry being processed.
	 *
	 * @since 1.0
	 * @var array
	 */
	private static $current_entry = [];

	/**
	 * The suffix appended to the hook used for Gravity Forms notifications.
	 *
	 * We register our own notification events in GF. We want to use the same hook name for the notification,
	 * but we don't want to trigger the notification every time the hook is called.
	 * So we separately call do_action and listen for a separate action to trigger the notification:
	 * the same action with `/notification` appended to it.
	 *
	 * Example: `gk/gravityboard/lane/deleted`, the action, becomes `gk/gravityboard/lane/deleted/notification`,
	 * the notification event.
	 *
	 * @since 1.0
	 */
	const NOTIFICATION_HOOK_SUFFIX = '/notification';

	/**
	 * The custom input type slug for the feed conditional logic field.
	 *
	 * @since 1.0.0
	 */
	const FEED_CONDITIONAL_LOGIC_INPUT_TYPE = 'gravityboard';

	/**
	 * The custom input type slug for the assignee conditional logic field.
	 *
	 * @since 1.0.0
	 */
	const ASSIGNEE_CONDITIONAL_LOGIC_INPUT_TYPE = 'gravityboard_assignee';

	/**
	 * The custom input type slug for the created by conditional logic field.
	 *
	 * @since 1.0.0
	 */
	const CREATED_BY_CONDITIONAL_LOGIC_INPUT_TYPE = 'created_by';

	/**
	 * Constructor for the Notifications class.
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	public function __construct() {
		// Add general board notification events and hooks.
		add_filter( 'gform_notification_events', [ $this, 'add_notification_events' ], 10, 2 );

		// Enqueue conditional logic scripts.
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_conditional_logic_scripts' ] );

		// Add filter for custom conditional logic evaluation.
		add_filter( 'gform_is_value_match', [ $this, 'filter_gform_is_value_match' ], 10, 6 );
	}

	/**
	 * Enqueues conditional logic scripts and localizes data.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function enqueue_conditional_logic_scripts() {

		$form_settings_page = sprintf( 'form_settings_%s', Feed::get_instance()->get_slug() );

		if ( ! in_array( \GFForms::get_page(), [ 'notification_edit', $form_settings_page ], true ) ) {
			return;
		}

		$form_id = rgget( 'id' );
		$form    = $form_id ? Helpers::get_form( $form_id ) : null;

		if ( ! $form ) {
			return; // No form context.
		}

		$script_path      = 'assets/js/admin-conditional-logic.js';
		$full_script_path = GRAVITYBOARD_PATH . $script_path;

		wp_register_script(
			'gravityboard_admin_conditional_logic_script',
			plugins_url( $script_path, GRAVITYBOARD_FILE ),
			[ 'gform_conditional_logic' ],
			filemtime( $full_script_path ),
			true
		);

		if ( 'notification_edit' === \GFForms::get_page() ) {
			$localizations = $this->get_conditional_logic_localization_notifications( $form );
		} else {
			$localizations = $this->get_conditional_logic_localization_form_settings();
		}

		wp_localize_script(
			'gravityboard_admin_conditional_logic_script',
			'gravityBoardConditionalLogic',
			$localizations
		);

		wp_print_scripts( 'gravityboard_admin_conditional_logic_script' );
	}

	/**
	 * Gathers data needed for conditional logic JS.
	 *
	 * @since 1.0
	 *
	 * @return array An array of localized data objects, one for each eligible feed.
	 */
	private function get_conditional_logic_localization_form_settings() {

		$localized_data = [
			'fields'  => [
				'feed'      => [
					'label'     => esc_html__( 'GravityBoard Feed', 'gk-gravityboard' ),
					'inputType' => self::FEED_CONDITIONAL_LOGIC_INPUT_TYPE,
				],
				'assignee'  => [
					'label'     => esc_html__( 'GravityBoard Assignee', 'gk-gravityboard' ),
					'inputType' => self::ASSIGNEE_CONDITIONAL_LOGIC_INPUT_TYPE,
				],
				'createdBy' => [
					'label'     => esc_html__( 'Created By', 'gk-gravityboard' ),
					'inputType' => self::CREATED_BY_CONDITIONAL_LOGIC_INPUT_TYPE,
				],
			],
			'feeds'   => [],
			'users'   => [
				[
					'value' => 'current_user',
					'text'  => esc_html__( 'Current User', 'gk-gravityboard' ),
				],
			],
			'strings' => [
				'is'    => esc_html__( 'is', 'gk-gravityboard' ),
				'isnot' => esc_html__( 'is not', 'gk-gravityboard' ),
			],
		];

		return $localized_data;
	}

	/**
	 * Gathers data needed for conditional logic JS.
	 * Each feed with assignees enabled will produce a separate conditional logic field.
	 *
	 * @since 1.0
	 *
	 * @param array $form The current form object.
	 *
	 * @return array An array of localized data objects, one for each eligible feed.
	 */
	private function get_conditional_logic_localization_notifications( $form ) {
		$localized_data = [];

		/** @var GravityKit\GravityBoard\Feed $feed_instance */
		$feed_instance = Feed::get_instance();

		// Assuming get_active_feeds exists and returns active feeds for the form_id.
		$feeds = $feed_instance->get_active_feeds( $form['id'], true );

		if ( empty( $feeds ) ) {
			return $localized_data; // Return empty array if no active feeds.
		}

		$formatted_users = [
			[
				'value' => '',
				'text'  => esc_html__( 'Empty (Card has no assignees)', 'gk-gravityboard' ),
			],
		];

		$feed_list = [];
		foreach ( $feeds as $feed ) {

			// Ensure feed meta exists and has expected structure.
			if ( ! isset( $feed['id'], $feed['meta'], $feed['meta']['feed_name'] ) ) {
				continue;
			}

			$feed_list[] = [
				'id'   => $feed['id'],
				'name' => $feed['meta']['feed_name'],
			];

			// Assuming get_feed_setting_value exists. Check if assignees are enabled for this feed.
			$assignees_enabled = $feed_instance->get_setting( 'enable_assignees', '1', $feed );

			if ( ! $assignees_enabled ) {
				continue;
			}

			// Assuming get_assignable_users exists. Get assignable users for this feed.
			/** @see GravityKit\GravityBoard\Feed::get_assignable_users() */
			$users = $feed_instance->get_assignable_users( $feed );

			foreach ( $users as $user ) {
				if ( ! isset( $user['id'], $user['text'] ) ) {
					continue;
				}

				$formatted_users[ $user['id'] ] = [
					'value' => (int) $user['id'],
					'text'  => esc_html( $user['text'] ),
				];
			}
		}

		$localized_data = [
			'fields'  => [
				'feed'     => [
					'label'     => esc_html__( 'GravityBoard Feed', 'gk-gravityboard' ),
					'inputType' => self::FEED_CONDITIONAL_LOGIC_INPUT_TYPE,
				],
				'assignee' => [
					'label'     => esc_html__( 'GravityBoard Assignee', 'gk-gravityboard' ),
					'inputType' => self::ASSIGNEE_CONDITIONAL_LOGIC_INPUT_TYPE,
				],
			],
			'feeds'   => $feed_list,
			'users'   => array_values( $formatted_users ),
			'strings' => [
				'is'    => esc_html__( 'is', 'gk-gravityboard' ),
				'isnot' => esc_html__( 'is not', 'gk-gravityboard' ),
			],
		];

		return $localized_data;
	}

	/**
	 * Add notification events for the GravityBoard.
	 *
	 * Each of these events has a corresponding do_action hook.
	 *
	 * The problem is that we don't want to always trigger the notification when the action is called.
	 * So we separately call do_action and listen for a separate action to trigger the notification:
	 * the same action with `/notification` appended to it.
	 *
	 * See Notifications::maybe_trigger_notifications().
	 *
	 * @since 1.0
	 *
	 * @param array $notification_events The existing notification events.
	 *
	 * @return array The updated notification events
	 */
	public function add_notification_events( $notification_events = [] ) {
		return array_merge(
			$notification_events,
			[
				'gk/gravityboard/card/added' . self::NOTIFICATION_HOOK_SUFFIX          => 'GravityBoard - ' . esc_html__( 'Card Added', 'gk-gravityboard' ),
				'gk/gravityboard/card/edited' . self::NOTIFICATION_HOOK_SUFFIX         => 'GravityBoard - ' . esc_html__( 'Card Edited', 'gk-gravityboard' ),
				'gk/gravityboard/card/deleted' . self::NOTIFICATION_HOOK_SUFFIX        => 'GravityBoard - ' . esc_html__( 'Card Deleted', 'gk-gravityboard' ),
				'gk/gravityboard/card/sorted' . self::NOTIFICATION_HOOK_SUFFIX         => 'GravityBoard - ' . esc_html__( 'Card Sorted', 'gk-gravityboard' ),
				'gk/gravityboard/card/changed-lane' . self::NOTIFICATION_HOOK_SUFFIX   => 'GravityBoard - ' . esc_html__( 'Card Changed Lane', 'gk-gravityboard' ),
				'gk/gravityboard/lane/added' . self::NOTIFICATION_HOOK_SUFFIX          => 'GravityBoard - ' . esc_html__( 'Lane Added', 'gk-gravityboard' ),
				'gk/gravityboard/lane/edited' . self::NOTIFICATION_HOOK_SUFFIX         => 'GravityBoard - ' . esc_html__( 'Lane Edited', 'gk-gravityboard' ),
				'gk/gravityboard/lane/deleted' . self::NOTIFICATION_HOOK_SUFFIX        => 'GravityBoard - ' . esc_html__( 'Lane Deleted', 'gk-gravityboard' ),
				'gk/gravityboard/lane/moved' . self::NOTIFICATION_HOOK_SUFFIX          => 'GravityBoard - ' . esc_html__( 'Lane Moved', 'gk-gravityboard' ),
				'gk/gravityboard/assignees/added' . self::NOTIFICATION_HOOK_SUFFIX     => 'GravityBoard - ' . esc_html__( 'Assignee(s) Added', 'gk-gravityboard' ),
				'gk/gravityboard/assignees/removed' . self::NOTIFICATION_HOOK_SUFFIX   => 'GravityBoard - ' . esc_html__( 'Assignee(s) Removed', 'gk-gravityboard' ),
				'gk/gravityboard/assignees/updated' . self::NOTIFICATION_HOOK_SUFFIX   => 'GravityBoard - ' . esc_html__( 'Assignee(s) Updated', 'gk-gravityboard' ),
				'gk/gravityboard/attachments/added' . self::NOTIFICATION_HOOK_SUFFIX   => 'GravityBoard - ' . esc_html__( 'Attachment(s) Added', 'gk-gravityboard' ),
				'gk/gravityboard/attachments/removed' . self::NOTIFICATION_HOOK_SUFFIX => 'GravityBoard - ' . esc_html__( 'Attachment(s) Removed', 'gk-gravityboard' ),
				'gk/gravityboard/attachments/updated' . self::NOTIFICATION_HOOK_SUFFIX => 'GravityBoard - ' . esc_html__( 'Attachment(s) Updated', 'gk-gravityboard' ),
			]
		);
	}

	/**
	 * Custom conditional logic evaluation for GravityBoard fields.
	 *
	 * @since 1.0.0
	 *
	 * @param bool   $is_match           Whether the rule is a match or not.
	 * @param mixed  $field_value        The current field value from the entry.
	 * @param mixed  $target_value       The target value specified in the rule.
	 * @param string $operation          The comparison operator (is, isnot, etc.).
	 * @param mixed  $source_field       The source field object or ID.
	 * @param array  $rule               The conditional logic rule being evaluated.
	 *
	 * @return bool Whether the custom field value matches the rule.
	 */
	public function filter_gform_is_value_match( $is_match, $field_value, $target_value, $operation, $source_field, $rule ) {
		// Only process our custom field types.
		if ( empty( $rule['fieldId'] ) || ! in_array(
			$rule['fieldId'],
			[
				self::CREATED_BY_CONDITIONAL_LOGIC_INPUT_TYPE,
				self::FEED_CONDITIONAL_LOGIC_INPUT_TYPE,
				self::ASSIGNEE_CONDITIONAL_LOGIC_INPUT_TYPE,
			],
			true
		) ) {
			return $is_match;
		}

		// Basic safety check - if we're missing context for our custom logic, return false.
		if ( empty( self::$current_entry ) && empty( self::$current_feed ) ) {
			Feed::get_instance()->log_error( 'GravityBoard: Missing current entry/feed context for conditional logic evaluation' );
			return false;
		}

		$custom_value = null;

		switch ( $rule['fieldId'] ) {
			case self::CREATED_BY_CONDITIONAL_LOGIC_INPUT_TYPE:
				// Get the entry creator's user ID.
				$custom_value = get_current_user_id();
				break;

			case self::FEED_CONDITIONAL_LOGIC_INPUT_TYPE:
				// Get the feed ID.
				$custom_value = rgar( self::$current_feed, 'id', null );
				break;

			case self::ASSIGNEE_CONDITIONAL_LOGIC_INPUT_TYPE:
				// Only process if we have a current entry.
				if ( empty( self::$current_entry['id'] ) ) {
					Feed::get_instance()->log_error( 'GravityBoard: Missing current entry for conditional logic evaluation' );
					return false;
				}

				// Get all assignees for the current entry.
				$assignees = Assignees::get_assignees( self::$current_entry['id'], false );

				// Check if the target_value (user ID) is in the assignees list.
				if ( 'is' === $operation ) {
					return in_array( (int) $target_value, $assignees, true );
				} else {
					return ! in_array( (int) $target_value, $assignees, true );
				}
				break;
		}

		// For other field types, do a simple comparison.
		if ( 'is' === $operation ) {
			return $custom_value === $target_value;
		} else {
			return $custom_value !== $target_value;
		}
	}

	/**
	 * Sends notifications for GravityBoard events
	 *
	 * @since 1.0
	 *
	 * @param array  $entry_or_lane The entry or lane array.
	 * @param string $hook The hook to trigger.
	 * @param array  $feed The feed object.
	 *
	 * @return void
	 */
	public function maybe_trigger_notifications( $entry_or_lane = [], $hook = 'gk/gravityboard/card/added', $feed = [] ) {
		if ( empty( $feed ) ) {
			return;
		}

		$trigger_notifications = rgars( $feed, 'meta/trigger_notifications', '1' ) === '1';

		if ( empty( $trigger_notifications ) ) {
			return;
		}

		$form = Helpers::get_form( $feed['form_id'] );

		if ( empty( $form ) ) {
			return;
		}

		$notification_event = $hook . self::NOTIFICATION_HOOK_SUFFIX;

		self::$current_feed  = $feed;
		self::$current_entry = $entry_or_lane;

		\GFAPI::send_notifications(
			$form,
			$entry_or_lane,
			$notification_event,
			[
				'current_feed'  => $feed,
				'current_entry' => $entry_or_lane,
			]
		);

		self::$current_feed  = [];
		self::$current_entry = [];
	}

	/**
	 * Get the instance of the Notifications class.
	 *
	 * @since 1.0
	 *
	 * @return Notifications The instance of the Notifications class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Sends email notifications to newly mentioned users.
	 *
	 * @since 1.0
	 *
	 * @param int[]  $newly_mentioned_user_ids Array of user IDs.
	 * @param int    $entry_id The ID of the entry the note belongs to.
	 * @param int    $note_id The ID of the note.
	 * @param string $note_content_raw The raw content of the note.
	 * @param array  $feed The GravityBoard feed.
	 * @param string $site_path The current site path.
	 *
	 * @return void
	 */
	public static function send_mention_notifications( $newly_mentioned_user_ids, $entry_id, $note_id, $note_content_raw, $feed, $site_path ) {
		if ( empty( $newly_mentioned_user_ids ) || empty( $entry_id ) || empty( $feed ) ) {
			return;
		}

		// Ensure feed exists and mention notifications are enabled for this feed.
		if ( ! $feed || '1' !== rgars( $feed, 'meta/enable_mention_notifications', '1' ) ) {
			return;
		}

		$feed_id = rgars( $feed, 'id', 0 );
		$entry   = Helpers::get_card( $entry_id );

		if ( ! $entry || is_wp_error( $entry ) ) {
			Feed::get_instance()->log_error( 'Error getting entry for sending notifications: ' . $entry_id . ' - ' . $entry->get_error_message() );
			return false;
		}

		$form_id = rgars( $feed, 'form_id', 0 );
		$form    = $form_id ? Helpers::get_form( $form_id ) : null;

		if ( ! $form ) {
			// Log error or handle missing form/entry if necessary.
			// For example: error_log( 'GravityBoard: Missing form or entry for mention notification. Form ID: ' . $form_id . ', Entry ID: ' . $entry_id );.
			return false;
		}

		$current_user  = wp_get_current_user();
		$notifier_name = $current_user && $current_user->exists() ? $current_user->display_name : esc_html__( 'Someone', 'gk-gravityboard' );

		$note_excerpt_plain = preg_replace_callback(
			Helpers::MENTION_REGEX,
			function ( $matches ) {
				return '@' . esc_html( $matches[1] );
			},
			$note_content_raw
		);
		$note_excerpt_plain = wp_strip_all_tags( $note_excerpt_plain );
		$note_excerpt_plain = wp_trim_words( $note_excerpt_plain, 50, '...' );

		$feed_name = rgars( $feed, 'meta/feed_name', '' );
		if ( empty( $feed_name ) ) {
			// translators: %1$s is Form Title, %2$d is Feed ID.
			$feed_name = sprintf( esc_html__( '%1$s Board #%2$d', 'gk-gravityboard' ), $form['title'], $feed_id );
		}

		$subject_template = rgars( $feed, 'meta/mention_notification_subject', '' );
		$body_template    = rgars( $feed, 'meta/mention_notification_body', '' );

		if ( empty( $subject_template ) || empty( $body_template ) ) {
			Feed::get_instance()->log_error( 'GravityBoard: Missing subject or body template for mention notification. Feed ID: ' . $feed_id );
			return;
		}

		foreach ( $newly_mentioned_user_ids as $user_id ) {
			$user_to_notify = get_userdata( $user_id );

			if ( ! $user_to_notify || empty( $user_to_notify->user_email ) ) {
				continue;
			}

			// Don't notify the user if they mentioned themselves.
			if ( $current_user->ID === $user_to_notify->ID ) {
				continue;
			}

			$gf_entry_link = '';
			// Check if the mentioned user has permission to view this specific entry.
			if ( Helpers::user_has_permission( 'view_board', $feed, $user_to_notify->ID ) ) {
				$gf_entry_link = esc_url_raw( admin_url( 'admin.php?page=gf_entries&view=entry&id=' . $form_id . '&lid=' . $entry_id ) . '#:~:text=' . rawurlencode( $note_content_raw ) );
			}

			if ( ! Helpers::user_has_permission( 'manage_entry_notes', $feed, $user_to_notify->ID ) ) {
				$board_card_link = '';
			} else {
				$board_card_link = esc_url_raw( site_url( $site_path ) . '#card=' . $entry_id );
			}

			// Custom merge tags specific to this notification context.
			$replacement_tags = [
				'{board_name}'          => $feed_name,
				'{notifier_name}'       => $notifier_name,
				'{mentioned_user_name}' => $user_to_notify->display_name,
				'{note_id}'             => $note_id,
				'{note_excerpt}'        => $note_excerpt_plain,
				'{entry_details_url}'   => $gf_entry_link ? $gf_entry_link : esc_html__( '[not available due to permissions]', 'gk-gravityboard' ),
				'{board_card_link}'     => $board_card_link ? $board_card_link : esc_html__( '[not available due to permissions]', 'gk-gravityboard' ),
			];

			$subject_template = str_replace( array_keys( $replacement_tags ), array_values( $replacement_tags ), $subject_template );
			$body_template    = str_replace( array_keys( $replacement_tags ), array_values( $replacement_tags ), $body_template );

			$body_template = apply_filters( 'the_content', $body_template );
			$body_template = make_clickable( $body_template );
			$body_template = Helpers::render_mentions_as_html( $body_template );

			$subject = GFCommon::replace_variables( $subject_template, $form, $entry, false, true, false, 'text' );
			$body    = GFCommon::replace_variables( $body_template, $form, $entry, false, true, false, 'html' );

			GFCommon::send_email(
				get_option( 'admin_email' ), // from.
				$user_to_notify->user_email, // to.
				'', // bcc.
				'', // reply_to.
				$subject, // subject.
				$body, // message.
				get_option( 'blogname' ), // from_name.
				'html', // Message format.
				'', // attachments.
				$entry, // entry.
				false, // notification.
				null // cc.
			);
		}
	}
}
