<?php
/**
 * GravityBoard Renderer
 *
 * @package GravityKit
 * @subpackage GravityBoard
 * @since 1.0
 */

namespace GravityKit\GravityBoard;

use GravityKit\GravityBoard\Helpers;
use GFAPI;
use GFForms;

use function add_shortcode;
use function wp_doing_ajax;
use function shortcode_atts;
use function esc_html__;
use function esc_attr;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Renderer {

	const SHORTCODE_NAME = 'gravityboard';

	const CSS_ASSET_NAME = 'gravityboard-app-styles';

	/**
	 * Singleton instance.
	 *
	 * @var Renderer|null
	 */
	private static $instance = null;

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_shortcode( self::SHORTCODE_NAME, [ $this, 'render_shortcode' ] );
	}

	/**
	 * Get singleton instance.
	 *
	 * @since 1.0
	 *
	 * @return Renderer
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Render the kanban board via shortcode.
	 *
	 * @since 1.0
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string HTML output for the kanban board.
	 */
	public function render_shortcode( $atts ) {

		// Bail early when in the admin area or during an AJAX request (follow WordPress conventions).
		if ( wp_doing_ajax() ) {
			return '';
		}

		static $rendered = false;

		$attributes = shortcode_atts(
            [
				'id' => '',
			],
			$atts
        );

		$feed_id    = (int) $attributes['id'];
		$feed_addon = Feed::get_instance();
		$feed       = $feed_addon->get_feed( $feed_id );

		if ( empty( $feed ) ) {
			return $this->render_error_message( esc_html__( 'GravityBoard feed not found.', 'gk-gravityboard' ) );
		}

		if ( empty( $feed['is_active'] ) ) {
			return $this->render_error_message( esc_html__( 'Board is not active.', 'gk-gravityboard' ) );
		}

		if ( ! $feed_addon->user_has_permission( 'view_board', $feed ) ) {
			return $this->render_error_message( esc_html__( 'You do not have permission to view this board.', 'gk-gravityboard' ) );
		}

		if ( $rendered ) {
			if ( Helpers::user_has_permission( 'manage_options', $feed ) ) {
				// translators: %s is the email for GravityKit support.
				return '<div class="gravityboard-error">' . strtr(
                    esc_html__( 'Admin-only notice: Only one board can be rendered per page. If you need this functionality, please [url]email us and let us know[/url].', 'gk-gravityboard' ),
                    [
						'[url]'  => '<a href="mailto:support@gravitykit.com?subject=GravityBoard%20Multiple%20Boards%20on%20Page%20Request&body=I%20need%20to%20render%20multiple%20boards%20on%20a%20page!">',
						'[/url]' => '</a>',
					]
                ) . '</div>';
			}

			return '';
		}

		$form = $feed_addon->get_form( $feed );
		if ( empty( $form ) || is_wp_error( $form ) ) {
			$error_message = is_wp_error( $form ) ? $form->get_error_message() : esc_html__( 'Form not found.', 'gk-gravityboard' );
			return $this->render_error_message( $error_message );
		}

		$rendered = true;

		$this->enqueue_styles( $feed_id );
		$this->enqueue_scripts( $feed_id );

		return '<div id="gravityboard" class="react-gravityboard-container" data-feed-id="' . esc_attr( $feed_id ) . '">
			<div class="gravityboard-loading">
				<div class="gravityboard-loading-content">
					<div class="gravityboard-loading-icon">
						<svg width="48" height="48" viewBox="0 0 47 44" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
							<path fill-rule="evenodd" clip-rule="evenodd" d="M20.7971 36.6666V38.5C20.7971 41.5376 18.3049 44 15.2691 44H5.52806C2.45215 44 0 41.5376 0 38.5L0 11.0001L0 7.33332L0 5.50006C0 2.46246 2.45215 6.10352e-05 5.52806 6.10352e-05L15.2691 6.10352e-05C18.3049 6.10352e-05 20.7971 2.46246 20.7971 5.50006L20.7971 33.0001V36.6666ZM15.2691 3.66673C16.2691 3.66673 17.1115 4.48745 17.1115 5.50006L17.1115 38.5C17.1115 39.5125 16.2691 40.3334 15.2691 40.3334H5.52806C4.48796 40.3334 3.68563 39.5125 3.68563 38.5L3.68563 5.50006C3.68563 4.48745 4.48796 3.66673 5.52806 3.66673L15.2691 3.66673Z" fill="currentColor"/>
							<path fill-rule="evenodd" clip-rule="evenodd" d="M26 28.8333L26 27C26 23.9624 28.4922 21.5 31.5281 21.5L41.2691 21.5C44.345 21.5 46.7971 23.9624 46.7971 27L46.7971 26L46.7971 29.6667L46.7971 38.5C46.7971 41.5376 44.345 44 41.2691 44L31.5281 44C28.4922 44 26 41.5376 26 38.5L26 39.4999L26 28.8333ZM31.5281 40.3333C30.528 40.3333 29.6856 39.5126 29.6856 38.5L29.6856 27C29.6856 25.9875 30.528 25.1666 31.5281 25.1666L41.2691 25.1666C42.3092 25.1666 43.1115 25.9875 43.1115 27L43.1115 38.5C43.1115 39.5126 42.3092 40.3333 41.2691 40.3333L31.5281 40.3333Z" fill="currentColor"/>
							<path fill-rule="evenodd" clip-rule="evenodd" d="M46.7971 10.1667V12C46.7971 15.0376 44.3049 17.5 41.2691 17.5L31.5281 17.5C28.4521 17.5 26 15.0376 26 12V13L26 9.33326V5.5C26 2.4624 28.4521 0 31.5281 0L41.2691 0C44.3049 0 46.7971 2.4624 46.7971 5.5V4.50008V10.1667ZM41.2691 3.66667C42.2691 3.66667 43.1115 4.48739 43.1115 5.5V12C43.1115 13.0125 42.2691 13.8334 41.2691 13.8334L31.5281 13.8334C30.488 13.8334 29.6856 13.0125 29.6856 12V5.5C29.6856 4.48739 30.488 3.66667 31.5281 3.66667L41.2691 3.66667Z" fill="currentColor"/>
						</svg>
					</div>
					<div class="gravityboard-loading-text">
						<h3>' . esc_html__( 'Loading GravityBoard…', 'gk-gravityboard' ) . '</h3>
					</div>
				</div>' . $this->get_admin_troubleshooting_content( $feed ) . '
			</div>
		</div>';
	}

	/**
	 * Enqueue the GravityBoard React app and its dependencies.
	 *
	 * @since 1.0
	 *
	 * @param int $feed_id The ID of the feed.
	 *
	 * @return void
	 */
	public function enqueue_scripts( $feed_id = 0 ) {

		$settings = Feed::get_instance()->get_settings_for_script( $feed_id );

		wp_enqueue_script(
			'gravityboard-app',
			plugins_url( '/gravityboard-app/build/gravityboard.js', GRAVITYBOARD_FILE ),
			[
				'wp-element',
				'wp-i18n',
				'jquery',
				'wp-api-fetch',
			],
			filemtime( plugin_dir_path( GRAVITYBOARD_FILE ) . 'gravityboard-app/build/gravityboard.js' ),
			true
		);

		/**
		 * Filters the translation overrides for the GravityBoard.
		 *
		 * @since 1.0
		 *
		 * @param array $translation_overrides The translation overrides.
		 * @param int   $feed_id The ID of the feed.
		 *
		 * @return array The translation overrides, with each key being the original string and the value being an array of strings to replace it with.
		 */
		$translation_overrides = apply_filters( 'gk/gravityboard/translation-overrides', [], $feed_id );

		// Add inline script to override translations directly in JavaScript.
		if ( ! empty( $translation_overrides ) ) {
			$inline_script = "
				if ( typeof wp !== 'undefined' && wp.i18n && wp.i18n.setLocaleData ) {
					wp.i18n.setLocaleData( " . wp_json_encode( $translation_overrides ) . ", 'gk-gravityboard' );
				}
			";
			wp_add_inline_script( 'gravityboard-app', $inline_script, 'before' );
		}

		// Localize script with settings - this will make the settings available via window.gravityBoardSettings.
		wp_localize_script( 'gravityboard-app', 'gravityBoardSettings', $settings );
	}

	/**
	 * Enqueue the styles and inline overrides.
	 *
	 * @since 1.0
	 *
	 * @param int $feed_id The ID of the feed.
	 *
	 * @return void
	 */
	public function enqueue_styles( $feed_id = 0 ) {

		$css_path      = '/gravityboard-app/build/gravityboard.css';
		$css_url       = plugins_url( $css_path, GRAVITYBOARD_FILE );
		$css_file_time = filemtime( plugin_dir_path( GRAVITYBOARD_FILE ) . $css_path );

		wp_enqueue_style( self::CSS_ASSET_NAME, $css_url, [], $css_file_time );

		// Get the feed to access its settings.
		$feed = Feed::get_instance()->get_feed( $feed_id );

		$background_image_id  = rgars( $feed, 'meta/board_background_image_id' );
		$background_image_url = $background_image_id ? wp_get_attachment_image_url( $background_image_id, 'full' ) : null;

		// Fallback to background color if no image.
		$background_color = rgars( $feed, 'meta/board_background_color', '' );

		$css_overrides = [];
		if ( ! empty( $background_color ) ) {
			$css_overrides['--gravityboard-board-background-color'] = sprintf( '%s !important', esc_attr( $background_color ) );
		}

		if ( $background_image_url ) {
			$css_overrides['--gravityboard-board-background-image'] = sprintf( 'url(%1$s) !important', esc_url( $background_image_url ) );
		}

		// Add minimum height if set.
		$min_height = rgars( $feed, 'meta/board_min_height', '' );
		if ( ! empty( $min_height ) ) {
			$css_overrides['--gravityboard-min-height'] = sprintf( '%s !important', esc_attr( $min_height ) );
		}

		// Add lane background colors as CSS variables.
		$lane_colors = rgars( $feed, 'meta/lane_colors', [] );
		if ( ! empty( $lane_colors ) && is_array( $lane_colors ) ) {
			$i = 0;
			foreach ( $lane_colors as $lane_value => $color ) {
				++$i;
				if ( ! empty( $color ) ) {
					// Sanitize the lane value for use in CSS variable names.
					// This should match the sanitizeForCss function in JavaScript.
					$sanitized_lane_value = strtolower( preg_replace( '/[^a-z0-9-]/i', '', str_replace( ' ', '-', $lane_value ) ) );
					$css_overrides[ '--gravityboard-lane-bg-' . $sanitized_lane_value ] = sprintf( '%s !important', esc_attr( $color ) );
				}
			}
		}

		/**
		 * Filters the global overrides for the GravityBoard.
		 *
		 * @since 1.0
		 *
		 * @param array $css_overrides The global overrides, with the key as the full CSS variable name and the value as the CSS value.
		 * @param array $feed The feed.
		 *
		 * @return array The global overrides.
		 */
		$css_overrides = apply_filters( 'gk/gravityboard/renderer/css-variable-overrides', $css_overrides, $feed );

		// Add loading screen and error styling.
		$loading_styles = $this->get_loading_styles();

		if ( ! empty( $css_overrides ) ) {

			$inline_style = ':root { ';
			foreach ( $css_overrides as $key => $value ) {
				$inline_style .= $key . ': ' . $value . ';';
			}
			$inline_style .= ' }';

			$inline_style .= $loading_styles;

			wp_add_inline_style( self::CSS_ASSET_NAME, $inline_style );
		} else {
			wp_add_inline_style( self::CSS_ASSET_NAME, $loading_styles );
		}
	}

	/**
	 * Render an error message with optional admin troubleshooting content.
	 *
	 * @since 1.0
	 *
	 * @param string $message The error message to display.
	 *
	 * @return string HTML output for the error message.
	 */
	private function render_error_message( $message ) {
		return '<div class="gravityboard-error">
			<div class="gravityboard-error-content">
				<div class="gravityboard-error-icon">
					<svg width="24" height="24" viewBox="0 0 47 44" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
						<path fill-rule="evenodd" clip-rule="evenodd" d="M20.7971 36.6666V38.5C20.7971 41.5376 18.3049 44 15.2691 44H5.52806C2.45215 44 0 41.5376 0 38.5L0 11.0001L0 7.33332L0 5.50006C0 2.46246 2.45215 6.10352e-05 5.52806 6.10352e-05L15.2691 6.10352e-05C18.3049 6.10352e-05 20.7971 2.46246 20.7971 5.50006L20.7971 33.0001V36.6666ZM15.2691 3.66673C16.2691 3.66673 17.1115 4.48745 17.1115 5.50006L17.1115 38.5C17.1115 39.5125 16.2691 40.3334 15.2691 40.3334H5.52806C4.48796 40.3334 3.68563 39.5125 3.68563 38.5L3.68563 5.50006C3.68563 4.48745 4.48796 3.66673 5.52806 3.66673L15.2691 3.66673Z" fill="currentColor"/>
						<path fill-rule="evenodd" clip-rule="evenodd" d="M26 28.8333L26 27C26 23.9624 28.4922 21.5 31.5281 21.5L41.2691 21.5C44.345 21.5 46.7971 23.9624 46.7971 27L46.7971 26L46.7971 29.6667L46.7971 38.5C46.7971 41.5376 44.345 44 41.2691 44L31.5281 44C28.4922 44 26 41.5376 26 38.5L26 39.4999L26 28.8333ZM31.5281 40.3333C30.528 40.3333 29.6856 39.5126 29.6856 38.5L29.6856 27C29.6856 25.9875 30.528 25.1666 31.5281 25.1666L41.2691 25.1666C42.3092 25.1666 43.1115 25.9875 43.1115 27L43.1115 38.5C43.1115 39.5126 42.3092 40.3333 41.2691 40.3333L31.5281 40.3333Z" fill="currentColor"/>
						<path fill-rule="evenodd" clip-rule="evenodd" d="M46.7971 10.1667V12C46.7971 15.0376 44.3049 17.5 41.2691 17.5L31.5281 17.5C28.4521 17.5 26 15.0376 26 12V13L26 9.33326V5.5C26 2.4624 28.4521 0 31.5281 0L41.2691 0C44.3049 0 46.7971 2.4624 46.7971 5.5V4.50008V10.1667ZM41.2691 3.66667C42.2691 3.66667 43.1115 4.48739 43.1115 5.5V12C43.1115 13.0125 42.2691 13.8334 41.2691 13.8334L31.5281 13.8334C30.488 13.8334 29.6856 13.0125 29.6856 12V5.5C29.6856 4.48739 30.488 3.66667 31.5281 3.66667L41.2691 3.66667Z" fill="currentColor"/>
					</svg>
				</div>
				<p>' . $message . '</p>
			</div>
		</div>';
	}

	/**
	 * Get admin-specific troubleshooting content for the loading screen.
	 *
	 * @since 1.0
	 *
	 * @param array $feed The feed.
	 *
	 * @return string HTML content with admin troubleshooting instructions.
	 */
	private function get_admin_troubleshooting_content( $feed = [] ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return '';
		}

		// translators: Do not translate the placeholders in square brackets.
		$email_template = esc_html__(
            'Board Feed ID: [feed_id]
WordPress Version: [wp_version]
GravityBoard Version: [plugin_version]
Gravity Forms Version: [gf_version]
PHP Version: [php_version]

Issue Description:
(Please describe what you were trying to do when the issue occurred)

Console Errors:
(Paste any copied console errors here)

Additional Details:
(Any other relevant information)',
            'gk-gravityboard'
        );

		$email_template = strtr(
            $email_template,
            [
				'[feed_id]'        => $feed['id'],
				'[wp_version]'     => get_bloginfo( 'version' ),
				'[gf_version]'     => class_exists( 'GFForms' ) ? GFForms::$version : 'GFForms class does not exist',
				'[php_version]'    => phpversion(),
				'[plugin_version]' => GRAVITYBOARD_VERSION,
			]
        );

		// Create support email link with the full email template as body.
		$support_email_link = sprintf(
			'<a href="mailto:support@gravitykit.com?subject=%s&body=%s">support@gravitykit.com</a>',
			rawurlencode( esc_html__( 'GravityBoard Loading Issue', 'gk-gravityboard' ) ),
			rawurlencode( $email_template )
		);

		return '
			<div class="gravityboard-admin-troubleshooting">
				<details class="gravityboard-troubleshooting-details">
					<summary>' . esc_html__( 'Board not loading? (Admin Only)', 'gk-gravityboard' ) . '</summary>
					<div class="gravityboard-troubleshooting-content">
						<p><strong>' . esc_html__( 'If the board doesn\'t load, try these steps:', 'gk-gravityboard' ) . '</strong></p>
						<ol>
							<li>' . esc_html__( 'Refresh the page', 'gk-gravityboard' ) . '</li>
							<li>' . esc_html__( 'Check browser console for JavaScript errors (F12 on Windows/Linux, Cmd+Option+i on Mac)', 'gk-gravityboard' ) . '</li>
							<li>' . sprintf(
								// translators: %s is the GravityKit support email.
								esc_html__( 'Still having issues? Contact %s', 'gk-gravityboard' ),
								$support_email_link
							) . '</li>
						</ol>
						<div class="gravityboard-email-template-section">
							<h4>' . esc_html__( 'Email Template for Support', 'gk-gravityboard' ) . '</h4>
							<p>' . esc_html__( 'Copy and paste this template into your support email:', 'gk-gravityboard' ) . '</p>
							<textarea class="gravityboard-email-template" readonly onclick="this.select()">' . esc_textarea( $email_template ) . '</textarea>
						</div>
					</div>
				</details>
			</div>';
	}

	/**
	 * Get CSS styles for the loading screen and error messages.
	 *
	 * @since 1.0
	 *
	 * @return string CSS styles for loading and error states.
	 */
	private function get_loading_styles() {
		return '
		.gravityboard-loading {
			display: flex;
			flex-direction: column;
			justify-content: center;
			align-items: center;
			min-height: 300px;
			padding: 40px 20px;
			text-align: center;
			color: #525252;
		}

		.gravityboard-loading-content {
			max-width: 400px;
		}

		.gravityboard-loading-icon {
			margin-bottom: 20px;
			color: #0073aa;
			animation: gravityboard-pulse 2s ease-in-out infinite;
		}

		@media (prefers-reduced-motion: reduce) {
			.gravityboard-loading-icon {
				animation: none;
			}
		}

		.gravityboard-loading-text h3 {
			margin: 0 0 10px 0;
			font-size: 1.3em;
			font-weight: 600;
			color: #333;
		}

		.gravityboard-loading-text p {
			margin: 0;
			font-size: 1em;
			color: #666;
		}

		@keyframes gravityboard-pulse {
			0%, 100% { opacity: 1; transform: scale(1); }
			50% { opacity: 0.8; transform: scale(1.1); }
		}

		.gravityboard-admin-troubleshooting {
			margin-top: 30px;
			width: 100%;
			max-width: 500px;
			align-self: center;
			align-items: center;
			opacity: 0;
			text-align: left;
			animation: gravityboard-troubleshooting-fadein 0.5s ease-in-out 5s forwards;
		}

		@keyframes gravityboard-troubleshooting-fadein {
			from { opacity: 0; }
			to { opacity: 1; }
		}

		.gravityboard-troubleshooting-details {
			background: #f8f9fa;
			border: 1px solid #e1e5e9;
			border-radius: 6px;
			padding: 0;
		}

		.gravityboard-troubleshooting-details summary {
			padding: 12px 16px;
			cursor: pointer;
			font-weight: 600;
			color: #0073aa;
			border-radius: 6px;
			transition: background-color 0.2s ease;
		}

		.gravityboard-troubleshooting-details summary:hover {
			background-color: #f1f1f1;
		}

		.gravityboard-troubleshooting-details[open] summary {
			border-bottom: 1px solid #e1e5e9;
			border-radius: 6px 6px 0 0;
		}

		.gravityboard-troubleshooting-content {
			padding: 16px;
			text-align: left;
		}

		.gravityboard-troubleshooting-content p {
			margin: 0 0 12px 0;
		}

		.gravityboard-troubleshooting-content ol {
			margin: 0;
			padding-left: 20px;
		}

		.gravityboard-troubleshooting-content li {
			margin-bottom: 8px;
			line-height: 1.5;
		}

		.gravityboard-troubleshooting-content a {
			color: #0073aa;
			text-decoration: none;
		}

		.gravityboard-troubleshooting-content a:hover {
			color: #005a87;
			text-decoration: underline;
		}

		.gravityboard-error {
			display: flex;
			flex-direction: column;
			justify-content: center;
			align-items: center;
			min-height: 200px;
			padding: 40px 20px;
			text-align: center;
			background: #fef2f2;
			border: 1px solid #fecaca;
			border-radius: 8px;
			margin: 20px 0;
		}

		.gravityboard-error-content {
			max-width: 400px;
		}

		.gravityboard-error-icon {
			margin-bottom: 16px;
			color: #dc2626;
		}

		.gravityboard-error-content p {
			margin: 0;
			font-size: 1.1em;
			color: #991b1b;
			font-weight: 500;
		}

		.gravityboard-email-template-section {
			margin-top: 20px;
			padding: 16px;
			background: #e7f3ff;
			border: 1px solid #b3d9ff;
			border-radius: 6px;
		}

		.gravityboard-email-template-section h4 {
			margin: 0 0 8px 0;
			color: #0056b3;
			font-size: 1em;
		}

		.gravityboard-email-template-section p {
			margin: 0 0 12px 0;
			color: #495057;
			font-size: 0.9em;
		}

		.gravityboard-email-template {
			width: calc(100% - 24px);
			min-height: 200px;
			padding: 12px;
			border: 1px solid #ced4da;
			border-radius: 4px;
			font-family: monospace;
			font-size: 0.85em;
			background: #ffffff!important;
			resize: vertical;
			margin-bottom: 10px;
		}

		.gravityboard-email-template:focus {
			outline: none;
			border-color: #0073aa;
			box-shadow: 0 0 0 2px rgba(0, 115, 170, 0.2);
		}
		';
	}
}
