<?php

/**
 * Gravity Flow Add Entry Step
 *
 *
 * @package     GravityFlow
 * @subpackage  Classes/Step
 * @copyright   Copyright (c) 2015-2018, Steven Henty S.L.
 * @license     http://opensource.org/licenses/gpl-3.0.php GNU Public License
 * @since       1.0
 */

if ( class_exists( 'Gravity_Flow_Step' ) ) {

	class Gravity_Flow_Step_Folders_Add extends Gravity_Flow_Step_Folders {
		public $_step_type = 'folders';

		/**
		 * Returns the label for the step.
		 *
		 *
		 * @since 1.0
		 *
		 * @return string
		 */
		public function get_label() {
			return esc_html__( 'Add to Folder', 'gravityflowformconnector' );
		}

		/**
		 * Returns the exisiting folder settings for this step.
		 *
		 * @since 1.8
		 *
		 * @return array
		 */		
		private function get_existing_folder_settings() {
			$existing_folder_settings               = parent::get_folder_setting_field();
			$existing_folder_settings['dependency'] = array(
				'live'   => true,
				'fields' => array(
					array(
						'field'  => 'add_to_type',
						'values' => array( 'existing_folder' ),
					),
				),
			);

			return $existing_folder_settings;
		}

		/**
		 * Returns the new folder settings for this step.
		 *
		 * @since 1.8
		 *
		 * @return array
		 */		
		private function get_new_folder_settings() {
			$new_folder_settings = array(
				'name'          => 'new_folder_name',
				'class'         => 'merge-tag-support mt-hide_all_fields',
				'label'         => __( 'New Folder Name', 'gravityflowfolders' ),
				'tooltip'       => __( 'Enter the name of the folder to create.', 'gravityflowfolders' ),
				'type'          => 'text',
				'required'      => true,
				'dependency'    => array(
					'live'        => true,
					'fields'      => array(
						array(
							'field'   => 'add_to_type',
							'values'  => array( 'new_folder' ),
						),
					),
				),
			);

			return $new_folder_settings;
		}

		/**
		 * Returns the settings for this step.
		 *
		 * @since 1.8
		 *
		 * @return array
		 */
		public function get_settings() {
			$existing_folder_settings = $this->get_existing_folder_settings();
			$new_folder_settings      = $this->get_new_folder_settings();

			$settings = array(
				array(
					'name'          => 'add_to_type',
					'label'         => esc_html__( 'Choose Folder Type', 'gravityflowfolders' ),
					'type'          => 'radio',
					'horizontal'    => true,
					'default_value' => rgar( $existing_folder_settings, 'html' ) == null ? 'existing_folder' : 'new_folder',
					'choices'       => array(
						array(
							'label' => esc_html__( 'Existing Folder', 'gravityflowfolders' ),
							'value' => 'existing_folder',
						),
						array(
							'label' => esc_html__( 'New Folder', 'gravityflowfolders' ),
							'value' => 'new_folder',
						),
					),
				),
			);

			$settings[] = $existing_folder_settings;
			$settings[] = $new_folder_settings;

			return array(
				'title'  => $this->get_label(),
				'fields' => $settings,
			);
		}

		/**
		 * Generate ID for new folder
		 *
		 * @since 1.8
		 *
		 * @return string
		 */
		private function generate_new_folder_id() {
			$folders       = $this->get_folders();
			$new_folder_id = strtolower( wp_generate_password( 9, false ) );
			while ( true ) {
				$duplicate_id = false;
				foreach ( $folders as $folder ) {
					if ( $folder->get_id() == $new_folder_id ) {
						$duplicate_id = true;
					}
				}

				if ( $duplicate_id ) {
					$new_folder_id = strtolower( wp_generate_password( 9, false ) );
				} else {
					break;
				}
			}

			return $new_folder_id;
		}

		/**
		 * Processes the step.
		 *
		 * @return bool
		 */
		function process() {
			// add to folders
			if ( $this->get_setting( 'add_to_type' ) == 'new_folder' ) {
				$new_folder_name = sanitize_text_field( $this->get_setting( 'new_folder_name' ) );
				$new_folder_name = GFCommon::replace_variables( $new_folder_name, $this->get_form(), $this->get_entry(), false, true, false, 'text' );
				if ( empty( trim( $new_folder_name ) ) ) {
					$note = sprintf( esc_html__( 'The Folder could not be created because the name was missing.', 'gravityflowfolders' ) );
					$this->add_note( $note );
					return true;
				}

				$option_name = 'gravityformsaddon_gravityflowfolders_app_settings';
				$option      = get_option( $option_name, array() );

				$folder_exists     = false;
				$new_folder_id     = $this->generate_new_folder_id();
				$option['folders'] = isset( $option['folders'] ) && is_array( $option['folders'] ) ? $option['folders'] : array();
				foreach ( $option['folders'] as $folder ) {
					if ( $folder['name'] == $new_folder_name ) {
						$folder_exists = true;
						$new_folder_id = $folder['id'];
					}
				}

				if ( ! $folder_exists ) {
					$new_folder = array(
						'id'             => $new_folder_id,
						'name'           => $new_folder_name,
						'permissions'    => 'all',
						'assignees'	     => array(),
					);

					array_push( $option['folders'], $new_folder );
					update_option( $option_name, $option );
				}

				$folders = $this->get_folders();
				foreach ( $folders as $folder ) {
					if ( $folder->get_id() == $new_folder_id ) {
						$entry_id = $this->get_entry_id();
						$folder->add_entry( $entry_id );
					}
				}

				if ( ! $folder_exists ) {
					$note = sprintf( esc_html__( 'Created folder: %s. Added entry to it.', 'gravityflowfolders' ), $new_folder_name );
				} else {
					$note = sprintf( esc_html__( 'Added to existing folder: %s', 'gravityflowfolders' ), $new_folder_name );
				}

				$this->add_note( $note );
			} else {
				$folders = $this->get_folders();

				if ( empty( $folders ) ) {
					$note = sprintf( esc_html__( 'Folders do not exist.', 'gravityflowfolders' ) );
					$this->add_note( $note );
				}

				foreach ( $folders as $folder ) {
					$setting_key                = $folder->get_meta_key();
					$setting_key_match          = $this->{$setting_key};
					$setting_key_match_original = $this->{$setting_key};
					$entry_id                   = $this->get_entry_id();

					/**
					 * Allows determination of whether entry should be added to a folder to be customized
					 *
					 * @since 1.3.1
					 *
					 * @param bool              $setting_key_match  Does the folder match current step settings.
					 * @param array             $folder             The folder which created this entry.
					 * @param int               $entry_id           The entry ID.
					 * @param Gravity_Flow_Step $current_step       The current step for this entry.
					 *
					 * @return bool
					 */
					$setting_key_match = apply_filters( 'gravityflowfolders_folder_match_add_step', $setting_key_match, $folder, $entry_id, $this );

					if ( $setting_key_match ) {
						if ( $setting_key_match !== $setting_key_match_original ) {
							gravity_flow_folders()->log_debug( __METHOD__ . '(): folder to add entry #' . $entry_id . ' to customized by gravityflowfolders_folder_match_add_step.' );
						}
						$folder->add_entry( $entry_id );
						$label = $folder->get_name();
						$note  = sprintf( esc_html__( 'Added to folder: %s', 'gravityflowfolders' ), $label );
						$this->add_note( $note );
					}
				}
			}

			return true;
		}
	}
}



