<?php

namespace WeDevs\Wpuf\Pro\Fields;

use WeDevs\Wpuf\Ajax\Upload_Ajax;

/**
 * Avatar Field Class
 *
 * @since 3.1.0
 **/
class Field_Profile_Photo extends Pro_Field_Contract {

    public function __construct() {
        $this->name       = __( 'Profile Photo', 'wpuf-pro' );
        $this->input_type = 'profile_photo';
        $this->icon       = 'user-circle';
    }

    /**
     * Render the Profile Photo field
     *
     * @param array   $field_settings
     *
     * @param integer $form_id
     *
     * @param string  $type
     *
     * @param integer $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        if ( method_exists( 'WeDevs\Wpuf\Fields\Field_Contract', 'enqueue_file_upload_scripts' ) ) {
            $this->enqueue_file_upload_scripts();
        }

        $has_photo     = false;
        $attachment_id = 0;

        // Get profile photo attachment ID
        if ( isset( $post_id ) && $post_id != 0 ) {
            $attachment_id = get_user_meta( $post_id, 'wpuf_profile_photo', true );
            if ( $attachment_id && wp_attachment_is_image( $attachment_id ) ) {
                $has_photo = true;
            }
        }

        $unique_id = sprintf( '%s-%d', $field_settings['name'], $form_id );
        ?>
        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <?php $this->print_label( $field_settings, $form_id ); ?>

            <div class="wpuf-fields">
                <div id="wpuf-<?php echo $unique_id; ?>-upload-container">
                    <div class="wpuf-attachment-upload-filelist" data-type="file"
                         data-required="<?php echo $field_settings['required']; ?>">
                        <a id="wpuf-<?php echo $unique_id; ?>-pickfiles" data-form_id="<?php echo $form_id; ?>"
                           class="button file-selector <?php echo ' wpuf_' . $field_settings['name'] . '_' . $form_id; ?>"
                           href="#"><?php echo $field_settings['button_label']; ?></a>

                        <ul class="wpuf-attachment-list thumbnail">
                            <?php
                            if ( $has_photo && $attachment_id ) {
                                // Use the same pattern as other file upload fields
                                echo Upload_Ajax::attach_html( $attachment_id, $field_settings['name'] );
                            }
                            ?>
                        </ul>
                    </div>
                </div><!-- .container -->

                <?php $this->help_text( $field_settings ); ?>

            </div> <!-- .wpuf-fields -->

            <script type="text/javascript">
                ;( function ( $ ) {
                    $( document ).ready( function () {
                        var uploader = new WPUF_Uploader( 'wpuf-<?php echo $unique_id; ?>-pickfiles', 'wpuf-<?php echo $unique_id; ?>-upload-container', 1, '<?php echo $field_settings['name']; ?>', 'jpg,jpeg,gif,png,bmp', <?php echo $field_settings['max_size'] ?>);
                        wpuf_plupload_items.push( uploader );
                    } );
                } )( jQuery );
            </script>
        </li>

        <?php
    }

    /**
     * field helper function
     *
     * @return void
     **/
    public function print_list_attributes( $field ) {
        $label      = isset( $field['label'] ) ? $field['label'] : '';
        $el_name    = ! empty( $field['name'] ) ? $field['name'] : '';
        $el_name    = 'wpuf_' . $el_name;
        $class_name = ! empty( $field['css'] ) ? ' ' . $field['css'] : '';
        $field_size = ! empty( $field['width'] ) ? ' field-size-' . $field['width'] : '';
        printf( 'class="wpuf-el %s%s%s" data-label="%s"', $el_name, $class_name, $field_size, $label );
    }

    /**
     * Get field options setting
     *
     * @return array
     **/
    public function get_options_settings() {
        $default_options = $this->get_default_option_settings( false, [ 'dynamic', 'count', 'width' ] );
        $settings = [
            [
                'name'         => 'name',
                'title'        => __( 'Meta Key', 'wp-user-frontend' ),
                'type'         => 'text-meta',
                'is_read_only' => true,
                'section'      => 'basic',
                'priority'     => 11,
                'help_text'    => __( 'Name of the meta key this field will save to', 'wp-user-frontend' ),
            ],
            [
                'name'      => 'button_label',
                'title'     => __( 'Button Label', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => __( 'Select Profile Photo', 'wpuf-pro' ),
                'section'   => 'basic',
                'priority'  => 20,
                'help_text' => __( 'Enter a label for the Profile Photo button', 'wpuf-pro' ),
            ],
            [
                'name'      => 'max_size',
                'title'     => __( 'Max. file size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Enter maximum upload size limit in KB', 'wpuf-pro' ),
            ],
        ];

        if ( function_exists( 'wpuf_field_profile_photo_allowed_extensions' ) ) {
            $settings[] = [
                'name'        => 'extension',
                'title'       => __( 'Allowed file types', 'wpuf-pro' ),
                'title_class' => 'label-hr',
                'type'        => 'checkbox',
                'options'     => wpuf_field_profile_photo_allowed_extensions(),
                'section'     => 'advanced',
                'priority'    => 21,
                'help_text'   => __( 'Enter The allowed file types', 'wpuf-pro' ),
            ];
        }

        return array_merge( $default_options, $settings );
    }

    /**
     * Get the field props
     *
     * @return array
     **/
    public function get_field_props() {
        $defaults = $this->default_attributes();
        $props = [
            'input_type'    => 'profile_photo',
            'required'      => 'yes',
            'is_meta'       => 'no', // we will handle this manually
            'button_label'  => __( 'Select Profile Photo', 'wpuf-pro' ),
            'name'          => 'wpuf_profile_photo',
            'max_size'      => '2048', // default 2MB
            'extension'     => [ 'jpg', 'jpeg', 'png' ],
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Prepare entry
     *
     * @param $field
     *
     * @return void
     **/
    public function prepare_entry( $field ) {
        return isset( $_POST['wpuf_files'][ $field['name'] ] ) ? $_POST['wpuf_files'][ $field['name'] ] : [];
    }

}
