<?php
/**
 * The Events Calendar Integration Template
 *
 * @since 2.9
 */

namespace WeDevs\Wpuf\Pro\Integrations\Events_Calendar\Templates;

use DateTimeZone;
use Tribe__Timezones as Timezones;
use WeDevs\Wpuf\Admin\Forms\Form_Template;

class Event_Form_Template extends Form_Template {
    public $normal_timezone;
    private $start_meta_key;
    private $end_meta_key;
    const FORMAT = 'Y-m-d H:i:s';

    public function __construct() {
        parent::__construct();
        $this->start_meta_key = '_EventStartDate';
        $this->end_meta_key   = '_EventEndDate';
        if ( class_exists( 'Tribe__Events__Main' ) ) {
            if ( Timezones::is_mode( 'site' ) ) {
                $this->normal_timezone = new DateTimeZone( 'UTC' );
                $this->start_meta_key  = '_EventStartDateUTC';
                $this->end_meta_key    = '_EventEndDateUTC';
            } else {
                $this->normal_timezone = Timezones::build_timezone_object();
                $this->start_meta_key  = '_EventStartDate';
                $this->end_meta_key    = '_EventEndDate';
            }
        }
        $this->enabled       = class_exists( 'Tribe__Events__Main' );
        $this->title         = __( 'The Events Calendar', 'wpuf-pro' );
        $this->description   = __( 'Form for creating events. The Events Calendar plugin is required.', 'wpuf-pro' );
        $this->image         = WPUF_ASSET_URI . '/images/templates/event.svg';
        $this->form_fields   = $this->get_form_fields();
        $this->form_settings = $this->get_form_settings();
    }

    public function get_form_fields() {
        // Check if this form already has saved fields in the database
        if (isset($_GET['id']) && is_numeric($_GET['id'])) {
            $existing_fields = wpuf_get_form_fields($_GET['id']);
            if (!empty($existing_fields)) {
                // Form already has saved fields, return them
                return $existing_fields;
            }
        }

        // Return template fields for new forms
        $form_fields = [
            [
                'input_type'      => 'text',
                'template'        => 'post_title',
                'required'        => 'yes',
                'label'           => __( 'Event Title', 'wpuf-pro' ),
                'name'            => 'post_title',
                'is_meta'         => 'no',
                'default'         => '',
                'size'            => '',
                'placeholder'     => __( 'Please enter your event title', 'wpuf-pro' ),
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'restriction_to'   => 'max',
                'restriction_type' => 'character',
                'width'           => 'large',
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'          => 'textarea',
                'template'            => 'post_content',
                'required'            => 'no',
                'label'               => __( 'Event details', 'wpuf-pro' ),
                'name'                => 'post_content',
                'is_meta'             => 'no',
                'default'             => '',
                'help'                => __( 'Write the full description of your event', 'wpuf-pro' ),
                'rows'                => '5',
                'cols'                => '25',
                'rich'                => 'yes',
                'insert_image'        => 'yes',
                'wpuf_cond'           => $this->conditionals,
                'wpuf_visibility'     => $this->get_default_visibility_prop(),
                'restriction_to'      => 'max',
                'restriction_type'    => 'character',
                'text_editor_control' => [],
                'width'               => 'large',
                'id'                  => uniqid('wpuf_', true),
                'is_new'              => true,
            ],
            [
                'input_type'      => 'image_upload',
                'template'        => 'featured_image',
                'count'           => '1',
                'required'        => 'no',
                'label'           => __( 'Featured Image', 'wpuf-pro' ),
                'button_label'    => 'Featured Image',
                'name'            => 'featured_image',
                'is_meta'         => 'no',
                'help'            => __( 'Upload the main image of your event', 'wpuf-pro' ),
                'max_size'        => '1024',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'taxonomy',
                'template'        => 'taxonomy',
                'required'        => 'no',
                'label'           => __( 'Event Categories', 'wpuf-pro' ),
                'name'            => 'tribe_events_cat',
                'is_meta'         => 'no',
                'width'           => 'small',
                'type'            => 'select',
                'first'           => '- Select -',
                'show_inline'     => 'inline',
                'orderby'         => 'name',
                'order'           => 'ASC',
                'exclude'         => [],
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'          => 'textarea',
                'template'            => 'post_excerpt',
                'required'            => 'no',
                'label'               => __( 'Short Description', 'wpuf-pro' ),
                'name'                => 'post_excerpt',
                'is_meta'             => 'no',
                'default'             => '',
                'placeholder'         => '',
                'help'                => __( 'Provide a short description of this event (optional)', 'wpuf-pro' ),
                'rows'                => '5',
                'cols'                => '25',
                'rich'                => 'no',
                'wpuf_cond'           => $this->conditionals,
                'wpuf_visibility'     => $this->get_default_visibility_prop(),
                'restriction_to'      => 'max',
                'restriction_type'    => 'character',
                'text_editor_control' => [],
                'width'               => 'large',
            ],
            [
                'input_type'      => 'text',
                'template'        => 'post_tags',
                'required'        => 'no',
                'label'           => __( 'Event Tags', 'wpuf-pro' ),
                'name'            => 'tags',
                'is_meta'         => 'no',
                'default'         => '',
                'placeholder'     => '',
                'help'            => __( 'Separate tags with commas.', 'wpuf-pro' ),
                'size'            => '40',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'width'           => 'large',
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'section_break',
                'template'        => 'section_break',
                'label'           => __( 'TIME & DATE', 'wpuf-pro' ),
                'description'     => '',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'date',
                'template'        => 'date_field',
                'required'        => 'yes',
                'label'           => __( 'Event Start', 'wpuf-pro' ),
                'name'            => $this->start_meta_key,
                'is_meta'         => 'yes',
                'help'            => '',
                'width'           => 'large',
                'format'          => 'yy-mm-dd',
                'time'            => 'yes',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'date',
                'template'        => 'date_field',
                'required'        => 'yes',
                'label'           => __( 'Event End', 'wpuf-pro' ),
                'name'            => $this->end_meta_key,
                'is_meta'         => 'yes',
                'help'            => '',
                'width'           => 'large',
                'format'          => 'yy-mm-dd',
                'time'            => 'yes',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'radio',
                'template'        => 'radio_field',
                'required'        => 'no',
                'label'           => __( 'All Day Event', 'wpuf-pro' ),
                'name'            => '_EventAllDay',
                'is_meta'         => 'yes',
                'inline'          => 'yes',
                'options'         => [
                    'yes' => 'Yes',
                    'no'  => 'No',
                ],
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'section_break',
                'template'        => 'section_break',
                'label'           => __( 'VENUE INFORMATION', 'wpuf-pro' ),
                'wpuf_cond'       => $this->conditionals,
                'description'     => '',
                'name'            => 'undefined_copy',
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'radio',
                'template'        => 'radio_field',
                'required'        => 'no',
                'label'           => __( 'Venue', 'wpuf-pro' ),
                'name'            => 'venue',
                'is_meta'         => 'yes',
                'selected'        => 'find_a_venue',
                'inline'          => 'yes',
                'options'         => [
                    'find_a_venue' => __( 'Find a Venue', 'wpuf-pro' ),
                    'create'       => __( 'Create', 'wpuf-pro' ),
                ],
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'select',
                'template'        => 'dropdown_field',
                'required'        => 'no',
                'label'           => __( 'Find a Venue', 'wpuf-pro' ),
                'name'            => '_EventVenueID',
                'is_meta'         => 'yes',
                'first'           => '- Select -',
                'options'         => [],
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'venue',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'find_a_venue',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'text',
                'template'        => 'text_field',
                'required'        => 'yes',
                'label'           => __( 'Venue Name', 'wpuf-pro' ),
                'name'            => 'venue_name',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => '',
                'size'            => 40,
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'venue',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'address',
                'template'        => 'address_field',
                'required'        => 'no',
                'label'           => __( 'Venue Address', 'wpuf-pro' ),
                'name'            => 'venue_address',
                'is_meta'         => 'yes',
                'help'            => '',
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'venue',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'address_desc'    => '',
                'address'         => [
                    'street_address'  => [
                        'checked'     => 'checked',
                        'type'        => 'text',
                        'required'    => '',
                        'label'       => __( 'Address', 'wpuf-pro' ),
                        'value'       => '',
                        'placeholder' => '',
                    ],
                    'street_address2' => [
                        'checked'     => '',
                        'type'        => 'text',
                        'required'    => '',
                        'label'       => __( 'Address Line 2', 'wpuf-pro' ),
                        'value'       => '',
                        'placeholder' => '',
                    ],
                    'city_name'       => [
                        'checked'     => 'checked',
                        'type'        => 'text',
                        'required'    => '',
                        'label'       => __( 'City', 'wpuf-pro' ),
                        'value'       => '',
                        'placeholder' => '',
                    ],
                    'state'           => [
                        'checked'     => 'checked',
                        'type'        => 'text',
                        'required'    => '',
                        'label'       => __( 'State', 'wpuf-pro' ),
                        'value'       => '',
                        'placeholder' => '',
                    ],
                    'zip'             => [
                        'checked'     => 'checked',
                        'type'        => 'text',
                        'required'    => '',
                        'label'       => __( 'Postal Code', 'wpuf-pro' ),
                        'value'       => '',
                        'placeholder' => '',
                    ],
                    'country_select'  => [
                        'checked'                          => 'checked',
                        'type'                             => 'select',
                        'required'                         => '',
                        'label'                            => __( 'Country', 'wpuf-pro' ),
                        'value'                            => '',
                        'country_list_visibility_opt_name' => 'all',
                        'country_select_hide_list'         => [],
                        'country_select_show_list'         => [],
                    ],
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'phone',
                'template'        => 'phone_field',
                'required'        => 'no',
                'label'           => __( 'Venue Phone', 'wpuf-pro' ),
                'name'            => 'venue_phone',
                'is_meta'         => 'yes',
                'width'           => 'large',
                'size'            => 40,
                'default'         => '',
                'placeholder'     => '',
                'help'            => '',
                'show_country_list' => 'yes',
                'auto_placeholder'  => 'yes',
                'country_list'    => [
                    'name'                              => '',
                    'country_list_visibility_opt_name'  => 'all',
                    'country_select_show_list'          => [],
                    'country_select_hide_list'          => [],
                ],
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'venue',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'url',
                'template'        => 'website_url',
                'required'        => 'no',
                'label'           => __( 'Venue Website', 'wpuf-pro' ),
                'name'            => 'venue_website',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => 'https://example.com',
                'width'           => 'large',
                'open_window'     => 'same',
                'size'            => 40,
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'venue',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'checkbox',
                'template'        => 'checkbox_field',
                'required'        => 'no',
                'label'           => '',
                'name'            => '_EventShowMap',
                'is_meta'         => 'yes',
                'inline'          => 'yes',
                'options'         => [
                    1 => __( 'Show Google Map', 'wpuf-pro' ),
                ],
                'wpuf_cond'       => [
                    'condition_status' => 'no',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'checkbox',
                'template'        => 'checkbox_field',
                'required'        => 'no',
                'label'           => '',
                'name'            => '_EventShowMapLink',
                'is_meta'         => 'yes',
                'inline'          => 'yes',
                'options'         => [
                    1 => __( 'Show Google Maps Link', 'wpuf-pro' ),
                ],
                'wpuf_cond'       => [
                    'condition_status' => 'no',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'section_break',
                'template'        => 'section_break',
                'label'           => __( 'ORGANIZERS', 'wpuf-pro' ),
                'description'     => '',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'radio',
                'template'        => 'radio_field',
                'required'        => 'no',
                'label'           => __( 'Organizer', 'wpuf-pro' ),
                'name'            => 'organizer',
                'is_meta'         => 'yes',
                'selected'        => 'find_organizer',
                'inline'          => 'yes',
                'options'         => [
                    'find_organizer' => 'Find an Organizer',
                    'create'         => 'Create',
                ],
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'select',
                'template'        => 'dropdown_field',
                'required'        => 'no',
                'label'           => __( 'Find an Organizer', 'wpuf-pro' ),
                'name'            => '_EventOrganizerID',
                'is_meta'         => 'yes',
                'first'           => '- Select -',
                'options'         => [],
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'organizer',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'find_organizer',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'text',
                'template'        => 'text_field',
                'required'        => 'yes',
                'label'           => __( 'Organizer Name', 'wpuf-pro' ),
                'name'            => 'organizer_name',
                'is_meta'         => 'yes',
                'placeholder'     => '',
                'default'         => '',
                'size'            => 40,
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'organizer',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'phone',
                'template'        => 'phone_field',
                'required'        => 'no',
                'label'           => __( 'Organizer Phone', 'wpuf-pro' ),
                'name'            => 'organizer_phone',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => '',
                'help'            => '',
                'width'           => 'large',
                'size'            => 40,
                'show_country_list' => 'yes',
                'auto_placeholder'  => 'yes',
                'country_list'    => [
                    'name'                              => '',
                    'country_list_visibility_opt_name'  => 'all',
                    'country_select_show_list'          => [],
                    'country_select_hide_list'          => [],
                ],
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'organizer',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'url',
                'template'        => 'website_url',
                'required'        => 'no',
                'label'           => __( 'Organizer Website', 'wpuf-pro' ),
                'name'            => 'organizer_website',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => '',
                'width'           => 'large',
                'open_window'     => 'same',
                'size'            => 40,
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'organizer',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'email',
                'template'        => 'email_address',
                'required'        => 'no',
                'label'           => __( 'Organizer Email', 'wpuf-pro' ),
                'name'            => 'organizer_email',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => '',
                'width'           => 'large',
                'size'            => 40,
                'wpuf_cond'       => [
                    'condition_status' => 'yes',
                    'cond_field'       => [
                        0 => 'organizer',
                    ],
                    'cond_operator'    => [
                        0 => '=',
                    ],
                    'cond_option'      => [
                        0 => 'create',
                    ],
                    'cond_logic'       => 'all',
                ],
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'section_break',
                'template'        => 'section_break',
                'label'           => __( 'EVENT WEBSITE', 'wpuf-pro' ),
                'description'     => '',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'url',
                'template'        => 'website_url',
                'required'        => 'no',
                'label'           => __( 'Event Website', 'wpuf-pro' ),
                'name'            => '_EventURL',
                'is_meta'         => 'yes',
                'placeholder'     => '',
                'default'         => '',
                'width'           => 'large',
                'size'            => 40,
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'section_break',
                'template'        => 'section_break',
                'label'           => __( 'EVENT COST', 'wpuf-pro' ),
                'description'     => '',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'text',
                'template'        => 'text_field',
                'required'        => 'no',
                'label'           => __( 'Currency Symbol', 'wpuf-pro' ),
                'name'            => '_EventCurrencySymbol',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => '',
                'size'            => '',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ],
            [
                'input_type'      => 'text',
                'template'        => 'text_field',
                'required'        => 'no',
                'label'           => __( 'Cost', 'wpuf-pro' ),
                'name'            => '_EventCost',
                'is_meta'         => 'yes',
                'default'         => '',
                'placeholder'     => '',
                'size'            => '',
                'help'            => 'Enter a 0 for events that are free or leave blank to hide the field.',
                'wpuf_cond'       => $this->conditionals,
                'wpuf_visibility' => $this->get_default_visibility_prop(),
                'id'              => uniqid('wpuf_', true),
                'is_new'          => true,
            ]
        ];

        /**
         * Opportunity to modify TEC form fields before they're used
         *
         * This filter allows developers to add, modify, or remove form fields from
         * the Events Calendar form template. Useful for custom event fields,
         * integration with other plugins, or custom field types.
         *
         * @since 4.1.8
         *
         * @param array                                                                   $form_fields The array of form field definitions
         * @param \WeDevs\Wpuf\Integrations\Events_Calendar\Templates\Event_Form_Template $this        The form template instance
         */
        $form_fields = apply_filters( 'wpuf_tec_form_fields', $form_fields, $this );

        return $form_fields;
    }

    public function get_form_settings() {
        return [
            'post_type'          => 'tribe_events',
            'post_status'        => 'publish',
            'default_cat'        => '-1',
            'guest_post'         => 'false',
            'message_restrict'   => __(
                'This page is restricted. Please Log in / Register to view this page.', 'wpuf-pro'
            ),
            'redirect_to'        => 'post',
            'comment_status'     => 'open',
            'submit_text'        => __( 'Create Event', 'wpuf-pro' ),
            'submit_button_cond' => [
                'condition_status' => 'no',
                'cond_logic'       => 'any',
                'conditions'       => [
                    [
                        'name'     => '',
                        'operator' => '=',
                        'option'   => '',
                    ],
                ],
            ],
            'edit_post_status'   => 'publish',
            'edit_redirect_to'   => 'same',
            'update_message'     => __(
                'Event has been updated successfully. <a target="_blank" href="{link}">View event</a>', 'wpuf-pro'
            ),
            'edit_url'           => '',
            'update_text'        => __( 'Update Event', 'wpuf-pro' ),
            'form_template'      => 'post_form_template_events_calendar',
            'notification'       => [
                'new'          => 'on',
                'new_to'       => get_option( 'admin_email' ),
                'new_subject'  => 'New event has been created',
                'new_body'     => 'Hi,
    A new event has been created in your site {sitename} ({siteurl}).

    Here are the details:
    Event Title: {post_title}
    Description: {post_content}
    Short Description: {post_excerpt}
    Author: {author}
    Post URL: {permalink}
    Edit URL: {editlink}',
                'edit'         => 'on',
                'edit_to'      => get_option( 'admin_email' ),
                'edit_subject' => 'An event has been updated',
                'edit_body'    => 'Hi Admin,
The event "{post_title}" has been updated.

Here are the event details:
Event Title: {post_title}
Event Start: {custom__EventStartDate}
Event End: {custom__EventEndDate}
Event Website: {custom__EventURL}
Event Cost: {custom__EventCost}
Description: {post_content}
Short Description: {post_excerpt}
Author: {author}
Event URL: {permalink}
Edit URL: {editlink}'
            ],
        ];
    }
}
