<?php
/**
 * Plugin Name: SMS Notification
 * Description: SMS notification for post
 * Plugin URI: https://wedevs.com/docs/wp-user-frontend-pro/modules/sms-notification/
 * Thumbnail Name: wpuf-sms.png
 * Author: weDevs
 * Author URI: http://wedevs.com/
 * Version: 0.1
 * License: GPL2
 * Text Domain: wpuf-sms-notification
 * Domain Path: languages
 *
 * Copyright (c) 2017 weDevs (email: info@wedevs.com). All rights reserved.
 *
 * Released under the GPL license
 * http://www.opensource.org/licenses/gpl-license.php
 *
 * This is an add-on for WordPress
 * http://wordpress.org/
 *
 * **********************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * **********************************************************************
 */

// don't call the file directly
if ( ! defined( 'ABSPATH' ) ) exit;

require_once dirname (__FILE__) . '/class/gateways.php';

class WPUF_Admin_sms {
    function __construct() {
        // Post form settings hooks
        add_filter( 'wpuf_post_form_module_settings_menu_titles', [ $this, 'sms_menu' ] );
        add_filter( 'wpuf_post_form_builder_setting_menu_contents', [ $this, 'sms_fields' ] );

        // Registration form settings hooks
        add_filter( 'wpuf_registration_form_module_settings_menu_sub_items', [ $this, 'sms_menu' ] );
        add_filter( 'wpuf_registration_form_builder_setting_menu_contents', [ $this, 'sms_fields' ] );

        add_action( 'wpuf_profile_setting', [ $this, 'settings_registration' ], 10, 2 );

        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_script' ] );
        add_filter( 'wpuf_settings_sections', [ $this, 'sms_settings_section' ], 10, 1 );
        add_filter( 'wpuf_settings_fields', [ $this, 'wpuf_sms_settings_fields' ], 10, 1 );
        add_action( 'wpuf_add_post_after_insert', [ $this, 'send_sms_after_post' ], 10, 4);
        add_action( 'wpuf_after_register', [ $this, 'send_sms_after_registration'], 10, 3 );
    }

    /**
     * Add sms fields on post form settings
     *
     * @since 4.1.0
     *
     * @param $fields
     *
     * @return array
     */
    public function sms_fields( $fields ) {
        // Detect if this is for registration form or post form
        $current_filter = current_filter();
        $is_registration = strpos( $current_filter, 'wpuf_registration_form' ) !== false;
        
        // Set dynamic text based on form type
        if ( $is_registration ) {
            $description = __( 
                'Configure SMS settings to enable mobile messages for new user registrations. This feature ensures you receive instant alerts on your mobile device whenever a user registers.',
                'wpuf-pro'
            );
            $toggle_label = __( 'Enable SMS for Registration', 'wp-user-frontend' );
            $help_text = __( 'Receive an SMS alert for each user registration.', 'wp-user-frontend' );
        } else {
            $description = __(
                'Configure SMS settings to enable mobile messages for post submissions. This feature ensures that you receive instant alerts on your mobile device for new or updated posts',
                'wpuf-pro'
            );
            $toggle_label = __( 'Enable SMS Per Post', 'wp-user-frontend' );
            $help_text = __( 'Receive an SMS alert for each post submission or update.', 'wp-user-frontend' );
        }
        
        $fields['modules']['sms'] = [
            'section' => [
                'mobile_message' => [
                    'label'  => __( 'Mobile Message', 'wp-user-frontend' ),
                    'desc'   => $description,
                    'fields' => [
                        'sms_enable'      => [
                            'label'     => $toggle_label,
                            'type'      => 'toggle',
                            'help_text' => $help_text,
                        ],
                        'mob_number'      => [
                            'label'     => __( 'Mobile Number', 'wp-user-frontend' ),
                            'type'      => 'text',
                            'help_text' => __( 'SMS will be sent to this number', 'wp-user-frontend' ),
                        ],
                        'sms_sender_name' => [
                            'label'     => __( 'Sender Name', 'wp-user-frontend' ),
                            'type'      => 'text',
                            'help_text' => __( 'SMS sender name', 'wp-user-frontend' ),
                        ],
                        'sms_body'        => [
                            'label'     => __( 'SMS Body', 'wp-user-frontend' ),
                            'type'      => 'text',
                            'help_text' => __( 'Customize the SMS message content', 'wp-user-frontend' ),
                        ],
                        'gateway'        => [
                            'label'     => __( 'Gateway', 'wp-user-frontend' ),
                            'type'      => 'pic-radio',
                            'options'   => $this->get_sms_pic_options(),
                            'help_text' => __( 'Select SMS gateway', 'wp-user-frontend' ),
                        ],
                    ],
                ],
            ],
        ];

        return $fields;
    }

    /**
     * Add sms menu for registration form modules section
     */
    public function sms_menu( $settings ) {
        $settings['sms'] = [
            'label' => __( 'SMS', 'wp-user-frontend' ),
            'icon'  => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="custom-stroke">
<path d="M21.75 6.75V17.25C21.75 18.4926 20.7426 19.5 19.5 19.5H4.5C3.25736 19.5 2.25 18.4926 2.25 17.25V6.75M21.75 6.75C21.75 5.50736 20.7426 4.5 19.5 4.5H4.5C3.25736 4.5 2.25 5.50736 2.25 6.75M21.75 6.75V6.99271C21.75 7.77405 21.3447 8.49945 20.6792 8.90894L13.1792 13.5243C12.4561 13.9694 11.5439 13.9694 10.8208 13.5243L3.32078 8.90894C2.65535 8.49945 2.25 7.77405 2.25 6.99271V6.75" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>',
        ];
        return $settings;
    }

    function send_sms_after_registration( $user_id, $form_id, $form_settings ) {
        $this->send_sms( $form_id, $form_settings );
    }

    function send_sms_after_post( $post_id, $form_id, $form_settings, $form_vars ) {
        $this->send_sms( $form_id, $form_settings );
    }

    function send_sms( $form_id, $form_settings ) {
        $validate = $this->validation( $form_id, $form_settings );

        if ( ! $validate ) {
            return;
        }

        $active_gateway = isset($form_settings['gateway']) ? $form_settings['gateway'] : '';
        do_action( 'wpuf_sms_via_' . $active_gateway, (object)$form_settings );
    }

    function validation( $form_id, $form_settings ) {
        $form_settings = get_post_meta( $form_id, 'wpuf_form_settings', true );
        if ( !isset( $form_settings['sms_enable'] ) || !in_array( $form_settings['sms_enable'], [ 'yes', 'on' ], true ) ) {
            return false;
        }

        $mob_number = isset( $form_settings['mob_number'] ) ? $form_settings['mob_number'] : '';
        if ( empty( $mob_number ) ) {
            return false;
        }

        return true;
    }

    function wpuf_sms_settings_fields( $settings_fields ) {

        $settings_fields['wpuf_sms'] = apply_filters( 'wpuf_sms_field', [
            [
                'name' => 'clickatell_name',
                'label' => __( 'Clickatell name', 'wpuf-pro' ),
                'desc' => __( 'Clickatell name', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'clickatell_api_id',
                'label' => __( 'Clickatell API ID', 'wpuf-pro' ),
                'desc' => __( 'Clickatell API ID', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'clickatell_api_key',
                'label' => __( 'Clickatell API key', 'wpuf-pro' ),
                'desc' => __( 'Clickatell API key', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'smsglobal_api_key',
                'label' => __( 'SMSGlobal Api Key', 'wpuf-pro' ),
                'desc' => __( 'SMSGlobal Api Key', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'smsglobal_api_secret',
                'label' => __( 'SMSGlobal Api Secret', 'wpuf-pro' ),
                'desc' => __( 'SMSGlobal Api Secret', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'nexmo_api_key',
                'label' => __( 'Nexmo API Key', 'wpuf-pro' ),
                'desc' => __( 'Nexmo API Key', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'nexmo_api_Secret',
                'label' => __( 'Nexmo API Secret', 'wpuf-pro' ),
                'desc' => __( 'Nexmo API Secret', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'twilio_number',
                'label' => __( 'Twilio From Number', 'wpuf-pro' ),
                'desc' => __( 'Twilio From Number', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'twilio_sid',
                'label' => __( 'Twilio Account SID', 'wpuf-pro' ),
                'desc' => __( 'Twilio Account SID', 'wpuf-pro' ),
                'type' => 'text',
            ],
            [
                'name' => 'twilio_token',
                'label' => __( 'Twilio Authro Token', 'wpuf-pro' ),
                'desc' => __( 'Twilio Authro Token', 'wpuf-pro' ),
                'type' => 'text',
            ]
        ] );

        return $settings_fields;
    }

    function sms_settings_section( $sections ) {
        $sections[] = [
            'id' => 'wpuf_sms',
            'title' => __( 'SMS', 'wpuf-pro' ),
            'icon' => 'dashicons-format-status'
        ];

        return $sections;
    }

    function enqueue_script() {
        wp_enqueue_script( 'wpuf_sms', plugins_url( 'assets/js/sms.js', __FILE__ ), ['jquery'], false, true  );
    }

    function settins_form( $form_settings, $post ) {
        $this->sms_form_generator( $form_settings, $post );
    }

    function settings_registration( $form_settings, $post ) {
        $this->sms_form_generator( $form_settings, $post );
    }

    function sms_form_generator( $form_settings, $post ) {

        $mob_number = isset( $form_settings['mob_number'] ) ? $form_settings['mob_number'] : '';
        $sms_enable = isset( $form_settings['sms_enable'] ) ? $form_settings['sms_enable'] : '';
        $is_sms_enabled = wpuf_is_checkbox_or_toggle_on( $sms_enable ) ? true : false;
        $getway_style = $is_sms_enabled ? '' : 'none';
        $sms_gateway = isset( $form_settings['gateway'] ) ? $form_settings['gateway'] : false;
        $sender_name = isset( $form_settings['sms_sender_name'] ) ? $form_settings['sms_sender_name'] : '';
        $sms_body = isset( $form_settings['sms_body'] ) ? $form_settings['sms_body'] : '';
        ?>
         <tr>
            <th><?php _e( 'Mobile Message', 'wpuf-pro' ); ?></th>
                <td class="wpuf-mobile-message">
                    <label>
                        <input type="checkbox" class="wpuf-sms-enable" name="wpuf_settings[sms_enable]" <?php checked( true, $is_sms_enabled ); ?> value="yes">
                        <?php _e( 'Enable SMS', 'wpuf-pro' ) ?>
                    </label>
                    <div class="description"><?php _e( 'Send sms for per post', 'wpuf-pro' ); ?></div>
                </td>
            </tr>
            <tr class="wpuf-sms-wrap" style="display: <?php echo $getway_style; ?>;">
                <th><?php _e( 'Mobile Number', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="text" name="wpuf_settings[mob_number]" placeholder="<?php _e( 'mobile number', 'wpuf-pro' ); ?>" value="<?php echo esc_attr( $mob_number ); ?>" />
                    </label>
                    <div class="description"><?php _e( 'SMS will be sent to this number', 'wpuf-pro' ); ?></div>
                </td>
            </tr>

            <tr class="wpuf-sms-wrap" style="display: <?php echo $getway_style; ?>;">
                <th><?php _e( 'Sender Nameeeee', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="text" name="wpuf_settings[sms_sender_name]" placeholder="<?php _e( 'SMS sender name', 'wpuf-pro' ); ?>" value="<?php echo esc_attr( $sender_name ); ?>" />
                    </label>
                    <div class="description"><?php _e( 'SMS sender name', 'wpuf-pro' ); ?></div>
                </td>
            </tr>

            <tr class="wpuf-sms-wrap" style="display: <?php echo $getway_style; ?>;">
                <th><?php _e( 'SMS body', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="text" name="wpuf_settings[sms_body]" placeholder="<?php _e( 'SMS body', 'wpuf-pro' ); ?>" value="<?php echo esc_attr( $sms_body ); ?>" />
                    </label>
                    <div class="description"><?php _e( 'SMS body', 'wpuf-pro' ); ?></div>
                </td>
            </tr>

            <tr class="wpuf-sms-wrap" style="display: <?php echo $getway_style; ?>;">
                <th><?php _e( 'Gateway', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[gateway]">
                        <?php
                        $gateways = $this->get_sms_gateway();

                        foreach ( $gateways as $key => $gateway ) {
                            printf('<option value="%s"%s>%s</option>', $key, selected( $key, $sms_gateway, false ), $gateway );
                        }
                        ?>
                    </select>
                </td>
            </tr>
        <?php
    }

    function get_sms_gateway( $gate_way = false ) {
        $sms_gate_way = [
            'nexmo' => __( 'Nexmo', 'wpuf-pro' ),
            'twilio' => __( 'Twilio', 'wpuf-pro' ),
            'clickatell' => __( 'Clickatel', 'wpuf-pro' ),
            'smsglobal' => __( 'SMS global', 'wpuf-pro' )
        ];

        if ( $gate_way ) {
            return $sms_gate_way[$gate_way];
        }

        return apply_filters( 'wpuf_sms_gateways', $sms_gate_way );

    }

    public function get_sms_pic_options() {
        $gateways = $this->get_sms_gateway();
        $options = [];

        foreach ( $gateways as $key => $gateway ) {
            $options[$key] = [
                'label' => $gateway,
                'image' => WPUF_PRO_ROOT_URI . '/modules/sms-notification/assets/images/' . $key . '.svg',
            ];
        }

        return $options;
    }
}

new WPUF_Admin_sms();
