/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';
import { Component } from '@wordpress/element';
import { Notice } from '@wordpress/components';

/**
 * Standard Error Boundary Component
 * 
 * Provides consistent error handling and fallback UI for blocks
 */
export default class StandardErrorBoundary extends Component {
    constructor(props) {
        super(props);
        this.state = { 
            hasError: false, 
            error: null,
            errorInfo: null 
        };
    }

    static getDerivedStateFromError(error) {
        return { hasError: true, error };
    }

    componentDidCatch(error, errorInfo) {
        this.setState({
            error,
            errorInfo
        });
    }

    render() {
        if (this.state.hasError) {
            return (
                <div className="wpuf-ud-error-boundary">
                    <Notice 
                        status="error" 
                        isDismissible={false}
                        className="wpuf-ud-error-notice"
                    >
                        <strong>{__('Block Error', 'wpuf-pro')}</strong>
                        <p>
                            {__('This block encountered an error and cannot be displayed properly.', 'wpuf-pro')}
                        </p>
                        {this.props.showDebugInfo && this.state.error && (
                            <details className="wpuf-ud-error-details">
                                <summary>{__('Error Details', 'wpuf-pro')}</summary>
                                <pre>{this.state.error.toString()}</pre>
                            </details>
                        )}
                    </Notice>
                </div>
            );
        }

        return this.props.children;
    }
}

/**
 * Standard Loading Component
 * 
 * Provides consistent loading states across blocks
 */
export function StandardLoading({ message = __('Loading...', 'wpuf-pro'), className = '' }) {
    return (
        <div className={`wpuf-ud-loading ${className}`}>
            <div className="wpuf-ud-loading-spinner"></div>
            <p className="wpuf-ud-loading-message">{message}</p>
        </div>
    );
}

/**
 * Standard Empty State Component
 * 
 * Provides consistent empty state UI across blocks
 */
export function StandardEmptyState({ 
    title = __('No Data Available', 'wpuf-pro'),
    message = __('There is no data to display at the moment.', 'wpuf-pro'),
    icon = null,
    className = ''
}) {
    return (
        <div className={`wpuf-ud-empty-state ${className}`}>
            {icon && <div className="wpuf-ud-empty-icon">{icon}</div>}
            <h4 className="wpuf-ud-empty-title">{title}</h4>
            <p className="wpuf-ud-empty-message">{message}</p>
        </div>
    );
}

/**
 * Standard Error Message Component
 * 
 * Provides consistent error message display
 */
export function StandardErrorMessage({ 
    error, 
    title = __('Error', 'wpuf-pro'),
    className = '',
    showRetry = false,
    onRetry = null
}) {
    const errorMessage = error?.message || error || __('An unknown error occurred', 'wpuf-pro');

    return (
        <div className={`wpuf-ud-error-message ${className}`}>
            <Notice 
                status="error" 
                isDismissible={false}
                className="wpuf-ud-error-notice"
            >
                <strong>{title}</strong>
                <p>{errorMessage}</p>
                {showRetry && onRetry && (
                    <button 
                        onClick={onRetry}
                        className="wpuf-ud-retry-button"
                    >
                        {__('Try Again', 'wpuf-pro')}
                    </button>
                )}
            </Notice>
        </div>
    );
}
