<?php
/**
 * WPSummarize Rating Notice
 */

// notice configuration
define('WPSUMMARIZE_RATING_DAYS_THRESHOLD', 7);
define('WPSUMMARIZE_RATING_USAGE_THRESHOLD', 3);
define('WPSUMMARIZE_RATING_PLUGIN_NAME', 'WPSummarize');
define('WPSUMMARIZE_RATING_PLUGIN_SLUG', 'wpsummarize');

/**
 * Initialize the rating notice functionality
 */
function wpsummarize_rating_notice_init() {
    add_action('admin_init', 'wpsummarize_check_and_display_rating_notice');
    add_action('wp_ajax_wpsummarize_dismiss_rating_notice', 'wpsummarize_dismiss_rating_notice');
    add_action('wp_ajax_wpsummarize_rate_later', 'wpsummarize_rate_later');
}
add_action('plugins_loaded', 'wpsummarize_rating_notice_init');

/**
 * Check if we should display the rating notice
 */
function wpsummarize_check_and_display_rating_notice() {
    
    // Only show to editors and admins
    if (!current_user_can('edit_posts')) {
        return;
    }
    
    // Don't show if already dismissed
    if (get_option('wpsummarize_rating_notice_dismissed', false)) {
        return;
    }
    
    // Check if we should delay (user clicked "maybe later")
    $delay_until = get_option('wpsummarize_rating_notice_delay_until', 0);
    if ($delay_until && time() < $delay_until) {
        return;
    }
    
    // Check if enough time has passed since activation
    $activation_time = get_option('wpsummarize_activation_time');
    if (!$activation_time) {
        // Set activation time if not set
        update_option('wpsummarize_activation_time', time());
        return;
    }
    
    $days_since_activation = (time() - $activation_time) / DAY_IN_SECONDS;
    if ($days_since_activation < WPSUMMARIZE_RATING_DAYS_THRESHOLD) {
        return;
    }
    
    // Check usage count
    $usage_count = get_option('wpsummarize_usage_count', 0);
    if ($usage_count < WPSUMMARIZE_RATING_USAGE_THRESHOLD) {
        return;
    }
    
    // All conditions met, show the notice
    add_action('admin_notices', 'wpsummarize_display_rating_notice');
    add_action('admin_enqueue_scripts', 'wpsummarize_enqueue_notice_scripts');
}

/**
 * Display the rating notice
 */
function wpsummarize_display_rating_notice() {
    $plugin_url = 'https://wordpress.org/support/plugin/' . WPSUMMARIZE_RATING_PLUGIN_SLUG . '/reviews/#new-post';
    $days = WPSUMMARIZE_RATING_DAYS_THRESHOLD;
    $plugin_name = WPSUMMARIZE_RATING_PLUGIN_NAME;
    ?>
    <div class="notice notice-info wpsummarize-rating-notice" style="position: relative; padding-right: 40px;">
        <div style="display: flex; align-items: center; padding: 10px 0;">
            <div style="margin-right: 15px;">
                <span class="dashicons dashicons-star-filled" style="color: #ffb900; font-size: 20px;"></span>
            </div>
            <div style="flex: 1;">
                <h3 style="margin: 0 0 5px 0; font-size: 16px;">
                    <?php echo sprintf(esc_html__('%s useful?', 'wpsummarize'), esc_html($plugin_name)); ?>
                </h3>
                <p style="margin: 0; font-size: 14px; line-height: 1.5;">
                    <?php echo sprintf(
                        esc_html__("We're thrilled you've been using %s! Your experience matters to us and helps other WordPress users discover our plugin.", 'wpsummarize'), 
                        esc_html($plugin_name), 
                        esc_html($days)
                    ); ?>
                </p>
                <p style="margin: 8px 0 0 0;">
                    <a href="<?php echo esc_url($plugin_url); ?>" 
                       class="button button-primary" 
                       target="_blank" 
                       style="margin-right: 10px;"
                       onclick="wpsummarizeRatingNotice.dismiss('rated')">
                        <?php echo sprintf(esc_html__('⭐ Rate %s', 'wpsummarize'), esc_html($plugin_name)); ?>
                    </a>
                    <a href="#" 
                       class="button button-secondary" 
                       style="margin-right: 10px;"
                       onclick="wpsummarizeRatingNotice.delay(); return false;">
                        <?php esc_html_e('Maybe Later', 'wpsummarize'); ?>
                    </a>
                    <a href="#" 
                       style="color: #666; text-decoration: none; font-size: 13px;"
                       onclick="wpsummarizeRatingNotice.dismiss('no-thanks'); return false;">
                        <?php esc_html_e('No thanks', 'wpsummarize'); ?>
                    </a>
                </p>
            </div>
        </div>
        <button type="button" 
                class="notice-dismiss" 
                onclick="wpsummarizeRatingNotice.dismiss('closed')">
            <span class="screen-reader-text"><?php esc_html_e('Dismiss this notice.', 'wpsummarize'); ?></span>
        </button>
    </div>
    <?php
}

/**
 * Enqueue JavaScript for the notice
 */
function wpsummarize_enqueue_notice_scripts() {
    ?>
    <script type="text/javascript">
    var wpsummarizeRatingNotice = {
        dismiss: function(reason) {
            jQuery.post(ajaxurl, {
                action: 'wpsummarize_dismiss_rating_notice',
                nonce: '<?php echo wp_create_nonce('wpsummarize_rating_nonce'); ?>',
                reason: reason
            });
            jQuery('.wpsummarize-rating-notice').fadeOut();
        },
        
        delay: function() {
            jQuery.post(ajaxurl, {
                action: 'wpsummarize_rate_later',
                nonce: '<?php echo wp_create_nonce('wpsummarize_rating_nonce'); ?>'
            });
            jQuery('.wpsummarize-rating-notice').fadeOut();
        }
    };
    </script>
    <?php
}

/**
 * Handle dismissing the notice permanently
 */
function wpsummarize_dismiss_rating_notice() {
    check_ajax_referer('wpsummarize_rating_nonce', 'nonce');
    
    if (!current_user_can('edit_posts')) {
        wp_die(esc_html__('Unauthorized', 'wpsummarize'));
    }
    
    update_option('wpsummarize_rating_notice_dismissed', true);
    
    // Optional: Track the reason for dismissal
    $reason = sanitize_text_field(isset($_POST['reason']) ? $_POST['reason'] : 'unknown');
    update_option('wpsummarize_rating_notice_dismiss_reason', $reason);
    
    wp_die('success');
}

/**
 * Handle "maybe later" - delay for 7 days
 */
function wpsummarize_rate_later() {
    check_ajax_referer('wpsummarize_rating_nonce', 'nonce');
    
    if (!current_user_can('edit_posts')) {
        wp_die(esc_html__('Unauthorized', 'wpsummarize'));
    }
    
    // Delay for 7 days
    update_option('wpsummarize_rating_notice_delay_until', time() + (7 * DAY_IN_SECONDS));
    
    wp_die('success');
}


function wpsummarize_increment_usage_count() {
    $current_count = get_option('wpsummarize_usage_count', 0);
    update_option('wpsummarize_usage_count', $current_count + 1);
}

/**
 * Reset the notice
 */
function wpsummarize_reset_rating_notice() {
    delete_option('wpsummarize_rating_notice_dismissed');
    delete_option('wpsummarize_rating_notice_delay_until');
    /*delete_option('wpsummarize_activation_time');
    delete_option('wpsummarize_usage_count');*/
}

?>