<?php
/**
 * New custom code snippet (replace this with snippet description).
 */

toolset_snippet_security_check() or die( 'Direct access is not allowed' );



use GuzzleHttp\Client;
use Smalot\PdfParser\Parser;

class Form6DeepSeekSummarizer {
    private $http_client;
    private $pdf_parser;
    private $api_key = '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';
    private $api_url = 'https://api.deepseek.com/v1/chat/completions';
    private $form_id = 6;
    
    public function __construct($api_key) {
        $this->api_key = $api_key;
        $this->http_client = new Client([
            'timeout' => 30,
            'verify' => true,
        ]);
        $this->pdf_parser = new Parser();
    }
    
    /**
     * Process all submissions for form ID 6 and generate summaries
     */
    public function processAllSubmissions() {
        $entries = $this->getAllFormEntries();
        $results = [];
        
        foreach ($entries as $entry) {
            $summary = $this->processSingleSubmission($entry);
            if ($summary) {
                $results[$entry['id']] = $summary;
            }
        }
        
        return $results;
    }
    
    /**
     * Process a single form submission
     */
    public function processSingleSubmission($entry) {
        // Verify this is form ID 6
        if ($entry['form_id'] != $this->form_id) {
            error_log("Entry {$entry['id']} is not from form {$this->form_id}");
            return false;
        }
        
        $form = GFAPI::get_form($this->form_id);
        $form_data = $this->extractAllFormFields($entry, $form);
        $pdf_content = $this->extractPDFContent($entry, $form);
        
        return $this->generateComprehensiveSummary($form_data, $pdf_content, $entry['id']);
    }
    
    /**
     * Extract all field data from form submission
     */
    private function extractAllFormFields($entry, $form) {
        $form_data = [
            'entry_id' => $entry['id'],
            'date_created' => $entry['date_created'],
            'ip' => $entry['ip'],
            'source_url' => $entry['source_url'],
            'fields' => []
        ];
        
        foreach ($form['fields'] as $field) {
            $field_id = $field->id;
            $field_label = $field->label;
            $field_type = $field->type;
            $field_value = rgar($entry, $field_id);
            
            // Skip file upload fields (handled separately)
            if ($field_type === 'fileupload') {
                continue;
            }
            
            // Format field value based on type
            $formatted_value = $this->formatFieldValue($field_value, $field_type);
            
            if (!empty($formatted_value)) {
                $form_data['fields'][] = [
                    'label' => $field_label,
                    'type' => $field_type,
                    'value' => $formatted_value
                ];
            }
        }
        
        return $form_data;
    }
    
    /**
     * Extract PDF content from file upload fields
     */
    private function extractPDFContent($entry, $form) {
        $pdf_content = [];
        
        foreach ($form['fields'] as $field) {
            if ($field->type === 'fileupload') {
                $file_urls = rgar($entry, $field->id);
                
                if (!empty($file_urls)) {
                    $urls = json_decode($file_urls);
                    if (!$urls) {
                        $urls = [$file_urls];
                    }
                    
                    foreach ($urls as $url) {
                        if (pathinfo($url, PATHINFO_EXTENSION) === 'pdf') {
                            $text = $this->extractTextFromPDF($url);
                            if ($text) {
                                $pdf_content[] = [
                                    'field_label' => $field->label,
                                    'file_url' => $url,
                                    'content' => $text
                                ];
                            }
                        }
                    }
                }
            }
        }
        
        return $pdf_content;
    }
    
    /**
     * Extract text from PDF file
     */
    private function extractTextFromPDF($file_url) {
        try {
            $file_path = $this->urlToPath($file_url);
            
            if (!$file_path || !file_exists($file_path)) {
                error_log("PDF file not found: $file_url");
                return null;
            }
            
            $pdf = $this->pdf_parser->parseFile($file_path);
            $text = $pdf->getText();
            
            // Clean and truncate text
            $text = preg_replace('/\s+/', ' ', $text);
            if (strlen($text) > 8000) {
                $text = substr($text, 0, 8000) . '... [content truncated]';
            }
            
            return trim($text);
            
        } catch (Exception $e) {
            error_log('PDF parsing error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Convert URL to local file path
     */
    private function urlToPath($url) {
        $upload_dir = wp_upload_dir();
        $base_url = $upload_dir['baseurl'];
        $base_path = $upload_dir['basedir'];
        
        $base_url = preg_replace('/^https?:/i', '', $base_url);
        $clean_url = preg_replace('/^https?:/i', '', $url);
        
        if (strpos($clean_url, $base_url) !== false) {
            $relative_path = str_replace($base_url, '', $clean_url);
            return $base_path . $relative_path;
        }
        
        return null;
    }
    
    /**
     * Format field value based on field type
     */
    private function formatFieldValue($value, $field_type) {
        if (empty($value)) {
            return '';
        }
        
        switch ($field_type) {
            case 'list':
                if (is_array($value)) {
                    return implode('; ', array_map(function($row) {
                        return is_array($row) ? implode(', ', $row) : $row;
                    }, $value));
                }
                return $value;
                
            case 'multiselect':
            case 'checkbox':
                if (is_array($value)) {
                    return implode(', ', $value);
                }
                return $value;
                
            case 'name':
                if (is_array($value)) {
                    return implode(' ', array_filter($value));
                }
                return $value;
                
            default:
                if (is_array($value)) {
                    return implode(', ', $value);
                }
                return $value;
        }
    }
    
    /**
     * Get all entries for form ID 6
     */
    private function getAllFormEntries() {
        $search_criteria = [
            'status' => 'active'
        ];
        
        $sorting = [
            'key' => 'date_created',
            'direction' => 'DESC'
        ];
        
        $paging = [
            'offset' => 0,
            'page_size' => 1000 // Adjust based on your needs
        ];
        
        return GFAPI::get_entries($this->form_id, $search_criteria, $sorting, $paging);
    }
    
    /**
     * Generate comprehensive summary using DeepSeek
     */
    private function generateComprehensiveSummary($form_data, $pdf_content, $entry_id) {
        $prompt = $this->createSummaryPrompt($form_data, $pdf_content, $entry_id);
        
        try {
            $response = $this->http_client->post($this->api_url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'model' => 'deepseek-chat',
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => 'You are a professional assistant that summarizes form submissions comprehensively. Analyze all form fields and attached PDF documents to create detailed, well-structured summaries that capture all important information.'
                        ],
                        [
                            'role' => 'user',
                            'content' => $prompt
                        ]
                    ],
                    'max_tokens' => 2500,
                    'temperature' => 0.2,
                    'stream' => false
                ]
            ]);
            
            $response_data = json_decode($response->getBody(), true);
            
            if (isset($response_data['choices'][0]['message']['content'])) {
                return $response_data['choices'][0]['message']['content'];
            } else {
                error_log('DeepSeek API response error for entry ' . $entry_id);
                return false;
            }
            
        } catch (Exception $e) {
            error_log('DeepSeek API Error for entry ' . $entry_id . ': ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create detailed prompt for DeepSeek
     */
    private function createSummaryPrompt($form_data, $pdf_content, $entry_id) {
        $prompt = "Please provide a comprehensive analysis and summary of Form Submission #{$entry_id}:\n\n";
        
        // Add submission metadata
        $prompt .= "## SUBMISSION METADATA\n";
        $prompt .= "- **Entry ID**: {$form_data['entry_id']}\n";
        $prompt .= "- **Submission Date**: {$form_data['date_created']}\n";
        $prompt .= "- **IP Address**: {$form_data['ip']}\n";
        $prompt .= "- **Source URL**: {$form_data['source_url']}\n\n";
        
        // Add all form fields
        $prompt .= "## FORM FIELD DATA\n\n";
        foreach ($form_data['fields'] as $field) {
            $prompt .= "### {$field['label']} ({$field['type']})\n";
            $prompt .= "{$field['value']}\n\n";
        }
        
        // Add PDF content
        if (!empty($pdf_content)) {
            $prompt .= "## ATTACHED PDF DOCUMENTS\n\n";
            foreach ($pdf_content as $index => $pdf) {
                $prompt .= "### Document " . ($index + 1) . ": {$pdf['field_label']}\n";
                $prompt .= "**File**: {$pdf['file_url']}\n";
                $prompt .= "**Content**:\n{$pdf['content']}\n\n";
            }
        }
        
        $prompt .= "## SUMMARY REQUIREMENTS\n\n";
        $prompt .= "Please provide a detailed summary that:\n\n";
        $prompt .= "1. **Synthesizes Information**: Combine data from all form fields and attached documents\n";
        $prompt .= "2. **Highlights Key Points**: Identify the most important information\n";
        $prompt .= "3. **Notes Connections**: Point out relationships between form data and document content\n";
        $prompt .= "4. **Identifies Action Items**: Note any required follow-ups or important details\n";
        $prompt .= "5. **Provides Context**: Explain the significance of the submission\n\n";
        $prompt .= "Format the summary with clear sections and use bullet points for readability.\n\n";
        $prompt .= "COMPREHENSIVE SUMMARY:";
        
        return $prompt;
    }
}
// Put the code of your snippet below this comment.